// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.cli.CliSugar.*;

import org.junit.jupiter.api.Test;
import org.refcodes.data.AsciiColorPalette;

public class MessageTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testErrorMessage1() {
		String[] theArgs = new String[] { "--string=someString", "--token=1234567890" };
		String theExpected = "No syntax branch (fully) matched the command line arguments, though one (XOR) syntax branch must match! Caused by: Failed as of superfluous \"--token=1234567890\" argument, though the arguments must be matched completely (ALL) by the syntax branches! Possible cause: No option \"--email\" was found in the command line arguments, though it must be specified to match the according syntax branch! Also notable: No option \"--name\" was found in the command line arguments, though it must be specified to match the according syntax branch! Also notable: Neither the short-option \"-h\" nor the long-option \"--help\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch! Neither the short-option \"-v\" nor the long-option \"--version\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch!";
		String theHeurisiticExprected = "Failed as of superfluous \"--token=1234567890\" argument, though the arguments must be matched completely (ALL) by the syntax branches! Possible cause: No option \"--email\" was found in the command line arguments, though it must be specified to match the according syntax branch! Also notable: No option \"--name\" was found in the command line arguments, though it must be specified to match the according syntax branch! Also notable: Neither the short-option \"-h\" nor the long-option \"--help\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch! Neither the short-option \"-v\" nor the long-option \"--version\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch!";
		ArgsParser theParser = createArgsParser();
		try {
			theParser.evalArgs( theArgs );
			fail( "Expected an exception of type <" + UnknownArgsException.class.getSimpleName() + ">!" );
		}
		catch ( ArgsSyntaxException e ) {
			String theMessage = e.toMessage();
			String theHeuristicMessage = e.toHeuristicMessage();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "toMessage() = " + theMessage );
				System.out.println( "toHeuristicMessage() = " + theHeuristicMessage );
			}
			assertEquals( theExpected, theMessage );
			assertEquals( theHeurisiticExprected, theHeuristicMessage );
		}
	}

	@Test
	public void testErrorMessage2() {
		String[] theArgs = new String[] { "--string=someString", "--email=someEmail", "--token=1234567890" };
		String theExpected = "No syntax branch (fully) matched the command line arguments, though one (XOR) syntax branch must match! Caused by: Failed as of superfluous \"--email=someEmail\", \"--token=1234567890\" arguments, though the arguments must be matched completely (ALL) by the syntax branches! Possible cause: No option \"--name\" was found in the command line arguments, though it must be specified to match the according syntax branch! Also notable: Neither the short-option \"-h\" nor the long-option \"--help\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch! Neither the short-option \"-v\" nor the long-option \"--version\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch!";
		String theHeurisiticExprected = "Failed as of superfluous \"--email=someEmail\", \"--token=1234567890\" arguments, though the arguments must be matched completely (ALL) by the syntax branches! Possible cause: No option \"--name\" was found in the command line arguments, though it must be specified to match the according syntax branch! Also notable: Neither the short-option \"-h\" nor the long-option \"--help\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch! Neither the short-option \"-v\" nor the long-option \"--version\"  was found in the command line arguments, at least one of them must be specified to match the according syntax branch!";
		ArgsParser theParser = createArgsParser();
		try {
			theParser.evalArgs( theArgs );
			fail( "Expected an exception of type <" + UnknownArgsException.class.getSimpleName() + ">!" );
		}
		catch ( ArgsSyntaxException e ) {
			String theMessage = e.toMessage();
			String theHeuristicMessage = e.toHeuristicMessage();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "toMessage() = " + theMessage );
				System.out.println( "toHeuristicMessage() = " + theHeuristicMessage );
			}
			assertEquals( theExpected, theMessage );
			assertEquals( theHeurisiticExprected, theHeuristicMessage );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static ArgsParser createArgsParser() {
		final ArgsParser theParser = new ArgsParser();
		Flag theHelpFlag = flag( 'h', "help", "Shows the help ..." );
		Flag theVersionFlag = flag( 'v', "version", "Shows the version ..." );
		StringProperty theString1Arg = stringProperty( "string", "..." );
		StringProperty theString2Arg = stringProperty( "string", "..." );
		BooleanProperty theboolArg = booleanProperty( "bool", "..." );
		Flag theBoolFlag = flag( "bool", "..." );
		StringProperty theEMailArg = stringProperty( "email", "..." );
		StringProperty theNameArg = stringProperty( "name", "..." );
		StringProperty theTokenArg = stringProperty( "token", "..." );
		IntProperty theIntArg = intProperty( "int", "..." );
		DoubleProperty theDoubleArg = doubleProperty( "double", "..." );

		// @formatter:off
		Term theArgsSyntax =  cases(
			and( theString1Arg, 
				optional( theString2Arg, 
					xor( theboolArg, theBoolFlag, 
						and( theEMailArg, theNameArg, theTokenArg ) 
					),
					theIntArg, theDoubleArg
				)
			),
			theHelpFlag,
			theVersionFlag
		);
		// @formatter:on

		return theParser.withArgsSyntax( theArgsSyntax ).withName( null ).withSyntaxMetrics( SyntaxNotation.LOGICAL ).withEscapeCodesEnabled( false ).withConsoleWidth( 80 ).withBannerFontPalette( AsciiColorPalette.MAX_LEVEL_GRAY ).withLicense( null ).withCopyright( null ).withTitle( "CROWD:IT" ).withDescription( null );
	}
}
