// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.struct;

/**
 * Provides an accessor for a double array property.
 */
public interface DoubleArrayAccessor {

	/**
	 * Retrieves the double array from the double array property.
	 * 
	 * @return The double array stored by the double array property.
	 */
	double[] getDoubles();

	/**
	 * Provides a mutator for a double array property.
	 */
	public interface DoubleArrayMutator {

		/**
		 * Sets the double array for the double array property.
		 * 
		 * @param aDoubles The double array to be stored by the double array
		 *        property.
		 */
		void setDoubles( double[] aDoubles );
	}

	/**
	 * Provides a builder method for a double array property returning the
	 * builder for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface DoubleArrayBuilder<B extends DoubleArrayBuilder<B>> {

		/**
		 * Sets the double array for the double array property.
		 * 
		 * @param aDoubles The double array to be stored by the double array
		 *        property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withDoubles( double[] aDoubles );
	}

	/**
	 * Provides a double array property.
	 */
	public interface DoubleArrayProperty extends DoubleArrayAccessor, DoubleArrayMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given double array (setter) as
		 * of {@link #setDoubles(double[])} and returns the very same value
		 * (getter).
		 * 
		 * @param aDoubles The double array to set (via
		 *        {@link #setDoubles(double[])}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default double[] letDoubles( double[] aDoubles ) {
			setDoubles( aDoubles );
			return aDoubles;
		}
	}
}
