// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.struct;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.refcodes.data.Literal;

/**
 * The Class TypeUtility.
 */
public class TypeUtility {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String[] SETTER_PREFIXES = { "set" };

	private static final String[] GETTER_PREFIXES = { "is", "has", "get" };

	private static final String[] FIELD_PREFIXES = { "_" };

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates an instance of the given type filled with the data provided by
	 * the given Data-Structure (a mixture of array and {@link Map} objects).
	 *
	 * @param <T> the generic type
	 * @param aValue The Data-Structure, a mixture of arrays and {@link Map}
	 *        instance, from which to construct the instance of the given type.
	 * @param aType The type for which an instance is to be created.
	 * 
	 * @return The instance filled by the data from the given Data-Structure
	 * 
	 * @throws InstantiationException thrown when an application tries to create
	 *         an instance of a class using the newInstance method in class
	 *         {@link Class}, but the specified class object cannot be
	 *         instantiated.
	 * @throws IllegalAccessException thrown when an application tries to
	 *         reflectively create an instance (other than an array), set or get
	 *         a field, or invoke a method, but the currently executing method
	 *         does not have access to the definition of the specified class,
	 *         field, method or constructor.
	 * @throws NoSuchMethodException thrown when a particular method cannot be
	 *         found.
	 * @throws SecurityException thrown by the security manager to indicate a
	 *         security violation.
	 * @throws InvocationTargetException wraps an exception thrown by an invoked
	 *         method or constructor.
	 * @throws ClassNotFoundException thrown when an application tries to load
	 *         in a class through its string name but no definition for the
	 *         class with the specified name could be found.
	 */
	@SuppressWarnings("unchecked")
	public static <T> T toType( Object aValue, Class<T> aType ) throws InstantiationException, IllegalAccessException, NoSuchMethodException, SecurityException, InvocationTargetException, ClassNotFoundException {
		T theInstance;
		// Array:
		if ( aType.isArray() ) {
			theInstance = toArray( aValue, aType );
		}
		else if ( aType.isAssignableFrom( aValue.getClass() ) ) {
			return (T) aValue;
		}
		// Other:
		else {
			theInstance = toInstance( aValue, aType );
		}
		return theInstance;
	}

	/**
	 * Updates the provided instance with the data provided by the given value
	 * (a mixture of array and {@link Map} objects). In case the instance to be
	 * updated represents an array, then at most it is filled with the provided
	 * value's elements, even if the value provides more elements than would
	 * fit. On the other hand, if the instance's array to be updated provides
	 * room for more elements than the value can provide, then the remaining
	 * instance's elements are left as are.
	 *
	 * @param <T> the generic type
	 * @param aValue The Data-Structure, a mixture of arrays and {@link Map}
	 *        instance, from which to update the instance of the given type.
	 * @param aInstance The instance to be updated.
	 * 
	 * @throws InstantiationException thrown when an application tries to create
	 *         an instance of a class using the newInstance method in class
	 *         {@link Class}, but the specified class object cannot be
	 *         instantiated.
	 * @throws IllegalAccessException thrown when an application tries to
	 *         reflectively create an instance (other than an array), set or get
	 *         a field, or invoke a method, but the currently executing method
	 *         does not have access to the definition of the specified class,
	 *         field, method or constructor.
	 * @throws NoSuchMethodException thrown when a particular method cannot be
	 *         found.
	 * @throws SecurityException thrown by the security manager to indicate a
	 *         security violation.
	 * @throws InvocationTargetException wraps an exception thrown by an invoked
	 *         method or constructor.
	 * @throws ClassNotFoundException thrown when an application tries to load
	 *         in a class through its string name but no definition for the
	 *         class with the specified name could be found.
	 */
	@SuppressWarnings("unchecked")
	public static <T> void toInstance( Object aValue, T aInstance ) throws InstantiationException, IllegalAccessException, NoSuchMethodException, SecurityException, InvocationTargetException, ClassNotFoundException {
		// Array:
		Class<T> theClass = (Class<T>) aInstance.getClass();
		if ( theClass.isArray() ) {
			T theInstance = (T) toArray( aValue, theClass );
			int fromLength = Array.getLength( theInstance );
			int toLength = Array.getLength( aInstance );
			for ( int i = 0; i < fromLength && i < toLength; i++ ) {
				Array.set( aInstance, i, Array.get( theInstance, i ) );
			}
		}
		// Other:
		else {
			updateInstance( aValue, aInstance );
		}
	}

	/**
	 * Tests whether we have a setter method. A setter method must not return a
	 * value and expect exactly one argument. It also must be public.
	 * 
	 * @param aMethod The {@link Method} to be tested.
	 * 
	 * @return True in case we have a setter method, else false.
	 */
	public static boolean isSetter( Method aMethod ) {
		String thePropertyName = fromSetterMethod( aMethod );
		if ( thePropertyName == null || thePropertyName.length() == 0 ) {
			return false;
		}
		return Modifier.isPublic( aMethod.getModifiers() ) && aMethod.getParameterCount() == 1 && Void.TYPE.equals( aMethod.getReturnType() );
	}

	/**
	 * Tests whether we have a getter method. A getter method must return a
	 * value (other than {@link Void}) and expect none arguments. It also must
	 * be public.
	 * 
	 * @param aMethod The {@link Method} to be tested.
	 * 
	 * @return True in case we have a getter method, else false.
	 */
	public static boolean isGetter( Method aMethod ) {
		String thePropertyName = fromGetterMethod( aMethod );
		if ( thePropertyName == null || thePropertyName.length() == 0 ) {
			return false;
		}
		return Modifier.isPublic( aMethod.getModifiers() ) && aMethod.getParameterCount() == 0 && !Void.TYPE.equals( aMethod.getReturnType() );
	}

	/**
	 * Converts a method's name to a property name.
	 * 
	 * @param aMethod The {@link Method} which's name is to be converted to a
	 *        property name.
	 * 
	 * @return The according property name or null if we do not have a property
	 *         method.
	 */
	public static String toPropertyName( Method aMethod ) {

		// Setter:
		if ( isSetter( aMethod ) ) {
			return fromSetterMethod( aMethod );
		}
		// Getter:
		else if ( isGetter( aMethod ) ) {
			return fromGetterMethod( aMethod );
		}
		return null;
	}

	/**
	 * Converts a field's name to a property name.
	 * 
	 * @param aField The {@link Field} which's name is to be converted to a
	 *        property name.
	 * 
	 * @return The according property name or null if we do not have a property
	 *         method.
	 */
	public static String toPropertyName( Field aField ) {
		String theName = aField.getName();
		return fromFieldName( theName );
	}

	/**
	 * Converts the provided object reference (which must point to the according
	 * array type) to the required array type.
	 * 
	 * @param <T> The type of the array to which to convert to.
	 * @param aValue The object reference which to convert.
	 * @param aType The type to which to convert to.
	 * 
	 * @return The according array type.
	 * 
	 * @throws IllegalArgumentException thrown in case there were reflection or
	 *         introspection problems when converting the object to the required
	 *         array type.
	 */
	public static <T> T toArrayType( Object aValue, Class<T> aType ) throws IllegalArgumentException {
		if ( !aValue.getClass().isArray() ) {
			throw new IllegalArgumentException( "The provided value <" + aValue + "> of type <" + aValue.getClass() + "> must be an array type!" );
		}
		if ( !aType.isArray() ) {
			throw new IllegalArgumentException( "The provided type <" + aType + "> must be an array type!" );
		}
		try {
			return toArray( aValue, aType );
		}
		catch ( NoSuchMethodException | InstantiationException | IllegalAccessException | InvocationTargetException | ClassNotFoundException e ) {
			throw new IllegalArgumentException( "Cannot create an array type <" + aType + "> for the given value <" + aValue + "): " + e.getMessage(), e );
		}
	}

	/**
	 * This method creates a {@link Map} with the attributes found in the given
	 * instance.The values of the attributes are not introspected any more!
	 * 
	 * @param <T> The type of the instance to be introspected.
	 * @param aInstance The instance from which to get the attributes.
	 * 
	 * @return The {@link Map} containing the attributes of the provided
	 *         instance, the keys being the attribute names, the values being
	 *         the attribute values
	 */
	@SuppressWarnings("unchecked")
	public static <T> Map<String, Object> toMap( T aInstance ) {
		Class<T> theType = (Class<T>) aInstance.getClass();
		Map<String, Object> theResult = new HashMap<>();

		// Methods |-->
		Method[] theMethods = theType.getDeclaredMethods();
		if ( theMethods != null && theMethods.length > 0 ) {
			String ePropertyName;
			Object eValue;
			for ( Method eMethod : theMethods ) {
				if ( isGetter( eMethod ) ) {
					ePropertyName = fromGetterMethod( eMethod );
					// Don't in Java 9 |-->
					try {
						eMethod.setAccessible( true );
					}
					catch ( Exception ignore ) {}
					// Don't in Java 9 <--|
					try {
						eValue = eMethod.invoke( aInstance );
						theResult.put( ePropertyName, eValue );
					}
					catch ( IllegalAccessException | IllegalArgumentException | InvocationTargetException ignore ) {}
				}
			}
		}
		// Methods <--|

		// Fields |-->
		Field[] theFields = theType.getDeclaredFields();
		if ( theFields != null && theFields.length > 0 ) {
			String ePropertyName;
			Object eValue;
			for ( Field eField : theFields ) {
				ePropertyName = toPropertyName( eField );
				// Don't in Java 9 |-->
				try {
					eField.setAccessible( true );
				}
				catch ( Exception ignore ) {}
				// Don't in Java 9 <--|
				try {
					eValue = eField.get( aInstance );
					theResult.put( ePropertyName, eValue );
				}
				catch ( IllegalArgumentException | IllegalAccessException ignore ) {}
			}
		}
		// Fields <--|

		return theResult;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	@SuppressWarnings("unchecked")
	private static <T> T toInstance( Object aValue, Class<T> aType ) throws NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException, ClassNotFoundException {
		if ( aValue instanceof String ) {
			try {
				Class<?> theClass = Class.forName( (String) aValue );
				if ( aType.isAssignableFrom( theClass ) ) {
					Constructor<?> theCtor = theClass.getConstructor();
					return (T) theCtor.newInstance();
				}
			}
			catch ( NoClassDefFoundError | Exception ignore ) {}
		}

		if ( aType.isEnum() && aValue instanceof String ) {
			Enum<?>[] theEnums = (Enum<?>[]) aType.getEnumConstants();
			for ( Enum<?> eEnum : theEnums ) {
				if ( eEnum.name().equals( aValue ) ) {
					return (T) eEnum;
				}
			}
		}

		Constructor<T> theCtor = aType.getConstructor();
		// Don't in Java 9 |-->
		try {
			theCtor.setAccessible( true );
		}
		catch ( Exception ignore ) {}
		// Don't in Java 9 <--|
		T theToInstance = theCtor.newInstance();
		toInstance( aValue, theToInstance );
		return theToInstance;
	}

	@SuppressWarnings("unchecked")
	private static <T> void updateInstance( Object aValue, T aInstance ) throws NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException, ClassNotFoundException {
		Class<T> aType = (Class<T>) aInstance.getClass();
		if ( aValue instanceof Map ) {
			Map<String, ?> theMap = (Map<String, ?>) aValue;
			Set<String> theInjected = new HashSet<>();

			// Methods |-->
			Method[] theMethods = aType.getDeclaredMethods();
			if ( theMethods != null && theMethods.length > 0 ) {
				// Take care of identically named methods with differing argument types |-->
				Set<String> theFailedProperties = new HashSet<>();
				Exception theFailedException = null;
				// Take care of identically named methods with differing argument types <--|
				String ePropertyName;
				for ( Method eMethod : theMethods ) {
					if ( isSetter( eMethod ) ) {
						try {
							ePropertyName = fromSetterMethod( eMethod );
							if ( !theInjected.contains( ePropertyName ) ) {
								Object eValue = theMap.get( ePropertyName );
								if ( eValue != null ) {
									// Don't in Java 9 |-->
									try {
										eMethod.setAccessible( true );
									}
									catch ( Exception ignore ) {}
									// Don't in Java 9 <--|
									// Special case: Array |-->
									if ( eValue.getClass().isArray() && !eMethod.getParameters()[0].getType().isArray() && !Collection.class.isAssignableFrom( eMethod.getParameters()[0].getType() ) ) {
										Object[] eArray = (Object[]) eValue;
										Object eObj = null;
										for ( Object eElement : eArray ) {
											if ( eElement != null ) {
												if ( eObj != null && eElement instanceof Map ) {
													updateInstance( eElement, eObj );
												}
												else {
													try {
														eObj = addToMethod( aInstance, eElement, eMethod );
														theInjected.add( ePropertyName );
													}
													catch ( Exception e ) {
														theFailedProperties.add( ePropertyName ); // Are there yet more overloaded candidates?
														theFailedException = theFailedException == null ? e : theFailedException;
													}
												}
											}
										}
									}
									// Special case: Array <--|
									else {
										try {
											addToMethod( aInstance, eValue, eMethod );
											theInjected.add( ePropertyName );
										}
										catch ( Exception e ) {
											theFailedProperties.add( ePropertyName ); // Are there yet more overloaded candidates?
											theFailedException = theFailedException == null ? e : theFailedException;
										}
									}
								}
							}
						}
						catch ( IllegalArgumentException | IllegalAccessException ignore ) {}
					}
				}
				// Are there yet unsatisfied candidates? |-->
				if ( !theFailedProperties.isEmpty() ) {
					for ( String eInjectedName : theInjected ) {
						theFailedProperties.remove( eInjectedName );

					}
					if ( !theFailedProperties.isEmpty() ) {
						if ( theFailedException != null ) {
							throw new NoSuchMethodNotInvokedException( "Unable to satisfy the following properties " + Arrays.toString( theFailedProperties.toArray() ) + " (without the square braces) as there are no according methods with the required argument types!", theFailedException );
						}
						else {
							throw new NoSuchMethodException( "Unable to satisfy the following properties " + Arrays.toString( theFailedProperties.toArray() ) + " (without the square braces) as there are no according methods with the required argument types!" );
						}
					}
				}
				// Are there yet unsatisfied candidates? <--|
			}
			// Methods <--|

			// Fields |-->
			Field[] theFields = aType.getDeclaredFields();
			if ( theFields != null && theFields.length > 0 ) {
				String ePropertyName;
				for ( Field eField : theFields ) {
					try {
						ePropertyName = toPropertyName( eField );
						if ( !theInjected.contains( ePropertyName ) ) {
							Object eValue = theMap.get( ePropertyName );
							if ( eValue != null ) {
								// Don't in Java 9 |-->
								try {
									eField.setAccessible( true );
								}
								catch ( Exception ignore ) {}
								// Don't in Java 9 <--|
								// Special case: Array |-->
								if ( eValue.getClass().isArray() && !eField.getType().isArray() && !Collection.class.isAssignableFrom( eField.getType() ) ) {
									Object[] eArray = (Object[]) eValue;
									for ( Object eElement : eArray ) {
										if ( eElement != null ) {
											addToField( aInstance, eElement, eField );
											theInjected.add( ePropertyName );
										}
									}
								}
								// Special case: Array <--|
								else {
									addToField( aInstance, eValue, eField );
									theInjected.add( ePropertyName );
								}
							}
						}
					}
					catch ( IllegalArgumentException | IllegalAccessException ignore ) {}
				}
			}
			// Fields <--|
		}
	}

	private static <T> Object addToMethod( T aToInstance, Object aToValue, Method aToMethod ) throws InstantiationException, IllegalAccessException, ClassNotFoundException, NoSuchMethodException, InvocationTargetException {
		String theToText = aToValue.toString();
		Class<?> theToType = aToMethod.getParameters()[0].getType();
		// String:
		if ( theToType.equals( String.class ) ) {
			aToMethod.invoke( aToInstance, theToText );
		}
		else if ( theToText.length() != 0 ) {
			if ( !Literal.NULL.getValue().equals( theToText ) ) {
				// Simple types:
				if ( !SimpleType.invokeSimpleType( aToInstance, aToMethod, theToText, theToType ) ) {
					// List:
					if ( List.class.isAssignableFrom( theToType ) ) {
						List<?> theToList = toList( aToValue, aToMethod.getParameters()[0].getType(), aToMethod.getParameters()[0].getParameterizedType() );
						aToMethod.invoke( aToInstance, theToList );
					}
					// Map:
					else if ( Map.class.isAssignableFrom( theToType ) ) {
						Map<?, ?> theToMap = toMap( aToValue, aToMethod.getParameters()[0].getType(), aToMethod.getParameters()[0].getParameterizedType() );
						aToMethod.invoke( aToInstance, theToMap );
					}
					// Other:
					else {
						Object theInstance = toType( aToValue, theToType );
						aToMethod.invoke( aToInstance, theInstance );
						return theInstance;
					}
				}
			}
		}
		return null;
	}

	private static <T> void addToField( T aToInstance, Object aToValue, Field aToField ) throws InstantiationException, IllegalAccessException, NoSuchMethodException, SecurityException, IllegalArgumentException, InvocationTargetException, ClassNotFoundException {
		String theToText = aToValue.toString();
		Class<?> theToType = aToField.getType();
		// String:
		if ( theToType.equals( String.class ) ) {
			aToField.set( aToInstance, theToText );
		}
		else if ( theToText.length() != 0 ) {
			if ( !Literal.NULL.getValue().equals( theToText ) ) {
				// Simple types:
				if ( !SimpleType.invokeSimpleType( aToInstance, aToField, theToText, theToType ) ) {
					// List:
					if ( List.class.isAssignableFrom( theToType ) ) {
						List<?> theToList = toList( aToValue, aToField.getType(), aToField.getGenericType() );
						aToField.set( aToInstance, theToList );
					}
					// Map:
					else if ( Map.class.isAssignableFrom( theToType ) ) {
						Map<?, ?> theToMap = toMap( aToValue, aToField.getType(), aToField.getGenericType() );
						aToField.set( aToInstance, theToMap );
					}
					// Other:
					else {
						aToField.set( aToInstance, toType( aToValue, theToType ) );
					}
				}
			}
		}
	}

	@SuppressWarnings("unchecked")
	private static <T> T toArray( Object aValue, Class<T> aType ) throws NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException, ClassNotFoundException {
		T theToInstance;
		Class<?> theToType = aType.getComponentType();
		Object[] theArray = (Object[]) aValue;
		theToInstance = (T) Array.newInstance( aType.getComponentType(), theArray.length );
		String eToText;
		for ( int i = 0; i < theArray.length; i++ ) {
			if ( theArray[i] != null ) {
				eToText = theArray[i].toString();
				// String:
				if ( theToType.equals( String.class ) ) {
					Array.set( theToInstance, i, eToText );
				}
				else if ( eToText.length() != 0 ) {
					if ( !Literal.NULL.getValue().equals( eToText ) ) {
						// Simple types:
						if ( !SimpleType.invokeSimpleType( theToInstance, i, eToText, theToType ) ) {
							// Other:
							Array.set( theToInstance, i, toType( theArray[i], theToType ) );
						}
					}
				}
			}
		}
		return theToInstance;
	}

	@SuppressWarnings("unchecked")
	private static Map<?, ?> toMap( Object aToValue, Class<?> aToType, Type aToGenericType ) throws InstantiationException, IllegalAccessException, ClassNotFoundException, NoSuchMethodException, InvocationTargetException {
		Map<Object, Object> theToMap;
		Map<Object, Object> theMap = (Map<Object, Object>) aToValue;
		if ( aToType.equals( Map.class ) ) {
			theToMap = new HashMap<>();
		}
		else {
			theToMap = (Map<Object, Object>) aToType.getConstructor().newInstance();
		}

		Class<?> theKeyType = Object.class;
		Class<?> theValueType = Object.class;
		if ( aToGenericType instanceof ParameterizedType ) {
			ParameterizedType theGenericType = (ParameterizedType) aToGenericType;
			theKeyType = toClass( theGenericType.getActualTypeArguments()[0] );
			theValueType = toClass( theGenericType.getActualTypeArguments()[1] );
		}
		for ( Object eKey : theMap.keySet() ) {
			Object theValue = theMap.get( eKey );
			if ( theValue != null ) {
				theToMap.put( toValue( eKey, theKeyType ), toValue( theValue, theValueType ) );
			}
		}
		return theMap;

	}

	@SuppressWarnings("unchecked")
	private static List<?> toList( Object aToValue, Class<?> aToType, Type aToGenericType ) throws InstantiationException, IllegalAccessException, ClassNotFoundException, NoSuchMethodException, InvocationTargetException {
		List<Object> theToList = null;
		Object[] theArray = (Object[]) aToValue;
		if ( aToType.equals( List.class ) ) {
			theToList = new ArrayList<>();
		}
		else {
			theToList = (List<Object>) aToType.getConstructor().newInstance();
		}
		Class<?> theType = Object.class;
		if ( aToGenericType instanceof ParameterizedType ) {
			theType = toClass( ((ParameterizedType) aToGenericType).getActualTypeArguments()[0] );
		}
		for ( int i = 0; i < theArray.length; i++ ) {
			if ( theArray[i] == null ) {
				theToList.add( null );
			}
			else {
				theToList.add( toValue( theArray[i], theType ) );
			}
		}
		return theToList;
	}

	private static Object toValue( Object aValue, Class<?> aType ) throws NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException, ClassNotFoundException {
		// Key |-->
		String theValue = aValue.toString();
		Object theResult = null;
		// String:
		if ( aType.equals( String.class ) ) {
			theResult = theValue;
		}
		else if ( theValue.length() != 0 ) {
			if ( !Literal.NULL.getValue().equals( theValue ) ) {
				// Simple type:
				theResult = SimpleType.toSimpleType( theValue, aType );
				// Other:
				if ( theResult == null ) {
					theResult = toType( aValue, aType );
				}
			}
		}
		return theResult;
	}

	private static Class<?> toClass( Type aType ) throws ClassNotFoundException {
		if ( aType instanceof Class ) {
			return (Class<?>) aType;
		}
		return Class.forName( aType.getTypeName() );
	}

	private static String toPropertyName( String aName ) {
		char[] theChars = aName.toCharArray();
		int i = 0;
		while ( i < theChars.length && Character.isUpperCase( theChars[i] ) ) {
			theChars[i] = Character.toLowerCase( theChars[i] );
			i++;
		}
		return new String( theChars );
	}

	private static String fromSetterMethod( Method aMethod ) {
		String theName = aMethod.getName();
		return fromSetterName( theName );
	}

	private static String fromSetterName( String aName ) {
		SETTER: {
			for ( String ePrefix : SETTER_PREFIXES ) {
				if ( aName.startsWith( ePrefix ) ) {
					aName = aName.substring( ePrefix.length() );
					break SETTER;
				}
			}
			return null;
		}
		if ( aName.length() > 0 ) {
			if ( Character.isLowerCase( aName.charAt( 0 ) ) ) {
				return null;
			}
			return toPropertyName( aName );
		}
		return null;
	}

	protected static String fromGetterMethod( Method aMethod ) {
		String theName = aMethod.getName();
		return fromGetterName( theName );
	}

	private static String fromGetterName( String aName ) {
		GETTER: {
			for ( String ePrefix : GETTER_PREFIXES ) {
				if ( aName.startsWith( ePrefix ) ) {
					aName = aName.substring( ePrefix.length() );
					break GETTER;
				}
			}
			return null;
		}
		if ( aName.length() > 0 ) {
			if ( Character.isLowerCase( aName.charAt( 0 ) ) ) {
				return null;
			}
			return toPropertyName( aName );
		}
		return null;
	}

	/**
	 * @param aFieldName
	 * 
	 * @return
	 */
	private static String fromFieldName( String aFieldName ) {
		String eName = aFieldName;
		for ( String ePrefix : FIELD_PREFIXES ) {
			if ( eName.startsWith( ePrefix ) ) {
				eName = aFieldName.substring( ePrefix.length() );
				break;
			}
		}
		if ( eName.length() > 0 ) {
			aFieldName = eName;
		}
		eName = fromGetterName( aFieldName );
		if ( eName != null && eName.length() > 0 ) {
			aFieldName = eName;
		}
		return toPropertyName( aFieldName );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Extension of the {@link NoSuchMethodException} with an additional cause
	 * {@link #getCause()}
	 */
	public static class NoSuchMethodNotInvokedException extends NoSuchMethodException {

		private static final long serialVersionUID = 1L;

		private Throwable _cause;

		/**
		 * {@inheritDoc}
		 */
		public NoSuchMethodNotInvokedException( String aMessage, Throwable aCause ) {
			super( aMessage );
			_cause = aCause;
		}

		@Override
		public Throwable getCause() {
			return _cause;
		}

	}

}
