/*-
 * #%L
 * Functional interfaces for SciJava-based libraries.
 * %%
 * Copyright (C) 2021 - 2024 SciJava developers.
 * %%
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * #L%
 */
/*
 * This is autogenerated source code -- DO NOT EDIT. Instead, edit the
 * corresponding template in templates/ and rerun bin/generate.groovy.
 */

package org.scijava.function;

import java.util.HashMap;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * Container class for computer-style functional interfaces at various
 * <a href="https://en.wikipedia.org/wiki/Arity">arities</a>.
 * <p>
 * A computer has functional method {@code compute} with a number of arguments
 * corresponding to the arity, plus an additional argument for the preallocated
 * output to be populated by the computation.
 * </p>
 * <p>
 * Each computer interface implements a corresponding {@link Consumer}-style
 * interface (see {@link Consumers}) with arity+1; the consumer's {@code accept}
 * method simply delegates to {@code compute}. This pattern allows computer ops
 * to be used directly as consumers as needed.
 * </p>
 *
 * @author Curtis Rueden
 * @author Gabriel Selzer
 */
public final class Computers {

	private Computers() {
		// NB: Prevent instantiation of utility class.
	}

	/**
	 * All known computer types and their arities. The types are indexed, first by the number of <em>pure inputs</em>
	 * in their functional method, then by the (1-indexed) <em>index</em> of the preallocated output.
	 * <p>
	 * Note that this data structure is populated at the bottom of the file, so it does not impede quick browsing
	 * of the functional interfaces
	 * </p>
	 */
	public static final Class<?>[][] ALL_COMPUTERS = new Class<?>[16 + 1][16 + 1];
	public static final HashMap<Class<?>, Integer> ALL_ARITIES = new HashMap<>();

	/**
	 * @return {@code true} if the given type is a known
	 *         computer type, {@code false} otherwise. <br>
	 *         Note that only the type itself and not its type hierarchy is
	 *         considered.
	 * @throws NullPointerException If {@code c} is {@code null}.
	 */
	public static boolean isComputer(Class<?> c) {
		return ALL_ARITIES.containsKey(c);
	}

	/**
	 * @param arity an {@code int} corresponding to a {@code Computer} of that
	 *          arity.
	 * @return the {@code Computer} of arity {@code arity}.
	 * @throws IllegalArgumentException iff there is no known {@code Computer} of
	 *           arity {@code arity}.
	 */
	public static Class<?> computerOfArity(int arity) {
		// If pos is not given, we assume we're looking for one of the ArityX implementations,
		// which is theoretically equivalent to a ArityX_X.
		return computerOfArity(arity, arity);
	}

	/**
	 * @param arity an {@code int} corresponding to a {@code Computer} of that
	 *          arity.
	 * @param pos an {@code int} corresponding to the position of the
	 *          {@code Computer}'s output
	 * @return the {@code Computer} of arity {@code arity}.
	 * @throws IllegalArgumentException iff there is no known {@code Computer} of
	 *           arity {@code arity}.
	 */
	public static Class<?> computerOfArity(int arity, int pos) {
		if ((arity > ALL_COMPUTERS.length + 1) || pos > arity) {
			throw new IllegalArgumentException( //
				"No Computer of arity " + arity + " with output index " + pos //
			);
		}
		return ALL_COMPUTERS[arity][pos];
	}

	/**
	 * @param c the {@link Class} of unknown arity
	 * @return the arity of {@code c}, or {@code -1} if {@code c} is <b>not</b> a
	 *         {@code Computer}.
	 */
	public static int arityOf(Class<?> c) {
		return ALL_ARITIES.getOrDefault(c, -1);
	}


	@FunctionalInterface
	public interface Arity0<O> extends
		Consumer<O>
	{

		void compute(@Container O out);

		@Override
		default void accept(final O out)
		{
			compute(out);
		}
	}

	@FunctionalInterface
	public interface Arity1<I, O> extends
		BiConsumer<I, O>
	{

		void compute(I in, @Container O out);

		@Override
		default void accept(final I in, final O out)
		{
			compute(in, out);
		}
	}

	@FunctionalInterface
	public interface Arity2<I1, I2, O> extends
		Consumers.Arity3<I1, I2, O>
	{

		void compute(I1 in1, I2 in2, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final O out)
		{
			compute(in1, in2, out);
		}
	}

	@FunctionalInterface
	public interface Arity3<I1, I2, I3, O> extends
		Consumers.Arity4<I1, I2, I3, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final O out)
		{
			compute(in1, in2, in3, out);
		}
	}

	@FunctionalInterface
	public interface Arity4<I1, I2, I3, I4, O> extends
		Consumers.Arity5<I1, I2, I3, I4, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final O out)
		{
			compute(in1, in2, in3, in4, out);
		}
	}

	@FunctionalInterface
	public interface Arity5<I1, I2, I3, I4, I5, O> extends
		Consumers.Arity6<I1, I2, I3, I4, I5, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final O out)
		{
			compute(in1, in2, in3, in4, in5, out);
		}
	}

	@FunctionalInterface
	public interface Arity6<I1, I2, I3, I4, I5, I6, O> extends
		Consumers.Arity7<I1, I2, I3, I4, I5, I6, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, out);
		}
	}

	@FunctionalInterface
	public interface Arity7<I1, I2, I3, I4, I5, I6, I7, O> extends
		Consumers.Arity8<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, out);
		}
	}

	@FunctionalInterface
	public interface Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O> extends
		Consumers.Arity9<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, out);
		}
	}

	@FunctionalInterface
	public interface Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O> extends
		Consumers.Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, out);
		}
	}

	@FunctionalInterface
	public interface Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O> extends
		Consumers.Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out);
		}
	}

	@FunctionalInterface
	public interface Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O> extends
		Consumers.Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final I11 in11, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, out);
		}
	}

	@FunctionalInterface
	public interface Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O> extends
		Consumers.Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final I11 in11, final I12 in12, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, out);
		}
	}

	@FunctionalInterface
	public interface Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O> extends
		Consumers.Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final I11 in11, final I12 in12, final I13 in13, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, out);
		}
	}

	@FunctionalInterface
	public interface Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O> extends
		Consumers.Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final I11 in11, final I12 in12, final I13 in13, final I14 in14, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, out);
		}
	}

	@FunctionalInterface
	public interface Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O> extends
		Consumers.Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final I11 in11, final I12 in12, final I13 in13, final I14 in14, final I15 in15, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, out);
		}
	}

	@FunctionalInterface
	public interface Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O> extends
		Consumers.Arity17<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out);

		@Override
		default void accept(final I1 in1, final I2 in2, final I3 in3, final I4 in4, final I5 in5, final I6 in6, final I7 in7, final I8 in8, final I9 in9, final I10 in10, final I11 in11, final I12 in12, final I13 in13, final I14 in14, final I15 in15, final I16 in16, final O out)
		{
			compute(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16, out);
		}
	}


	@FunctionalInterface
	public interface Arity1_1<O, I> extends
		Arity1<I, O>
	{

		void compute1(@Container O out, I in);

		@Override
		default void compute(I in, @Container O out)
		{
			compute1(out, in);
		}
	}

	@FunctionalInterface
	public interface Arity2_1<O, I1, I2> extends
		Arity2<I1, I2, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2);

		@Override
		default void compute(I1 in1, I2 in2, @Container O out)
		{
			compute1(out, in1, in2);
		}
	}

	@FunctionalInterface
	public interface Arity2_2<I1, O, I2> extends
		Arity2<I1, I2, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2);

		@Override
		default void compute(I1 in1, I2 in2, @Container O out)
		{
			compute2(in1, out, in2);
		}
	}

	@FunctionalInterface
	public interface Arity3_1<O, I1, I2, I3> extends
		Arity3<I1, I2, I3, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, @Container O out)
		{
			compute1(out, in1, in2, in3);
		}
	}

	@FunctionalInterface
	public interface Arity3_2<I1, O, I2, I3> extends
		Arity3<I1, I2, I3, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, @Container O out)
		{
			compute2(in1, out, in2, in3);
		}
	}

	@FunctionalInterface
	public interface Arity3_3<I1, I2, O, I3> extends
		Arity3<I1, I2, I3, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, @Container O out)
		{
			compute3(in1, in2, out, in3);
		}
	}

	@FunctionalInterface
	public interface Arity4_1<O, I1, I2, I3, I4> extends
		Arity4<I1, I2, I3, I4, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out)
		{
			compute1(out, in1, in2, in3, in4);
		}
	}

	@FunctionalInterface
	public interface Arity4_2<I1, O, I2, I3, I4> extends
		Arity4<I1, I2, I3, I4, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out)
		{
			compute2(in1, out, in2, in3, in4);
		}
	}

	@FunctionalInterface
	public interface Arity4_3<I1, I2, O, I3, I4> extends
		Arity4<I1, I2, I3, I4, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out)
		{
			compute3(in1, in2, out, in3, in4);
		}
	}

	@FunctionalInterface
	public interface Arity4_4<I1, I2, I3, O, I4> extends
		Arity4<I1, I2, I3, I4, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out)
		{
			compute4(in1, in2, in3, out, in4);
		}
	}

	@FunctionalInterface
	public interface Arity5_1<O, I1, I2, I3, I4, I5> extends
		Arity5<I1, I2, I3, I4, I5, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5);
		}
	}

	@FunctionalInterface
	public interface Arity5_2<I1, O, I2, I3, I4, I5> extends
		Arity5<I1, I2, I3, I4, I5, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5);
		}
	}

	@FunctionalInterface
	public interface Arity5_3<I1, I2, O, I3, I4, I5> extends
		Arity5<I1, I2, I3, I4, I5, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5);
		}
	}

	@FunctionalInterface
	public interface Arity5_4<I1, I2, I3, O, I4, I5> extends
		Arity5<I1, I2, I3, I4, I5, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5);
		}
	}

	@FunctionalInterface
	public interface Arity5_5<I1, I2, I3, I4, O, I5> extends
		Arity5<I1, I2, I3, I4, I5, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5);
		}
	}

	@FunctionalInterface
	public interface Arity6_1<O, I1, I2, I3, I4, I5, I6> extends
		Arity6<I1, I2, I3, I4, I5, I6, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6);
		}
	}

	@FunctionalInterface
	public interface Arity6_2<I1, O, I2, I3, I4, I5, I6> extends
		Arity6<I1, I2, I3, I4, I5, I6, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6);
		}
	}

	@FunctionalInterface
	public interface Arity6_3<I1, I2, O, I3, I4, I5, I6> extends
		Arity6<I1, I2, I3, I4, I5, I6, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6);
		}
	}

	@FunctionalInterface
	public interface Arity6_4<I1, I2, I3, O, I4, I5, I6> extends
		Arity6<I1, I2, I3, I4, I5, I6, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6);
		}
	}

	@FunctionalInterface
	public interface Arity6_5<I1, I2, I3, I4, O, I5, I6> extends
		Arity6<I1, I2, I3, I4, I5, I6, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6);
		}
	}

	@FunctionalInterface
	public interface Arity6_6<I1, I2, I3, I4, I5, O, I6> extends
		Arity6<I1, I2, I3, I4, I5, I6, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6);
		}
	}

	@FunctionalInterface
	public interface Arity7_1<O, I1, I2, I3, I4, I5, I6, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7);
		}
	}

	@FunctionalInterface
	public interface Arity7_2<I1, O, I2, I3, I4, I5, I6, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7);
		}
	}

	@FunctionalInterface
	public interface Arity7_3<I1, I2, O, I3, I4, I5, I6, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7);
		}
	}

	@FunctionalInterface
	public interface Arity7_4<I1, I2, I3, O, I4, I5, I6, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7);
		}
	}

	@FunctionalInterface
	public interface Arity7_5<I1, I2, I3, I4, O, I5, I6, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7);
		}
	}

	@FunctionalInterface
	public interface Arity7_6<I1, I2, I3, I4, I5, O, I6, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7);
		}
	}

	@FunctionalInterface
	public interface Arity7_7<I1, I2, I3, I4, I5, I6, O, I7> extends
		Arity7<I1, I2, I3, I4, I5, I6, I7, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7);
		}
	}

	@FunctionalInterface
	public interface Arity8_1<O, I1, I2, I3, I4, I5, I6, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_2<I1, O, I2, I3, I4, I5, I6, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_3<I1, I2, O, I3, I4, I5, I6, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_4<I1, I2, I3, O, I4, I5, I6, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_5<I1, I2, I3, I4, O, I5, I6, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_6<I1, I2, I3, I4, I5, O, I6, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_7<I1, I2, I3, I4, I5, I6, O, I7, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8);
		}
	}

	@FunctionalInterface
	public interface Arity8_8<I1, I2, I3, I4, I5, I6, I7, O, I8> extends
		Arity8<I1, I2, I3, I4, I5, I6, I7, I8, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8);
		}
	}

	@FunctionalInterface
	public interface Arity9_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9);
		}
	}

	@FunctionalInterface
	public interface Arity9_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9> extends
		Arity9<I1, I2, I3, I4, I5, I6, I7, I8, I9, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9);
		}
	}

	@FunctionalInterface
	public interface Arity10_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10);
		}
	}

	@FunctionalInterface
	public interface Arity10_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10> extends
		Arity10<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10);
		}
	}

	@FunctionalInterface
	public interface Arity11_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10, in11);
		}
	}

	@FunctionalInterface
	public interface Arity11_11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O, I11> extends
		Arity11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O>
	{

		void compute11(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out, I11 in11);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out)
		{
			compute11(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out, in11);
		}
	}

	@FunctionalInterface
	public interface Arity12_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O, I11, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute11(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out, I11 in11, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute11(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out, in11, in12);
		}
	}

	@FunctionalInterface
	public interface Arity12_12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O, I12> extends
		Arity12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O>
	{

		void compute12(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out, I12 in12);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out)
		{
			compute12(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, out, in12);
		}
	}

	@FunctionalInterface
	public interface Arity13_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O, I11, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute11(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out, I11 in11, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute11(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out, in11, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O, I12, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute12(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out, I12 in12, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute12(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, out, in12, in13);
		}
	}

	@FunctionalInterface
	public interface Arity13_13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O, I13> extends
		Arity13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O>
	{

		void compute13(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out, I13 in13);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out)
		{
			compute13(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, out, in13);
		}
	}

	@FunctionalInterface
	public interface Arity14_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O, I11, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute11(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out, I11 in11, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute11(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out, in11, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O, I12, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute12(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out, I12 in12, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute12(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, out, in12, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O, I13, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute13(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out, I13 in13, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute13(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, out, in13, in14);
		}
	}

	@FunctionalInterface
	public interface Arity14_14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O, I14> extends
		Arity14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O>
	{

		void compute14(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out, I14 in14);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out)
		{
			compute14(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, out, in14);
		}
	}

	@FunctionalInterface
	public interface Arity15_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O, I11, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute11(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute11(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out, in11, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O, I12, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute12(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out, I12 in12, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute12(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, out, in12, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O, I13, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute13(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out, I13 in13, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute13(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, out, in13, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O, I14, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute14(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out, I14 in14, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute14(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, out, in14, in15);
		}
	}

	@FunctionalInterface
	public interface Arity15_15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O, I15> extends
		Arity15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O>
	{

		void compute15(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out, I15 in15);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out)
		{
			compute15(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, out, in15);
		}
	}

	@FunctionalInterface
	public interface Arity16_1<O, I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute1(@Container O out, I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute1(out, in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_2<I1, O, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute2(I1 in1, @Container O out, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute2(in1, out, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_3<I1, I2, O, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute3(I1 in1, I2 in2, @Container O out, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute3(in1, in2, out, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_4<I1, I2, I3, O, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute4(I1 in1, I2 in2, I3 in3, @Container O out, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute4(in1, in2, in3, out, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_5<I1, I2, I3, I4, O, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute5(I1 in1, I2 in2, I3 in3, I4 in4, @Container O out, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute5(in1, in2, in3, in4, out, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_6<I1, I2, I3, I4, I5, O, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute6(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, @Container O out, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute6(in1, in2, in3, in4, in5, out, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_7<I1, I2, I3, I4, I5, I6, O, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute7(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, @Container O out, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute7(in1, in2, in3, in4, in5, in6, out, in7, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_8<I1, I2, I3, I4, I5, I6, I7, O, I8, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute8(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, @Container O out, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute8(in1, in2, in3, in4, in5, in6, in7, out, in8, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_9<I1, I2, I3, I4, I5, I6, I7, I8, O, I9, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute9(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, @Container O out, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute9(in1, in2, in3, in4, in5, in6, in7, in8, out, in9, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_10<I1, I2, I3, I4, I5, I6, I7, I8, I9, O, I10, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute10(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, @Container O out, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute10(in1, in2, in3, in4, in5, in6, in7, in8, in9, out, in10, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_11<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, O, I11, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute11(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, @Container O out, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute11(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, out, in11, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_12<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, O, I12, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute12(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, @Container O out, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute12(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, out, in12, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_13<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, O, I13, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute13(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, @Container O out, I13 in13, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute13(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, out, in13, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_14<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, O, I14, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute14(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, @Container O out, I14 in14, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute14(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, out, in14, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_15<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, O, I15, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute15(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, @Container O out, I15 in15, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute15(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, out, in15, in16);
		}
	}

	@FunctionalInterface
	public interface Arity16_16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, O, I16> extends
		Arity16<I1, I2, I3, I4, I5, I6, I7, I8, I9, I10, I11, I12, I13, I14, I15, I16, O>
	{

		void compute16(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, @Container O out, I16 in16);

		@Override
		default void compute(I1 in1, I2 in2, I3 in3, I4 in4, I5 in5, I6 in6, I7 in7, I8 in8, I9 in9, I10 in10, I11 in11, I12 in12, I13 in13, I14 in14, I15 in15, I16 in16, @Container O out)
		{
			compute16(in1, in2, in3, in4, in5, in6, in7, in8, in9, in10, in11, in12, in13, in14, in15, out, in16);
		}
	}

	static {
		ALL_COMPUTERS[0][0] = Computers.Arity0.class;
		ALL_ARITIES.put(Computers.Arity0.class, 0);
		ALL_COMPUTERS[1][0] = Computers.Arity1_1.class;
		ALL_ARITIES.put(Computers.Arity1_1.class, 1);
		ALL_COMPUTERS[1][1] = Computers.Arity1.class;
		ALL_ARITIES.put(Computers.Arity1.class, 1);
		ALL_COMPUTERS[2][0] = Computers.Arity2_1.class;
		ALL_ARITIES.put(Computers.Arity2_1.class, 2);
		ALL_COMPUTERS[2][1] = Computers.Arity2_2.class;
		ALL_ARITIES.put(Computers.Arity2_2.class, 2);
		ALL_COMPUTERS[2][2] = Computers.Arity2.class;
		ALL_ARITIES.put(Computers.Arity2.class, 2);
		ALL_COMPUTERS[3][0] = Computers.Arity3_1.class;
		ALL_ARITIES.put(Computers.Arity3_1.class, 3);
		ALL_COMPUTERS[3][1] = Computers.Arity3_2.class;
		ALL_ARITIES.put(Computers.Arity3_2.class, 3);
		ALL_COMPUTERS[3][2] = Computers.Arity3_3.class;
		ALL_ARITIES.put(Computers.Arity3_3.class, 3);
		ALL_COMPUTERS[3][3] = Computers.Arity3.class;
		ALL_ARITIES.put(Computers.Arity3.class, 3);
		ALL_COMPUTERS[4][0] = Computers.Arity4_1.class;
		ALL_ARITIES.put(Computers.Arity4_1.class, 4);
		ALL_COMPUTERS[4][1] = Computers.Arity4_2.class;
		ALL_ARITIES.put(Computers.Arity4_2.class, 4);
		ALL_COMPUTERS[4][2] = Computers.Arity4_3.class;
		ALL_ARITIES.put(Computers.Arity4_3.class, 4);
		ALL_COMPUTERS[4][3] = Computers.Arity4_4.class;
		ALL_ARITIES.put(Computers.Arity4_4.class, 4);
		ALL_COMPUTERS[4][4] = Computers.Arity4.class;
		ALL_ARITIES.put(Computers.Arity4.class, 4);
		ALL_COMPUTERS[5][0] = Computers.Arity5_1.class;
		ALL_ARITIES.put(Computers.Arity5_1.class, 5);
		ALL_COMPUTERS[5][1] = Computers.Arity5_2.class;
		ALL_ARITIES.put(Computers.Arity5_2.class, 5);
		ALL_COMPUTERS[5][2] = Computers.Arity5_3.class;
		ALL_ARITIES.put(Computers.Arity5_3.class, 5);
		ALL_COMPUTERS[5][3] = Computers.Arity5_4.class;
		ALL_ARITIES.put(Computers.Arity5_4.class, 5);
		ALL_COMPUTERS[5][4] = Computers.Arity5_5.class;
		ALL_ARITIES.put(Computers.Arity5_5.class, 5);
		ALL_COMPUTERS[5][5] = Computers.Arity5.class;
		ALL_ARITIES.put(Computers.Arity5.class, 5);
		ALL_COMPUTERS[6][0] = Computers.Arity6_1.class;
		ALL_ARITIES.put(Computers.Arity6_1.class, 6);
		ALL_COMPUTERS[6][1] = Computers.Arity6_2.class;
		ALL_ARITIES.put(Computers.Arity6_2.class, 6);
		ALL_COMPUTERS[6][2] = Computers.Arity6_3.class;
		ALL_ARITIES.put(Computers.Arity6_3.class, 6);
		ALL_COMPUTERS[6][3] = Computers.Arity6_4.class;
		ALL_ARITIES.put(Computers.Arity6_4.class, 6);
		ALL_COMPUTERS[6][4] = Computers.Arity6_5.class;
		ALL_ARITIES.put(Computers.Arity6_5.class, 6);
		ALL_COMPUTERS[6][5] = Computers.Arity6_6.class;
		ALL_ARITIES.put(Computers.Arity6_6.class, 6);
		ALL_COMPUTERS[6][6] = Computers.Arity6.class;
		ALL_ARITIES.put(Computers.Arity6.class, 6);
		ALL_COMPUTERS[7][0] = Computers.Arity7_1.class;
		ALL_ARITIES.put(Computers.Arity7_1.class, 7);
		ALL_COMPUTERS[7][1] = Computers.Arity7_2.class;
		ALL_ARITIES.put(Computers.Arity7_2.class, 7);
		ALL_COMPUTERS[7][2] = Computers.Arity7_3.class;
		ALL_ARITIES.put(Computers.Arity7_3.class, 7);
		ALL_COMPUTERS[7][3] = Computers.Arity7_4.class;
		ALL_ARITIES.put(Computers.Arity7_4.class, 7);
		ALL_COMPUTERS[7][4] = Computers.Arity7_5.class;
		ALL_ARITIES.put(Computers.Arity7_5.class, 7);
		ALL_COMPUTERS[7][5] = Computers.Arity7_6.class;
		ALL_ARITIES.put(Computers.Arity7_6.class, 7);
		ALL_COMPUTERS[7][6] = Computers.Arity7_7.class;
		ALL_ARITIES.put(Computers.Arity7_7.class, 7);
		ALL_COMPUTERS[7][7] = Computers.Arity7.class;
		ALL_ARITIES.put(Computers.Arity7.class, 7);
		ALL_COMPUTERS[8][0] = Computers.Arity8_1.class;
		ALL_ARITIES.put(Computers.Arity8_1.class, 8);
		ALL_COMPUTERS[8][1] = Computers.Arity8_2.class;
		ALL_ARITIES.put(Computers.Arity8_2.class, 8);
		ALL_COMPUTERS[8][2] = Computers.Arity8_3.class;
		ALL_ARITIES.put(Computers.Arity8_3.class, 8);
		ALL_COMPUTERS[8][3] = Computers.Arity8_4.class;
		ALL_ARITIES.put(Computers.Arity8_4.class, 8);
		ALL_COMPUTERS[8][4] = Computers.Arity8_5.class;
		ALL_ARITIES.put(Computers.Arity8_5.class, 8);
		ALL_COMPUTERS[8][5] = Computers.Arity8_6.class;
		ALL_ARITIES.put(Computers.Arity8_6.class, 8);
		ALL_COMPUTERS[8][6] = Computers.Arity8_7.class;
		ALL_ARITIES.put(Computers.Arity8_7.class, 8);
		ALL_COMPUTERS[8][7] = Computers.Arity8_8.class;
		ALL_ARITIES.put(Computers.Arity8_8.class, 8);
		ALL_COMPUTERS[8][8] = Computers.Arity8.class;
		ALL_ARITIES.put(Computers.Arity8.class, 8);
		ALL_COMPUTERS[9][0] = Computers.Arity9_1.class;
		ALL_ARITIES.put(Computers.Arity9_1.class, 9);
		ALL_COMPUTERS[9][1] = Computers.Arity9_2.class;
		ALL_ARITIES.put(Computers.Arity9_2.class, 9);
		ALL_COMPUTERS[9][2] = Computers.Arity9_3.class;
		ALL_ARITIES.put(Computers.Arity9_3.class, 9);
		ALL_COMPUTERS[9][3] = Computers.Arity9_4.class;
		ALL_ARITIES.put(Computers.Arity9_4.class, 9);
		ALL_COMPUTERS[9][4] = Computers.Arity9_5.class;
		ALL_ARITIES.put(Computers.Arity9_5.class, 9);
		ALL_COMPUTERS[9][5] = Computers.Arity9_6.class;
		ALL_ARITIES.put(Computers.Arity9_6.class, 9);
		ALL_COMPUTERS[9][6] = Computers.Arity9_7.class;
		ALL_ARITIES.put(Computers.Arity9_7.class, 9);
		ALL_COMPUTERS[9][7] = Computers.Arity9_8.class;
		ALL_ARITIES.put(Computers.Arity9_8.class, 9);
		ALL_COMPUTERS[9][8] = Computers.Arity9_9.class;
		ALL_ARITIES.put(Computers.Arity9_9.class, 9);
		ALL_COMPUTERS[9][9] = Computers.Arity9.class;
		ALL_ARITIES.put(Computers.Arity9.class, 9);
		ALL_COMPUTERS[10][0] = Computers.Arity10_1.class;
		ALL_ARITIES.put(Computers.Arity10_1.class, 10);
		ALL_COMPUTERS[10][1] = Computers.Arity10_2.class;
		ALL_ARITIES.put(Computers.Arity10_2.class, 10);
		ALL_COMPUTERS[10][2] = Computers.Arity10_3.class;
		ALL_ARITIES.put(Computers.Arity10_3.class, 10);
		ALL_COMPUTERS[10][3] = Computers.Arity10_4.class;
		ALL_ARITIES.put(Computers.Arity10_4.class, 10);
		ALL_COMPUTERS[10][4] = Computers.Arity10_5.class;
		ALL_ARITIES.put(Computers.Arity10_5.class, 10);
		ALL_COMPUTERS[10][5] = Computers.Arity10_6.class;
		ALL_ARITIES.put(Computers.Arity10_6.class, 10);
		ALL_COMPUTERS[10][6] = Computers.Arity10_7.class;
		ALL_ARITIES.put(Computers.Arity10_7.class, 10);
		ALL_COMPUTERS[10][7] = Computers.Arity10_8.class;
		ALL_ARITIES.put(Computers.Arity10_8.class, 10);
		ALL_COMPUTERS[10][8] = Computers.Arity10_9.class;
		ALL_ARITIES.put(Computers.Arity10_9.class, 10);
		ALL_COMPUTERS[10][9] = Computers.Arity10_10.class;
		ALL_ARITIES.put(Computers.Arity10_10.class, 10);
		ALL_COMPUTERS[10][10] = Computers.Arity10.class;
		ALL_ARITIES.put(Computers.Arity10.class, 10);
		ALL_COMPUTERS[11][0] = Computers.Arity11_1.class;
		ALL_ARITIES.put(Computers.Arity11_1.class, 11);
		ALL_COMPUTERS[11][1] = Computers.Arity11_2.class;
		ALL_ARITIES.put(Computers.Arity11_2.class, 11);
		ALL_COMPUTERS[11][2] = Computers.Arity11_3.class;
		ALL_ARITIES.put(Computers.Arity11_3.class, 11);
		ALL_COMPUTERS[11][3] = Computers.Arity11_4.class;
		ALL_ARITIES.put(Computers.Arity11_4.class, 11);
		ALL_COMPUTERS[11][4] = Computers.Arity11_5.class;
		ALL_ARITIES.put(Computers.Arity11_5.class, 11);
		ALL_COMPUTERS[11][5] = Computers.Arity11_6.class;
		ALL_ARITIES.put(Computers.Arity11_6.class, 11);
		ALL_COMPUTERS[11][6] = Computers.Arity11_7.class;
		ALL_ARITIES.put(Computers.Arity11_7.class, 11);
		ALL_COMPUTERS[11][7] = Computers.Arity11_8.class;
		ALL_ARITIES.put(Computers.Arity11_8.class, 11);
		ALL_COMPUTERS[11][8] = Computers.Arity11_9.class;
		ALL_ARITIES.put(Computers.Arity11_9.class, 11);
		ALL_COMPUTERS[11][9] = Computers.Arity11_10.class;
		ALL_ARITIES.put(Computers.Arity11_10.class, 11);
		ALL_COMPUTERS[11][10] = Computers.Arity11_11.class;
		ALL_ARITIES.put(Computers.Arity11_11.class, 11);
		ALL_COMPUTERS[11][11] = Computers.Arity11.class;
		ALL_ARITIES.put(Computers.Arity11.class, 11);
		ALL_COMPUTERS[12][0] = Computers.Arity12_1.class;
		ALL_ARITIES.put(Computers.Arity12_1.class, 12);
		ALL_COMPUTERS[12][1] = Computers.Arity12_2.class;
		ALL_ARITIES.put(Computers.Arity12_2.class, 12);
		ALL_COMPUTERS[12][2] = Computers.Arity12_3.class;
		ALL_ARITIES.put(Computers.Arity12_3.class, 12);
		ALL_COMPUTERS[12][3] = Computers.Arity12_4.class;
		ALL_ARITIES.put(Computers.Arity12_4.class, 12);
		ALL_COMPUTERS[12][4] = Computers.Arity12_5.class;
		ALL_ARITIES.put(Computers.Arity12_5.class, 12);
		ALL_COMPUTERS[12][5] = Computers.Arity12_6.class;
		ALL_ARITIES.put(Computers.Arity12_6.class, 12);
		ALL_COMPUTERS[12][6] = Computers.Arity12_7.class;
		ALL_ARITIES.put(Computers.Arity12_7.class, 12);
		ALL_COMPUTERS[12][7] = Computers.Arity12_8.class;
		ALL_ARITIES.put(Computers.Arity12_8.class, 12);
		ALL_COMPUTERS[12][8] = Computers.Arity12_9.class;
		ALL_ARITIES.put(Computers.Arity12_9.class, 12);
		ALL_COMPUTERS[12][9] = Computers.Arity12_10.class;
		ALL_ARITIES.put(Computers.Arity12_10.class, 12);
		ALL_COMPUTERS[12][10] = Computers.Arity12_11.class;
		ALL_ARITIES.put(Computers.Arity12_11.class, 12);
		ALL_COMPUTERS[12][11] = Computers.Arity12_12.class;
		ALL_ARITIES.put(Computers.Arity12_12.class, 12);
		ALL_COMPUTERS[12][12] = Computers.Arity12.class;
		ALL_ARITIES.put(Computers.Arity12.class, 12);
		ALL_COMPUTERS[13][0] = Computers.Arity13_1.class;
		ALL_ARITIES.put(Computers.Arity13_1.class, 13);
		ALL_COMPUTERS[13][1] = Computers.Arity13_2.class;
		ALL_ARITIES.put(Computers.Arity13_2.class, 13);
		ALL_COMPUTERS[13][2] = Computers.Arity13_3.class;
		ALL_ARITIES.put(Computers.Arity13_3.class, 13);
		ALL_COMPUTERS[13][3] = Computers.Arity13_4.class;
		ALL_ARITIES.put(Computers.Arity13_4.class, 13);
		ALL_COMPUTERS[13][4] = Computers.Arity13_5.class;
		ALL_ARITIES.put(Computers.Arity13_5.class, 13);
		ALL_COMPUTERS[13][5] = Computers.Arity13_6.class;
		ALL_ARITIES.put(Computers.Arity13_6.class, 13);
		ALL_COMPUTERS[13][6] = Computers.Arity13_7.class;
		ALL_ARITIES.put(Computers.Arity13_7.class, 13);
		ALL_COMPUTERS[13][7] = Computers.Arity13_8.class;
		ALL_ARITIES.put(Computers.Arity13_8.class, 13);
		ALL_COMPUTERS[13][8] = Computers.Arity13_9.class;
		ALL_ARITIES.put(Computers.Arity13_9.class, 13);
		ALL_COMPUTERS[13][9] = Computers.Arity13_10.class;
		ALL_ARITIES.put(Computers.Arity13_10.class, 13);
		ALL_COMPUTERS[13][10] = Computers.Arity13_11.class;
		ALL_ARITIES.put(Computers.Arity13_11.class, 13);
		ALL_COMPUTERS[13][11] = Computers.Arity13_12.class;
		ALL_ARITIES.put(Computers.Arity13_12.class, 13);
		ALL_COMPUTERS[13][12] = Computers.Arity13_13.class;
		ALL_ARITIES.put(Computers.Arity13_13.class, 13);
		ALL_COMPUTERS[13][13] = Computers.Arity13.class;
		ALL_ARITIES.put(Computers.Arity13.class, 13);
		ALL_COMPUTERS[14][0] = Computers.Arity14_1.class;
		ALL_ARITIES.put(Computers.Arity14_1.class, 14);
		ALL_COMPUTERS[14][1] = Computers.Arity14_2.class;
		ALL_ARITIES.put(Computers.Arity14_2.class, 14);
		ALL_COMPUTERS[14][2] = Computers.Arity14_3.class;
		ALL_ARITIES.put(Computers.Arity14_3.class, 14);
		ALL_COMPUTERS[14][3] = Computers.Arity14_4.class;
		ALL_ARITIES.put(Computers.Arity14_4.class, 14);
		ALL_COMPUTERS[14][4] = Computers.Arity14_5.class;
		ALL_ARITIES.put(Computers.Arity14_5.class, 14);
		ALL_COMPUTERS[14][5] = Computers.Arity14_6.class;
		ALL_ARITIES.put(Computers.Arity14_6.class, 14);
		ALL_COMPUTERS[14][6] = Computers.Arity14_7.class;
		ALL_ARITIES.put(Computers.Arity14_7.class, 14);
		ALL_COMPUTERS[14][7] = Computers.Arity14_8.class;
		ALL_ARITIES.put(Computers.Arity14_8.class, 14);
		ALL_COMPUTERS[14][8] = Computers.Arity14_9.class;
		ALL_ARITIES.put(Computers.Arity14_9.class, 14);
		ALL_COMPUTERS[14][9] = Computers.Arity14_10.class;
		ALL_ARITIES.put(Computers.Arity14_10.class, 14);
		ALL_COMPUTERS[14][10] = Computers.Arity14_11.class;
		ALL_ARITIES.put(Computers.Arity14_11.class, 14);
		ALL_COMPUTERS[14][11] = Computers.Arity14_12.class;
		ALL_ARITIES.put(Computers.Arity14_12.class, 14);
		ALL_COMPUTERS[14][12] = Computers.Arity14_13.class;
		ALL_ARITIES.put(Computers.Arity14_13.class, 14);
		ALL_COMPUTERS[14][13] = Computers.Arity14_14.class;
		ALL_ARITIES.put(Computers.Arity14_14.class, 14);
		ALL_COMPUTERS[14][14] = Computers.Arity14.class;
		ALL_ARITIES.put(Computers.Arity14.class, 14);
		ALL_COMPUTERS[15][0] = Computers.Arity15_1.class;
		ALL_ARITIES.put(Computers.Arity15_1.class, 15);
		ALL_COMPUTERS[15][1] = Computers.Arity15_2.class;
		ALL_ARITIES.put(Computers.Arity15_2.class, 15);
		ALL_COMPUTERS[15][2] = Computers.Arity15_3.class;
		ALL_ARITIES.put(Computers.Arity15_3.class, 15);
		ALL_COMPUTERS[15][3] = Computers.Arity15_4.class;
		ALL_ARITIES.put(Computers.Arity15_4.class, 15);
		ALL_COMPUTERS[15][4] = Computers.Arity15_5.class;
		ALL_ARITIES.put(Computers.Arity15_5.class, 15);
		ALL_COMPUTERS[15][5] = Computers.Arity15_6.class;
		ALL_ARITIES.put(Computers.Arity15_6.class, 15);
		ALL_COMPUTERS[15][6] = Computers.Arity15_7.class;
		ALL_ARITIES.put(Computers.Arity15_7.class, 15);
		ALL_COMPUTERS[15][7] = Computers.Arity15_8.class;
		ALL_ARITIES.put(Computers.Arity15_8.class, 15);
		ALL_COMPUTERS[15][8] = Computers.Arity15_9.class;
		ALL_ARITIES.put(Computers.Arity15_9.class, 15);
		ALL_COMPUTERS[15][9] = Computers.Arity15_10.class;
		ALL_ARITIES.put(Computers.Arity15_10.class, 15);
		ALL_COMPUTERS[15][10] = Computers.Arity15_11.class;
		ALL_ARITIES.put(Computers.Arity15_11.class, 15);
		ALL_COMPUTERS[15][11] = Computers.Arity15_12.class;
		ALL_ARITIES.put(Computers.Arity15_12.class, 15);
		ALL_COMPUTERS[15][12] = Computers.Arity15_13.class;
		ALL_ARITIES.put(Computers.Arity15_13.class, 15);
		ALL_COMPUTERS[15][13] = Computers.Arity15_14.class;
		ALL_ARITIES.put(Computers.Arity15_14.class, 15);
		ALL_COMPUTERS[15][14] = Computers.Arity15_15.class;
		ALL_ARITIES.put(Computers.Arity15_15.class, 15);
		ALL_COMPUTERS[15][15] = Computers.Arity15.class;
		ALL_ARITIES.put(Computers.Arity15.class, 15);
		ALL_COMPUTERS[16][0] = Computers.Arity16_1.class;
		ALL_ARITIES.put(Computers.Arity16_1.class, 16);
		ALL_COMPUTERS[16][1] = Computers.Arity16_2.class;
		ALL_ARITIES.put(Computers.Arity16_2.class, 16);
		ALL_COMPUTERS[16][2] = Computers.Arity16_3.class;
		ALL_ARITIES.put(Computers.Arity16_3.class, 16);
		ALL_COMPUTERS[16][3] = Computers.Arity16_4.class;
		ALL_ARITIES.put(Computers.Arity16_4.class, 16);
		ALL_COMPUTERS[16][4] = Computers.Arity16_5.class;
		ALL_ARITIES.put(Computers.Arity16_5.class, 16);
		ALL_COMPUTERS[16][5] = Computers.Arity16_6.class;
		ALL_ARITIES.put(Computers.Arity16_6.class, 16);
		ALL_COMPUTERS[16][6] = Computers.Arity16_7.class;
		ALL_ARITIES.put(Computers.Arity16_7.class, 16);
		ALL_COMPUTERS[16][7] = Computers.Arity16_8.class;
		ALL_ARITIES.put(Computers.Arity16_8.class, 16);
		ALL_COMPUTERS[16][8] = Computers.Arity16_9.class;
		ALL_ARITIES.put(Computers.Arity16_9.class, 16);
		ALL_COMPUTERS[16][9] = Computers.Arity16_10.class;
		ALL_ARITIES.put(Computers.Arity16_10.class, 16);
		ALL_COMPUTERS[16][10] = Computers.Arity16_11.class;
		ALL_ARITIES.put(Computers.Arity16_11.class, 16);
		ALL_COMPUTERS[16][11] = Computers.Arity16_12.class;
		ALL_ARITIES.put(Computers.Arity16_12.class, 16);
		ALL_COMPUTERS[16][12] = Computers.Arity16_13.class;
		ALL_ARITIES.put(Computers.Arity16_13.class, 16);
		ALL_COMPUTERS[16][13] = Computers.Arity16_14.class;
		ALL_ARITIES.put(Computers.Arity16_14.class, 16);
		ALL_COMPUTERS[16][14] = Computers.Arity16_15.class;
		ALL_ARITIES.put(Computers.Arity16_15.class, 16);
		ALL_COMPUTERS[16][15] = Computers.Arity16_16.class;
		ALL_ARITIES.put(Computers.Arity16_16.class, 16);
		ALL_COMPUTERS[16][16] = Computers.Arity16.class;
		ALL_ARITIES.put(Computers.Arity16.class, 16);
    }
}
