package uk.num.numlib.service;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.apache.commons.lang3.StringUtils;
import org.xbill.DNS.Record;
import uk.num.numlib.dns.DNSServices;
import uk.num.numlib.exc.*;
import uk.num.numlib.util.DomainLookupGenerator;
import uk.num.numlib.util.EmailLookupGenerator;
import uk.num.numlib.util.LookupGenerator;

@RequiredArgsConstructor
public class IndependentRecordLookupService {

    public static final int MAX_SANE_NUMID_LENGTH = 255;

    private final DNSServices dnsServices;

    /**
     * Loads a record content from the independent zone of the DNS. A null value is returned when the record is not
     * available.
     *
     * @param numIdName NUM id name for the record (for example domain name)
     * @param path      path of the record in the DNS
     * @param module    module for the record to retrieve
     * @param timeoutMs maximum time allowed to retrieve the record
     * @return a LoadRecordResponse or null
     * @throws NumInvalidParameterException if any of the supplied parameters are out of an acceptable range (catch NumException to handle oll exceptions)
     * @throws NumInvalidDNSQueryException  if the parameters cannot be used to generated a valid DNS query (catch NumException to handle oll exceptions)
     * @throws RrSetIncompleteException     if the NUM record is invalid (catch NumException to handle oll exceptions)
     * @throws RrSetNoHeadersException      if the NUM record is invalid (catch NumException to handle oll exceptions)
     * @throws RrSetHeaderFormatException   if the NUM record is invalid (catch NumException to handle oll exceptions)
     */
    public LoadRecordResponse loadRecordFromIndependentZone(@NonNull final String numIdName, @NonNull final String path, final int module, final int timeoutMs) throws
                                                                                                                                                                NumInvalidParameterException,
                                                                                                                                                                NumInvalidDNSQueryException,
                                                                                                                                                                RrSetIncompleteException,
                                                                                                                                                                RrSetNoHeadersException,
                                                                                                                                                                RrSetHeaderFormatException {
        validateParameters(numIdName, path, module, timeoutMs);

        final LookupGenerator generator = numIdName.contains("@") ? new EmailLookupGenerator(numIdName + path) : new DomainLookupGenerator(numIdName + path);

        try {
            final DNSServices.GetRecordResponse response = dnsServices.getRecordFromDnsNoCache(generator.getIndependentLocation(module), timeoutMs);
            final Record[] records = response.getRecords();

            if (records != null && records.length > 0) {
                final String record = dnsServices.rebuildTXTRecordContent(records);
                return new LoadRecordResponse(response.isSigned(), record);
            }
        } catch (final NumNoRecordAvailableException e) {
            return null;
        }
        return null;
    }

    private void validateParameters(final @NonNull String numIdName, final @NonNull String path, final int module, final int timeoutMs) throws
                                                                                                                                        NumInvalidParameterException {
        if (StringUtils.isBlank(numIdName) || numIdName.length() > MAX_SANE_NUMID_LENGTH) {
            throw new NumInvalidParameterException(String.format("numIdName is null, blank, or longer than " + MAX_SANE_NUMID_LENGTH + " characters: '%s'", numIdName));
        }
        if (path.length() > MAX_SANE_NUMID_LENGTH) {
            throw new NumInvalidParameterException(String.format("path is longer than " + MAX_SANE_NUMID_LENGTH + " characters: '%s'", path));
        }
        if (module < 0) {
            throw new NumInvalidParameterException("module number cannot be negative.");
        }
        if (timeoutMs <= 0) {
            throw new NumInvalidParameterException("timeoutMs cannot be negative or zero.");
        }
    }

    @Value
    public static class LoadRecordResponse {

        boolean isSigned;

        String record;

    }

}
