package uk.num.numlib.service;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.xbill.DNS.Record;
import uk.num.numlib.dns.DNSServices;
import uk.num.numlib.exc.*;

@RequiredArgsConstructor
class LookupServiceHelper {

    public static final int MAX_SANE_NUMID_LENGTH = 255;

    private final DNSServices dnsServices;

    public static void validateParameters(final @NonNull String numIdName, final @NonNull String path, final int module, final int timeoutMs) throws
                                                                                                                                              NumInvalidParameterException {
        if (StringUtils.isBlank(numIdName) || numIdName.length() > MAX_SANE_NUMID_LENGTH) {
            throw new NumInvalidParameterException(String.format("numIdName is null, blank, or longer than " + MAX_SANE_NUMID_LENGTH + " characters: '%s'", numIdName));
        }
        if (path.length() > MAX_SANE_NUMID_LENGTH) {
            throw new NumInvalidParameterException(String.format("path is longer than " + MAX_SANE_NUMID_LENGTH + " characters: '%s'", path));
        }
        if (module < 0) {
            throw new NumInvalidParameterException("module number cannot be negative.");
        }
        if (timeoutMs <= 0) {
            throw new NumInvalidParameterException("timeoutMs cannot be negative or zero.");
        }
    }

    /**
     * Loads a record content from the specified zone of the DNS. A null value is returned when the record is not
     * available.
     *
     * @param location  the DNS location, expected to be the FQDN of a NUM TXT record
     * @param timeoutMs maximum time allowed to retrieve the record
     * @return a LoadRecordResponse or null
     * @throws NumInvalidDNSQueryException if the parameters cannot be used to generated a valid DNS query (catch NumException to handle all exceptions)
     * @throws RrSetIncompleteException    if the NUM record is invalid (catch NumException to handle all exceptions)
     * @throws RrSetNoHeadersException     if the NUM record is invalid (catch NumException to handle all exceptions)
     * @throws RrSetHeaderFormatException  if the NUM record is invalid (catch NumException to handle all exceptions)
     */
    public LoadRecordResponse loadRecordFromZone(final int timeoutMs, @NonNull String location) throws
                                                                                                NumInvalidDNSQueryException,
                                                                                                RrSetIncompleteException,
                                                                                                RrSetNoHeadersException,
                                                                                                RrSetHeaderFormatException {
        try {
            final DNSServices.GetRecordResponse response = dnsServices.getRecordFromDnsNoCache(location, timeoutMs);
            if (response != null) {
                final Record[] records = response.getRecords();

                if (records != null && records.length > 0) {
                    final String record = dnsServices.rebuildTXTRecordContent(records);
                    return new LoadRecordResponse(response.isSigned(), record);
                }
            }
        } catch (final NumNoRecordAvailableException e) {
            return null;
        }
        return null;
    }

}
