package cloud.proxi.sdkv3.api

import android.content.Context
import android.os.Build
import cloud.proxi.sdkv3.utils.StringUtils
import java.net.URLEncoder
import java.nio.charset.StandardCharsets

/**
 * [UserAgent] Implementation of a User Agent for Android.
 * It contains info about device, app version, SDK version
 * It's is sent as a header with api calls
 *
 */

actual class UserAgent(private val context: Context) {
    companion object {
        const val UNKNOWN: String = "<unknown>"
    }

    actual val get: String
        get() {
            val packageName = encode(context.packageName)
            val agent = String.format(
                "%s/%s/%s (Android %s %s) (%s:%s:%s) Proxi.cloud SDK %s",
                appLabel, packageName, appVersion,
                Build.VERSION.RELEASE, Build.SUPPORTED_ABIS[0],
                Build.MANUFACTURER, Build.MODEL, Build.PRODUCT,
                "3.0.0"
                //TODO: Replace with real variable
            )
            return StringUtils.normalizeToAscii(agent)
        }

    private val appLabel: String
        get() = try {
            val packageManager = context.packageManager
            val applicationInfo = context.applicationInfo
            val label = packageManager.getApplicationLabel(applicationInfo).toString()
            encode(label)
        } catch (e: Exception) {
            UNKNOWN
        }

    private val appVersion: String
        get() = try {
            val packageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            encode(packageInfo.versionName) + "/" + versionCode
        } catch (e: Exception) {
            UNKNOWN
        }

    private val versionCode: Long
        get() {
            val packageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                packageInfo.longVersionCode
            } else {
                packageInfo.versionCode.toLong()
            }
        }

    private fun encode(value: String) =
        try {
            URLEncoder.encode(value, StandardCharsets.UTF_8.toString())
        } catch (e: Exception) {
            UNKNOWN
        }
}