package cloud.proxi.sdkv3

import cloud.proxi.sdkv3.api.Api
import cloud.proxi.sdkv3.api.UserAgent
import cloud.proxi.sdkv3.api.services.CreateHeaders
import cloud.proxi.sdkv3.database.*
import cloud.proxi.sdkv3.geofence.AnalyticsRepository
import cloud.proxi.sdkv3.geofence.GeofenceRepository
import cloud.proxi.sdkv3.geofence.model.PCGeofence
import cloud.proxi.sdkv3.logger.DummyLogger
import cloud.proxi.sdkv3.logger.Logger
import cloud.proxi.sdkv3.model.*
import cloud.proxi.sdkv3.model.Action
import com.russhwolf.settings.Settings


/**
 *
 * Entry point of a shared SDK module. Contains API that is consumed by platform specific implementations
 *
 */
object PCShared {

    lateinit var api: Api

    lateinit var geofenceRepository: GeofenceRepository

    lateinit var analyticsRepository: AnalyticsRepository

    lateinit var settings: Settings

    lateinit var idStorage: IdStorage

    lateinit var database: ProxiCloudDatabase

    lateinit var logger: Logger;

    fun init(
        driverFactory: DriverFactory,
        settingsManager: SettingsManager,
        userAgent: UserAgent,
        logger: Logger = DummyLogger()
    ) {
        settings = settingsManager.createSettings()
        idStorage = IdStorage(settings)
        api = Api(idStorage, CreateHeaders(userAgent), settings)
        database = createDatabase(driverFactory)
        analyticsRepository = AnalyticsRepository(database, api)
        geofenceRepository = GeofenceRepository(database, api)
        this.logger = logger
    }

    /**
     *
     * Retrieves geofences closest to user's location
     *
     * @param latitude app user's location latitude
     * @param longitude app user's location longitude
     * @param count how many nearest geofences to return, default is 20
     * @return Nearest geofences to the app user's location
     *
     */
    suspend fun getNearestGeofences(
        latitude: Double,
        longitude: Double,
        count: Int = 20
    ): List<PCGeofence> {
        return geofenceRepository.getNearestGeofences(latitude, longitude, count)
    }

    /**
     *
     * Retrieves list of actions associated with a given trigger
     *
     * @param triggerId identifier of a trigger, ex. PCGeofence identifier
     * @return List of actions triggered by given trigger (ex. geofence)
     */
    suspend fun getActionsForTrigger(triggerId: String): List<Action> {
        return geofenceRepository.actionsForTrigger(triggerId) ?: emptyList()
    }

    /**
     * Reports that event (like geofence entry) has occurred
     *
     * @param event reported event that contains an action with additional data,
     * like location and timestamp
     */
    suspend fun reportEvent(event: Event) {
        analyticsRepository.saveEvent(event)
    }

    suspend fun sendCustomEvent(customEvent: CustomEvent) {
        analyticsRepository.saveCustomEvent(customEvent)
    }

    /**
     * Sets API Key used to identify an app
     */
    fun setApiKey(apiKey: String) {
        idStorage.setApiKey(apiKey)
    }

    /**
     * Sets advertising identifier
     */
    fun setAdvertisingIdentifier(aid: String) {
        idStorage.saveAdvertisingIdentifier(aid)
    }

    suspend fun saveWifiNetworks(wiFiNetworks: List<WiFiNetwork>) {
        analyticsRepository.saveWiFiNetworks(wiFiNetworks)
    }

    /**
     * Send to the backend all events and logs that are stored in the database
     */
    suspend fun syncWithApi() {
        analyticsRepository.sendEvents()
    }

    fun setEnabled(enabled: Boolean) {
        idStorage.setEnabled(enabled)
    }
}