package cloud.proxi.sdkv3.geofence

import cloud.proxi.sdkv3.api.Api
import cloud.proxi.sdkv3.api.model.ActionApiModel
import cloud.proxi.sdkv3.api.model.AnalyticsRequestApiModel
import cloud.proxi.sdkv3.api.model.LocationDetailsApiModel
import cloud.proxi.sdkv3.api.model.ScanApiModel
import cloud.proxi.sdkv3.database.ProxiCloudDatabase
import cloud.proxi.sdkv3.database.runGettingLastId
import cloud.proxi.sdkv3.model.CustomEvent
import cloud.proxi.sdkv3.model.Event
import cloud.proxi.sdkv3.model.PCLocation
import cloud.proxi.sdkv3.model.WiFiNetwork
import cloud.proxi.sdkv3.utils.LocationUtils
import cloud.proxi.sdkv3.utils.PCSchedulers
import kotlinx.coroutines.withContext
import kotlinx.datetime.Clock

class AnalyticsRepository(val database: ProxiCloudDatabase, val api: Api) {

    suspend fun saveEvent(event: Event) = withContext(PCSchedulers.IO) {
        database.analyticsActionQueries.insert(
            event.action.uuid,
            event.time,
            event.transition.type.toLong(),
            event.triggerId,
            if (event.location != null) LocationUtils.locationToGeohash(event.location) else null,
            event.action.instanceUuid,
            event.lifecycle.toLong(),
            event.locationPermission.toLong(),
            event.location?.let { insertLocation(it, event.time) }
        )
    }

    suspend fun sendEvents() = withContext(PCSchedulers.IO) {
        val actions = database.analyticsActionQueries.selectAll().executeAsList()
        val triggers = database.triggerLogQueriesQueries.selectAll().executeAsList()
        val wifiNetworks = database.wifiNetworksQueries.selectAll().executeAsList()
        if (actions.isEmpty() && triggers.isEmpty() && wifiNetworks.isEmpty()) {
            return@withContext
        }
        val networks = wifiNetworks.map {
            val locationDetails = it.locationDetailsId?.let { id ->
                database.locationDetailsQueries.getDetailsForId(
                    id
                ).executeAsOne()
            }
            ScanApiModel(
                "ssid_" + it.pid,
                it.dt,
                it.geohash,
                it.mac,
                it.rssi,
                it.frequency,
                null,
                locationDetails?.let { it1 ->
                    LocationDetailsApiModel(
                        it1.latitude,
                        locationDetails.longitude,
                        locationDetails.timeDiff,
                        locationDetails.accuracy,
                        locationDetails.altitude,
                        locationDetails.verticalAccuracy,
                        locationDetails.speed,
                        locationDetails.speedAccuracy,
                        locationDetails.bearing,
                        locationDetails.bearingAccuracy
                    )
                },
                it.lifecycle.toInt(),
                it.locationPermission.toInt()
            )
        }
        val apiActions = actions.map {
            val locationDetails = it.locationDetailsId?.let { id ->
                database.locationDetailsQueries.getDetailsForId(
                    id
                ).executeAsOne()
            }
            ActionApiModel(
                it.actionId,
                it.dt,
                it.trigger.toInt(),
                it.pid.substring(0,14),
                it.geohash,
                it.actionInstanceUuid,
                locationDetails?.let { it1 ->
                    LocationDetailsApiModel(
                        it1.latitude,
                        locationDetails.longitude,
                        locationDetails.timeDiff,
                        locationDetails.accuracy,
                        locationDetails.altitude,
                        locationDetails.verticalAccuracy,
                        locationDetails.speed,
                        locationDetails.speedAccuracy,
                        locationDetails.bearing,
                        locationDetails.bearingAccuracy
                    )
                },
                it.lifecycle.toInt(),
                it.locationPermission.toInt()
            )
        }

        val scans = triggers.map {
            val locationDetails = it.locationDetailsId?.let { id ->
                database.locationDetailsQueries.getDetailsForId(
                    id
                ).executeAsOne()
            }
            ScanApiModel(
                it.pid,
                it.dt,
                it.geohash,
                null,
                null,
                null,
                it.pairingId!!,
                locationDetails?.let { it1 ->
                    LocationDetailsApiModel(
                        it1.latitude,
                        locationDetails.longitude,
                        locationDetails.timeDiff,
                        locationDetails.accuracy,
                        locationDetails.altitude,
                        locationDetails.verticalAccuracy,
                        locationDetails.speed,
                        locationDetails.speedAccuracy,
                        locationDetails.bearing,
                        locationDetails.bearingAccuracy
                    )
                },
                it.lifecycle.toInt(),
                it.locationPermission.toInt()
            )
        }

        try {
            val response = api.sendAnalytics(
                AnalyticsRequestApiModel(
                    scans + networks,
                    apiActions,
                    emptyList()
                )
            )
            if (response.status.value in 200..300) {
                for (action in actions) {
                    database.analyticsActionQueries.delete(action.actionInstanceUuid)
                }
                for (scan in scans) {
                    database.triggerLogQueriesQueries.delete()
                    database.wifiNetworksQueries.delete()
                }
            }
        } catch (e: Exception) {
            println("DATAA_TEST" + "FINISHED " + e)
        }
    }

    suspend fun saveCustomEvent(customEvent: CustomEvent) =
        withContext(PCSchedulers.IO) {
            val time = Clock.System.now().toEpochMilliseconds()
            database.triggerLogQueriesQueries.insert(
                time,
                customEvent.key,
                customEvent.location?.let { LocationUtils.locationToGeohash(it) },
                customEvent.value,
                customEvent.lifecycle.toLong(),
                customEvent.locationPermission.toLong(),
                customEvent.location?.let { insertLocation(it, time) }
            )
        }

    suspend fun saveWiFiNetworks(wiFiNetworks: List<WiFiNetwork>) = withContext(PCSchedulers.IO) {
        wiFiNetworks.forEach { wiFiNetwork ->
            val locationId = insertLocation(wiFiNetwork.location, wiFiNetwork.dt)
            database.wifiNetworksQueries.transaction {
                val dt = database.wifiNetworksQueries.checkLastInserted(wiFiNetwork.bssid)
                    .executeAsOneOrNull()
                if (dt == null || (wiFiNetwork.dt - dt) > 60_000) {
                    database.wifiNetworksQueries.insert(
                        wiFiNetwork.dt,
                        wiFiNetwork.ssid.trim { it == 0.toChar() },
                        LocationUtils.locationToGeohash(wiFiNetwork.location),
                        locationId,
                        wiFiNetwork.rssi,
                        wiFiNetwork.frequency,
                        wiFiNetwork.bssid,
                        wiFiNetwork.lifecycle.toLong(),
                        wiFiNetwork.locationPermission.toLong()
                    )
                }
            }
        }
    }

    private fun insertLocation(pcLocation: PCLocation, eventTime: Long) =
        database.runGettingLastId {
            database.locationDetailsQueries.insert(
                pcLocation.latitude,
                pcLocation.longitude,
                pcLocation.accuracy,
                (eventTime - pcLocation.timestamp) / 1000,
                pcLocation.altitude,
                pcLocation.verticalAccuracy,
                pcLocation.speed,
                pcLocation.speedAccuracy,
                pcLocation.bearing,
                pcLocation.bearingAccuracy
            )
        }
}