package cloud.proxi.sdkv3.geofence

import cloud.proxi.sdkv3.api.model.ApiResult
import cloud.proxi.sdkv3.api.model.LayoutApiModel
import cloud.proxi.sdkv3.api.services.ApiService
import cloud.proxi.sdkv3.database.ProxiCloudDatabase
import cloud.proxi.sdkv3.database.runGettingLastId
import cloud.proxi.sdkv3.model.Action
import cloud.proxi.sdkv3.model.ActionType
import cloud.proxi.sdkv3.model.Content
import cloud.proxi.sdkv3.model.TransitionType
import cloud.proxi.sdkv3.utils.PCSchedulers
import kotlinx.coroutines.withContext
import kotlinx.datetime.LocalDateTime
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

class Repository(val apiService: ApiService, val database: ProxiCloudDatabase) {

    suspend fun getActionsForTrigger(triggerId: String) = withContext(PCSchedulers.IO) {
        saveLayout()
        getActionsForTriggerFromDatabase(triggerId)
    }

    suspend fun saveLayout() = withContext(PCSchedulers.IO) {
        when (val response = apiService.retrieveLayout()) {
            is ApiResult.HttpError -> {

            }
            is ApiResult.IOError -> {

            }
            is ApiResult.NotModified -> {

            }
            is ApiResult.Success -> handleSuccess(response.data)
        }
    }

    private suspend fun handleSuccess(layoutApiModel: LayoutApiModel) =
        withContext(PCSchedulers.IO) {
            layoutApiModel.actions.forEach { action ->
                database.transaction {
                    val actionId = database.runGettingLastId {
                        database.layoutActionModelQueries.insert(
                            action.eid,
                            action.trigger,
                            action.type.toLong(),
                            action.suppressionTime,
                            action.delay?.toLong() ?: 0,
                            Json.encodeToString(action.content),
                            action.deliveryAt?.toString()
                        )
                    }
                    action.beacons.forEach { trigger ->
                        database.geofenceQueries.also {
                            val triggerWithDelay = if (isGeofence(trigger)) {
                                trigger + (action.delay?.toLong() ?: 0)
                            } else {
                                trigger
                            }
                            it.insert(triggerWithDelay, actionId.toString())
                            it.update(actionId.toString(), triggerWithDelay)
                        }
                    }
                }
            }
        }

    private suspend fun getActionsForTriggerFromDatabase(triggerId: String) =
        withContext(PCSchedulers.IO) {
            val actions = ArrayList<Action>()
            val actionIds =
                database.geofenceQueries.findActionsForTrigger(triggerId).executeAsOne()
            actionIds.split(',').forEach { actionId ->
                val action =
                    database.layoutActionModelQueries.findTriggersForId(actionId.toLong())
                        .executeAsOneOrNull()
                if (action != null)
                    actions.add(
                        Action(
                            ActionType.fromInt(action.type.toInt()),
                            action.delay?.toInt(),
                            action.deliveryAt?.let { LocalDateTime.parse(it) },
                            action.eid,
                            TransitionType.fromInt(action.trigger),
                            null,
                            Content.fromCampaignData(Json.decodeFromString(action.content))
                        )
                    )
            }
            actions
        }

    private fun isGeofence(trigger: String): Boolean {
        return trigger.length in 14..41
                && !trigger.startsWith("aid_")
                && !trigger.startsWith("ssid_")
                && !trigger.startsWith("fcm_")
    }
}