package cloud.proxi.sdkv3.geofence.model

import cloud.proxi.sdkv3.utils.geohash.GeoHash

/**
 * Representation of a Geofence
 * contains all information required to observe one and get entry/dwell/exit updates
 */
data class PCGeofence(val identifier: String) {

    /**
     * Latitude of the Geofence's center
     */
    val latitude: Double

    /**
     * Longitude of the Geofence's center
     */
    val longitude: Double

    /**
     * Geofence's radius in meters
     */
    val radius: Double

    /**
     * dwell time in seconds
     * Geofence entry should be registered only if user stops inside a geofence for a period of time equal to dwell
     *
     * default value: 0, means that geofence entry should be registered as soon as possible
     */
    var dwell: Long = 0
        private set

    init {
        radius = identifier.substring(8, 14).toDouble()
        dwell = setDwell(identifier)
        try {
            val hash: GeoHash = GeoHash.fromGeohashString(identifier.substring(0, 8))
            latitude = hash.getPoint()!!.latitude
            longitude = hash.getPoint()!!.longitude
        } catch (e: Exception) {
            throw IllegalArgumentException("Invalid geofence geohash: $identifier")
        }
    }

    private fun setDwell(identifier: String): Long {
        val dwell = identifier.substring(14)
        return if (dwell.isBlank()) {
            0L
        } else {
            dwell.toLong()
        }
    }

    override fun toString(): String {
        return "[identifier=$identifier, latitude=$latitude, longitude=$longitude, radius=$radius, dwellSeconds=$dwell]"
    }
}