package cloud.proxi.sdkv3.utils

import cloud.proxi.sdkv3.model.PCLocation
import cloud.proxi.sdkv3.utils.geohash.GeoHash
import kotlinx.serialization.builtins.LongAsStringSerializer
import kotlin.math.*

object LocationUtils {
    private const val DEGREES_TO_RADIANS = 0.017453292519943295

    private const val EARTH_RADIUS_IN_METERS = 6371007.177356707

    /**
     * [calculateDistance] Calculates distance between two points on earth, in meters
     *
     */
    fun calculateDistance(
        lat1: Double,
        lon1: Double,
        lat2: Double,
        lon2: Double
    ): Double {
        LongAsStringSerializer
        val latDiff = toRadians(abs(lat2 - lat1))
        val lngDiff = toRadians(abs(lon2 - lon1))
        val a = sin(latDiff / 2) * sin(latDiff / 2) +
                cos(toRadians(lat1)) * cos(toRadians(lat2)) *
                sin(lngDiff / 2) * sin(lngDiff / 2)
        val c = 2 * atan2(sqrt(a), sqrt(1 - a))
        return EARTH_RADIUS_IN_METERS * c
    }

    private fun toRadians(value: Double): Double {
        return value * DEGREES_TO_RADIANS
    }

    /**
     * [locationToGeohash] Converts location to Geohash string
     *
     */
    fun locationToGeohash(location: PCLocation): String {
        return GeoHash.geoHashStringWithCharacterPrecision(location.latitude, location.longitude, 8)
    }
}