package cloud.proxi.sdkv3.utils.geohash

import kotlin.math.ceil
import kotlin.math.min

/*
* Copyright 2010, Silvio Heuberger @ IFS www.ifs.hsr.ch
*
* This code is release under the Apache License 2.0.
* You should have received a copy of the license
* in the LICENSE file. If you have not, see
* http://www.apache.org/licenses/LICENSE-2.0
*/
class GeoHash : Comparable<GeoHash?> {
    companion object {
        private const val MAX_BIT_PRECISION = 64
        private const val MAX_CHARACTER_PRECISION = 12
        private val BITS = intArrayOf(16, 8, 4, 2, 1)
        private const val BASE32_BITS = 5
        const val FIRST_BIT_FLAGGED = (-0x800000000000000L)
        private val base32 = charArrayOf(
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'b', 'c', 'd', 'e', 'f',
            'g', 'h', 'j', 'k', 'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'
        )
        private val decodeMap: MutableMap<Char, Int> = HashMap()


        fun geoHashStringWithCharacterPrecision(
            latitude: Double,
            longitude: Double,
            numberOfCharacters: Int
        ): String {
            val hash: GeoHash =
                withCharacterPrecision(latitude, longitude, numberOfCharacters)
            return hash.toBase32()
        }

        fun withCharacterPrecision(
            latitude: Double,
            longitude: Double,
            numberOfCharacters: Int
        ): GeoHash {
            require(numberOfCharacters <= MAX_CHARACTER_PRECISION) { "A geohash can only be $MAX_CHARACTER_PRECISION character long." }
            val desiredPrecision = if (numberOfCharacters * 5 <= 60) numberOfCharacters * 5 else 60
            return GeoHash(latitude, longitude, desiredPrecision)
        }

        /**
         * build a new [GeoHash] from a base32-encoded [String].<br></br>
         * This will also set up the hashes bounding box and other values, so it can
         * also be used with functions like within().
         */
        fun fromGeohashString(geohash: String): GeoHash {
            val latitudeRange = doubleArrayOf(-90.0, 90.0)
            val longitudeRange = doubleArrayOf(-180.0, 180.0)
            var isEvenBit = true
            val hash = GeoHash()
            for (i in 0 until geohash.length) {
                val cd = decodeMap[geohash[i]]!!
                for (j in 0 until BASE32_BITS) {
                    val mask = BITS[j]
                    if (isEvenBit) {
                        divideRangeDecode(hash, longitudeRange, cd and mask != 0)
                    } else {
                        divideRangeDecode(hash, latitudeRange, cd and mask != 0)
                    }
                    isEvenBit = !isEvenBit
                }
            }
            val latitude = (latitudeRange[0] + latitudeRange[1]) / 2
            val longitude = (longitudeRange[0] + longitudeRange[1]) / 2
            hash.point = WGS84Point(latitude, longitude)
            hash.bits = hash.bits shl MAX_BIT_PRECISION - hash.significantBits
            return hash
        }

        private fun divideRangeDecode(hash: GeoHash, range: DoubleArray, b: Boolean) {
            val mid = (range[0] + range[1]) / 2
            if (b) {
                hash.addOnBitToEnd()
                range[0] = mid
            } else {
                hash.addOffBitToEnd()
                range[1] = mid
            }
        }

        init {
            val sz = base32.size
            for (i in 0 until sz) {
                decodeMap[base32[i]] = i
            }
        }
    }

    protected var bits: Long = 0
    private var point: WGS84Point? = null
    protected var significantBits: Byte = 0

    protected constructor() {}

    private constructor(latitude: Double, longitude: Double, desiredPrecision: Int) {
        var desiredPrecision = desiredPrecision
        point = WGS84Point(latitude, longitude)
        desiredPrecision = min(desiredPrecision, MAX_BIT_PRECISION)
        var isEvenBit = true
        val latitudeRange = doubleArrayOf(-90.0, 90.0)
        val longitudeRange = doubleArrayOf(-180.0, 180.0)
        while (significantBits < desiredPrecision) {
            if (isEvenBit) {
                divideRangeEncode(longitude, longitudeRange)
            } else {
                divideRangeEncode(latitude, latitudeRange)
            }
            isEvenBit = !isEvenBit
        }
        bits = bits shl MAX_BIT_PRECISION - desiredPrecision
    }


    /**
     * Returns the number of characters that represent this hash.
     *
     * @throws IllegalStateException
     * when the hash cannot be encoded in base32, i.e. when the
     * precision is not a multiple of 5.
     */
    val characterPrecision: Int
        get() {
            check(significantBits % 5 == 0) { "precision of GeoHash is not divisble by 5: $this" }
            return significantBits / 5
        }

    private fun divideRangeEncode(value: Double, range: DoubleArray) {
        val mid = (range[0] + range[1]) / 2
        if (value >= mid) {
            addOnBitToEnd()
            range[0] = mid
        } else {
            addOffBitToEnd()
            range[1] = mid
        }
    }


    /**
     * how many significant bits are there in this [GeoHash]?
     */
    fun significantBits(): Int {
        return significantBits.toInt()
    }

    fun longValue(): Long {
        return bits
    }

    /**
     * get the base32 string for this [GeoHash].<br></br>
     * this method only makes sense, if this hash has a multiple of 5
     * significant bits.
     *
     * @throws IllegalStateException
     * when the number of significant bits is not a multiple of 5.
     */
    fun toBase32(): String {
        check(significantBits % 5 == 0) { "Cannot convert a geohash to base32 if the precision is not a multiple of 5." }
        val buf = StringBuilder()
        val firstFiveBitsMask = -0x800000000000000L
        var bitsCopy = bits
        val partialChunks = ceil(significantBits.toDouble() / 5).toInt()
        for (i in 0 until partialChunks) {
            val pointer = (bitsCopy and firstFiveBitsMask ushr 59).toInt()
            buf.append(base32[pointer])
            bitsCopy = bitsCopy shl 5
        }
        return buf.toString()
    }

    /**
     * returns true iff this is within the given geohash bounding box.
     */
    fun within(boundingBox: GeoHash): Boolean {
        return bits and boundingBox.mask() == boundingBox.bits
    }


    /**
     * returns the [WGS84Point] that was originally used to set up this.<br></br>
     * If it was built from a base32-[String], this is the center point of
     * the bounding box.
     */
    fun getPoint(): WGS84Point? {
        return point
    }


    fun enclosesCircleAroundPoint(point: WGS84Point?, radius: Double): Boolean {
        return false
    }

    protected val rightAlignedLatitudeBits: LongArray
        protected get() {
            val copyOfBits = bits shl 1
            val value = extractEverySecondBit(copyOfBits, numberOfLatLonBits[0])
            return longArrayOf(value, numberOfLatLonBits[0].toLong())
        }
    protected val rightAlignedLongitudeBits: LongArray
        protected get() {
            val copyOfBits = bits
            val value = extractEverySecondBit(copyOfBits, numberOfLatLonBits[1])
            return longArrayOf(value, numberOfLatLonBits[1].toLong())
        }

    private fun extractEverySecondBit(copyOfBits: Long, numberOfBits: Int): Long {
        var copyOfBits = copyOfBits
        var value: Long = 0
        for (i in 0 until numberOfBits) {
            if (copyOfBits and FIRST_BIT_FLAGGED == FIRST_BIT_FLAGGED) {
                value = value or 0x1
            }
            value = value shl 1
            copyOfBits = copyOfBits shl 2
        }
        value = value ushr 1
        return value
    }

    protected val numberOfLatLonBits: IntArray
        protected get() = if (significantBits % 2 == 0) {
            intArrayOf(significantBits / 2, significantBits / 2)
        } else {
            intArrayOf(significantBits / 2, significantBits / 2 + 1)
        }

    protected fun addOnBitToEnd() {
        significantBits++
        bits = bits shl 1
        bits = bits or 0x1
    }

    protected fun addOffBitToEnd() {
        significantBits++
        bits = bits shl 1
    }

    override fun toString(): String {
        return if (significantBits % 5 == 0) {
            "${bits.toString(2)} -> ${toBase32()}"
        } else {
            "${bits.toString(2)} -> $significantBits"
        }
    }

    fun toBinaryString(): String {
        val bui = StringBuilder()
        var bitsCopy = bits
        for (i in 0 until significantBits) {
            if (bitsCopy and FIRST_BIT_FLAGGED == FIRST_BIT_FLAGGED) {
                bui.append('1')
            } else {
                bui.append('0')
            }
            bitsCopy = bitsCopy shl 1
        }
        return bui.toString()
    }

    override fun equals(obj: Any?): Boolean {
        if (obj === this) {
            return true
        }
        if (obj is GeoHash) {
            val other = obj
            if (other.significantBits == significantBits && other.bits == bits) {
                return true
            }
        }
        return false
    }

    override fun hashCode(): Int {
        var f = 17
        f = 31 * f + (bits xor (bits ushr 32)).toInt()
        f = 31 * f + significantBits
        return f
    }

    /**
     * return a long mask for this hashes significant bits.
     */
    private fun mask(): Long {
        return if (significantBits.toInt() == 0) {
            0
        } else {
            var value = FIRST_BIT_FLAGGED
            value = value shr significantBits - 1
            value
        }
    }

    private fun maskLastNBits(value: Long, n: Long): Long {
        var mask = -0x1L
        mask = mask ushr (MAX_BIT_PRECISION - n).toInt()
        return value and mask
    }

    override operator fun compareTo(o: GeoHash?): Int {
        if (o == null) {
            throw NullPointerException()
        }
        val bitsCmp =
            (bits xor FIRST_BIT_FLAGGED).compareTo(o.bits xor FIRST_BIT_FLAGGED)
        return if (bitsCmp != 0) {
            bitsCmp
        } else {
            significantBits.toInt().compareTo(o.significantBits.toInt())
        }
    }
}