package com.jfplugin.xsql.core;

import java.util.HashMap;
import java.util.Map;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.jfplugin.xsql.exception.NodeNotAllowedException;
import com.jfplugin.xsql.exception.StatementException;
import com.jfplugin.xsql.statement.CommentStatement;
import com.jfplugin.xsql.statement.ForeachStatement;
import com.jfplugin.xsql.statement.IsEmptyStatement;
import com.jfplugin.xsql.statement.IsNotEmptyStatement;
import com.jfplugin.xsql.statement.SqlStatement;
import com.jfplugin.xsql.statement.Statement;
import com.jfplugin.xsql.statement.TextStatement;

/**
 * 语法构建
 * @author farmer
 *
 */
public class StatementBuilder {

	/**
	 * 构建SqlStatement
	 * @param sqlnode
	 * 	SQL节点
	 * @return
	 * 
	 */
	public static SqlStatement builder(Node sqlnode){
		SqlStatement sqlNodeStatement = new SqlStatement(sqlnode);
		buildChild(sqlNodeStatement,sqlnode.getChildNodes());
		return sqlNodeStatement;
	}
	
	
	/**
	 * 构建语法树
	 * @param statement
	 * 	statement
	 * @param nodeList
	 * 	子节点
	 */
	private static void buildChild(Statement statement, NodeList nodeList){
		for (int i = 0; i < nodeList.getLength(); i++) {
			Node node = nodeList.item(i);
			Statement nodeStatement = createStatement(node);
			if(node.getChildNodes().getLength() > 0){
				buildChild(nodeStatement, node.getChildNodes());
			}
			statement.addChild(nodeStatement);
		}
	}
	
	/**
	 * 将XML的NODE节点，转成语法节点
	 * @param node
	 * 	节点
	 * @return
	 * 	Statement
	 */
	private static Statement createStatement(Node node){
		String tag = node.getNodeName().toLowerCase();
		Class<? extends Statement> clazz = registerMap.get(tag);
		if(clazz != null){
			try {
				return clazz.getConstructor(new Class[]{Node.class}).newInstance(node);
			} catch (Exception e) {
				throw new StatementException("创建"+clazz+"异常!", e);
			}
		}
		throw new NodeNotAllowedException(tag+"标签无法识别");
	}
	
	private static Map<String, Class<? extends Statement> > registerMap = new HashMap<String, Class<? extends Statement>>();
	
	static {
		registerMap.put("foreach", ForeachStatement.class);
		registerMap.put("isempty", IsEmptyStatement.class);
		registerMap.put("isnotempty", IsNotEmptyStatement.class);
		registerMap.put("#text", TextStatement.class);
		registerMap.put("#comment", CommentStatement.class);
	}
	
	/**
	 * 注册标签处理方式
	 * @param tag
	 * 	标签名称
	 * @param clazz
	 * 	处理的Statement
	 */
	public static void register(String tag , Class<? extends Statement> clazz){
		registerMap.put(tag, clazz);	
	}

	/**
	 * 注册
	 * @param tag
	 * 	标签名称
	 * @param clazz
	 * 	处理的Statement
	 */
	@SuppressWarnings("unchecked")
	public static void register(String tag, String clazz) {
		try {
			registerMap.put(tag, (Class<? extends Statement>)Class.forName(clazz));
		} catch (Exception e) {
			throw new RuntimeException(clazz+"注册失败!", e);
		}
	}
	
	
	
}
