package fi.evolver.script;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Environment {
    private static final Pattern REGEX_EXPORT = Pattern.compile("^\\s*export +(?<name>\\w+)=(?<value>\"[^\"]*\"|'[^']*'|\\S*)\\s*(?:#\\s*(?<comment>\\S.*?)?\\s*)?$");

    /**
     * Sets a personal environment variable into a hidden variable file.
     * The variable file will be sourced from the .bashrc file.
     * Any previous variable with the same name will be replaced.
     * 
     * @param name The name of the variable to be set.
     * @param value The new value for the variable.
     */
    public static void setPersonalVariable(String name, String value) {
        setPersonalVariable(name, value, null);
    }


    /**
     * Sets a personal environment variable into a hidden variable file.
     * The variable file will be sourced from the .bashrc file.
     * Any previous variable with the same name will be replaced.
     * 
     * @param name The name of the variable to be set.
     * @param value The new value for the variable.
     * @param comment A comment for the variable
     */
    public static void setPersonalVariable(String name, String value, String comment) {
        try (Step step = Step.start("Set personal variable %s".formatted(name))) {
            try {
                Path variableFile = Shell.HOME.resolve(".personal-variables");
                FileUtils.writeShellBlock(Shell.BASHRC, "Personal-ENV-variables", "source %s".formatted(variableFile));

                Map<String, Variable> variables = readVariables(variableFile);
                variables.put(name, new Variable(name, value, comment));
                writeVariables(variableFile, variables.values());
            }
            catch (IOException e) {
                step.fail(e);
            }
        }
    }


    public static Optional<String> getPersonalVariable(String name) {
        try (Step step = Step.start("Read personal variable %s".formatted(name))) {
            try {
                Path variableFile = Shell.HOME.resolve(".personal-variables");
                return Optional.ofNullable(readVariables(variableFile).get(name)).map(Variable::value);
            }
            catch (IOException e) {
                step.fail(e);
                throw new IllegalStateException("Should not reach this");
            }
        }
    }


    static Map<String, Variable> readVariables(Path file) throws IOException {
        Map<String, Variable> results = new TreeMap<>();
        List<String> lines = Files.exists(file) ? Files.readAllLines(file) : List.of();
        lines.stream()
                .map(Variable::parseExportLine)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .forEach(v -> results.put(v.name(), v));
        return results;
    }


    static void writeVariables(Path file, Collection<Variable> variables) throws IOException {
        StringBuilder builder = new StringBuilder();
        builder.append("# Generated programmatically: avoid invasive changes\n\n");
        for (Variable variable: variables) {
            builder.append(variable.toExportLine());
            builder.append("\n");
        }
        Files.writeString(file, builder, StandardOpenOption.WRITE, StandardOpenOption.CREATE);
    }


    static String escapeValue(String value) {
        return "'" + value.replace("'", "'\\''") + "'";
    }


    static String unescapeValue(String value) {
        String result = value;
        if (result.startsWith("'")) {
            result = result.substring(1, result.length() - 1).replace("'\\''", "'");
        }
        else {
            if (result.startsWith("\""))
                result = result.substring(1, result.length() - 1);
            result = result.replaceAll("\\\\([\\\"$])", "$1");
        }
        return result;
    }



    record Variable(
            String name,
            String value,
            String comment) {

        public static Optional<Variable> parseExportLine(String line) {
            Matcher matcher = REGEX_EXPORT.matcher(line);
            if (!matcher.find())
                return Optional.empty();

            String name = matcher.group("name");
            String value = unescapeValue(matcher.group("value"));
            String comment = matcher.group("comment");

            return Optional.of(new Variable(name, value, comment));
        }

        public String toExportLine() {
            String result = "export %s=%s".formatted(name, escapeValue(value));
            if (comment != null)
                result = "%-39s # %s".formatted(result, comment);
            return result;
        }
    }

}
