package fi.evolver.script.app;

import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;

import fi.evolver.script.Download;
import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;
import fi.evolver.script.app.Apt.Source;

public class Firefox {
	private static final Source APT_SOURCE = Source.builder("mozilla", URI.create("https://packages.mozilla.org/apt"), "mozilla", "main")
			.setSignedBy("packages.mozilla.org.gpg", URI.create("https://packages.mozilla.org/apt/repo-signing-key.gpg"))
			.build();

	public static final Extension EXTENSION_BITWARDEN = new Extension(
			"{446900e4-71c2-419f-a6a7-df9c091e268b}", // https://addons.mozilla.org/en-US/firefox/addon/bitwarden-password-manager/
			URI.create("https://addons.mozilla.org/firefox/downloads/file/4562769/bitwarden_password_manager-2025.8.1.xpi"));

	public static final Extension EXTENSION_MULTI_ACCOUNT_CONTAINERS = new Extension(
			"@testpilot-containers", // https://addons.mozilla.org/en-US/firefox/addon/multi-account-containers/
			URI.create("https://addons.mozilla.org/firefox/downloads/file/4355970/multi_account_containers-8.2.0.xpi"));

	public static final Extension EXTENSION_GRANTED_CONTAINERS = new Extension(
			"{b5e0e8de-ebfe-4306-9528-bcc18241a490}", // https://addons.mozilla.org/en-US/firefox/addon/granted/
			URI.create("https://addons.mozilla.org/firefox/downloads/file/4086622/granted-1.1.2.xpi"));

	public static final Extension EXTENSION_UBLOCK_ORIGIN = new Extension(
			"uBlock0@raymondhill.net", // https://addons.mozilla.org/en-US/firefox/addon/ublock-origin/
			URI.create("https://addons.mozilla.org/firefox/downloads/file/4531307/ublock_origin-1.65.0.xpi"));


	public static void installOfficial() {
		try (Step step = Step.start("Firefox: Install")) {
			Apt.addSource(APT_SOURCE);
			Apt.update();
			Apt.install("firefox");
		}
	}

	/**
	 * Downloads and a Firefox extension and adds it to the system autoinstall directory.
	 * This will install the extension automatically to new profiles.
	 *
	 * @param firefoxInstallDir the directory of the system Firefox installation
	 */
	public static void addDefaultExtensions(Path firefoxInstallDir, List<Extension> extensions) {
		try (Step step = Step.start("Firefox: install extensions")) {
			Path externalExtensionsDir = firefoxInstallDir.resolve("distribution/extensions");

			Shell.sudo("mkdir", "-p", externalExtensionsDir.toString());

			for (Extension extension : extensions) {
				Path downloadPath = Download.intoTempFile(extension.downloadUrl);
				Path targetPath = externalExtensionsDir.resolve(extension.id + ".xpi");
				if (Files.exists(targetPath)) {
					continue;
				}
				FileUtils.sudoCopy(downloadPath.toString(), targetPath.toString(), "root:root", "644");
			}
		}
	}

	/**
	 * @param id          the identifier of the extension (for example, "@testpilot-containers") This can be seen e.g.
	 *                    by inspecting the DOM for the Firefox about:extensions listing.
	 * @param downloadUrl the URL to download the extension .xpi file from
	 */
	public record Extension(String id, URI downloadUrl) {
	}


	/**
	 * Installs bookmarks from an existing file to the default Firefox profile.
	 * The source file should be in Firefox bookmark format (JSON or HTML).
	 *
	 * @param bookmarksSourcePath the path to the bookmarks file to install
	 * @throws IOException if there's an error copying the bookmarks file
	 */
	public static void installBookmarks(Path bookmarksSourcePath) throws IOException {
		try (Step step = Step.start("Firefox: Install Bookmarks")) {
			Path defaultProfileDir = findProfileDir("default-release").orElse(null);
			if (defaultProfileDir == null)
				initDefaultProfile();
			defaultProfileDir = findProfileDir("default-release").orElse(null);
			if (defaultProfileDir == null) {
				step.fail("Failed creating / finding the default profile");
				return;
			}

			String sourceFileName = bookmarksSourcePath.getFileName().toString();
			String targetFileName;
			if (sourceFileName.toLowerCase().endsWith(".json"))
				targetFileName = "bookmarks.json";
			else if (sourceFileName.toLowerCase().endsWith(".html"))
				targetFileName = "bookmarks.html";
			else
				targetFileName = "bookmarks.json";


			Path targetFile = defaultProfileDir.resolve(targetFileName);
			if (Files.exists(targetFile)) {
				step.skip("Bookmarks already exist");
				return;
			}

			// Remove old bookmarks if present, or the new ones won't be imported automatically.
			Path placesSqlite = defaultProfileDir.resolve("places.sqlite");
			if (Files.exists(placesSqlite))
				Files.delete(placesSqlite);

			Files.copy(bookmarksSourcePath, targetFile);
		}
	}


	private static Optional<Path> findProfileDir(String profileName) throws IOException {
		Path profilesDir = Shell.HOME.resolve(".mozilla/firefox");
		Files.createDirectories(profilesDir);
		String suffix = "." + profileName;
		return Files.list(profilesDir)
						.filter(Files::isDirectory)
						.filter(path -> path.getFileName().toString().endsWith(suffix))
						.findFirst();
	}


	private static void initDefaultProfile() {
		Shell.Command.user("firefox", "--headless", "--new-instance", "--no-remote")
				.failOnError(false)
				.runAsync();

		try {
			Thread.sleep(2000);
		}
		catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
		Shell.user("killall", "firefox-bin");
	}

}
