package io.github.andreyzebin.gitSql.config;

import io.github.zebin.javabash.sandbox.DirectoryTree;
import io.github.zebin.javabash.sandbox.PosixPath;

import java.io.IOException;
import java.io.Writer;
import java.util.*;
import java.util.stream.Stream;

public class ConfigTree {
    private final DirectoryTree dt;

    public ConfigTree(DirectoryTree delegate) {
        this.dt = delegate;
    }

    public Stream<PosixPath> getLeafs() {
        // conf.properties
        List<PosixPath> propLeafs = new LinkedList<>();

        dt.traverse(
                PosixPath.ofPosix(""),
                d -> {
                    boolean hasPropertyMarker = dt.list(d)
                            .anyMatch(f -> f.getEnd().equals("conf.properties") && !dt.isDir(f));
                    if (hasPropertyMarker) {
                        propLeafs.add(d);
                    }
                    return true;
                }
        );

        return propLeafs.stream();
    }


    /**
     * Get effective property list. Moving from root directory to given leaf, collect all unique property keys.
     * Value of property declared closer to leaf on a downstream path has priority
     *
     * @param leaf Directory where leaf property stored
     * @return Effective property list, consider all upstream property stores, starting from root directory
     */
    public Set<String> getPropertyKeys(PosixPath leaf) {
        return getProperties(leaf).keySet();
    }

    private Map<String, String> getProperties(PosixPath leaf) {
        Map<String, String> effProps = new HashMap<>();

        leaf.streamClimbing().forEach(pp -> {
            PosixPath currProp = pp.climb("conf.properties");

            if (dt.exists(currProp) && !dt.isDir(currProp)) {
                Properties prop = new Properties();
                try {
                    prop.load(dt.get(currProp));
                    prop.stringPropertyNames()
                            .forEach(pr -> effProps.put(pr, prop.getProperty(pr)));
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        });

        return effProps;
    }

    /**
     * Get effective property value. Moving from root directory to given leaf, collect all unique property keys.
     * Value of property declared closer to leaf on a downstream path has priority
     *
     * @param leaf Directory where leaf property stored
     * @return Effective property value
     */
    public String getProperty(PosixPath leaf, String key) {
        return getProperties(leaf).get(key);
    }

    /**
     * Get effective property value. Moving from root directory to given leaf, collect all unique property keys.
     * Value of property declared closer to leaf on a downstream path has priority
     *
     * @param leaf Directory where leaf property stored
     * @return Effective property value
     */
    public Optional<String> setProperty(PosixPath leaf, String key, String value) {
        PosixPath currProp = leaf.climb("conf.properties");
        boolean hasFile = dt.exists(currProp) && !dt.isDir(currProp);
        Optional<String> ret = Optional.empty();

        if (hasFile) {
            Properties prop = new Properties();
            try {
                prop.load(dt.get(currProp));
                if (prop.stringPropertyNames().contains(key)) {
                    ret = Optional.ofNullable(prop.getProperty(key));
                }
                prop.setProperty(key, value);
                try (Writer upd = dt.put(currProp)) {
                    prop.store(upd, "The file was updated via ConfigTree");
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else {
            try (Writer patch = dt.put(currProp)) {
                patch.write("The file was generated via ConfigTree");
                patch.write(System.lineSeparator());
                patch.write(String.format("%s=%s", key, value));
                patch.write(System.lineSeparator());
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }


        return ret;
    }
}
