package io.github.andreyzebin.gitSql.git;

import io.github.zebin.javabash.process.TextTerminal;
import io.github.zebin.javabash.sandbox.BashUtils;

import java.nio.file.Path;
import java.time.Instant;
import java.util.Optional;
import java.util.stream.Stream;

public abstract class AbstractVersionControl implements VersionControl {

    protected final TextTerminal term;
    private Optional<String> origin = null;

    protected AbstractVersionControl(TextTerminal term) {
        this.term = term;
    }

    /**
     * Fast-forward
     */
    @Override
    public VersionControl seek(String commit) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            GitBindings.checkout(commit, term);
            return this;
        } finally {
            term.eval("cd " + pwd);
        }
    }

    /**
     * Fast-forward
     */
    @Override
    public Stream<Commit> commits() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.commitsList(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Stream<? extends Change> changes(String hashFrom) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.filesChangedQuery(
                    GitBindings.ALL_BRANCHES,
                    GitBindings.sinceHash(hashFrom),
                    term
            );
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Instant timestamp() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.commitsList(term)
                    .findFirst()
                    .get()
                    .getTimestampInstant();
        } finally {
            term.eval("cd " + pwd);
        }
    }

    public Optional<String> origin() {
        String pwd = term.eval("pwd");
        try {
            if (origin == null) {
                term.eval("cd " + BashUtils.encode(getRoot()));
                origin = GitBindings.getOrigin(term);
            }
            return origin;
        } finally {
            term.eval("cd " + pwd);
        }
    }

    public Optional<String> branch() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.getBranch(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    abstract public Path getRoot();
}
