package io.github.andreyzebin.gitSql.git;

import io.github.andreyzebin.gitSql.FileSystem;
import io.github.andreyzebin.gitSql.FileSystemSubDirectory;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import io.github.andreyzebin.gitSql.FileSystemUtils;
import io.github.zebin.javabash.process.TextTerminal;
import io.github.zebin.javabash.sandbox.BashUtils;
import io.github.zebin.javabash.sandbox.PosixPath;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class VersionedFiles implements FileSystem {

    private final boolean isReadOnly;
    private final GitVersions source;
    private final TextTerminal term;
    private StringBuffer filesToAdd;

    public VersionedFiles(boolean isReadOnly, GitVersions source, TextTerminal term) {
        this.isReadOnly = isReadOnly;
        this.source = source;
        this.term = term;
        filesToAdd = new StringBuffer();
    }

    public GitVersions getSource() {
        return source;
    }

    @Override
    public Writer put(Path path) {
        try {
            Path newFile = getAbs(path);
            Files.createDirectories(newFile.getParent());
            BashUtils.append(filesToAdd, path.toString());

            return new FileWriter(newFile.toFile(), false);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private Path getAbs(Path path) {
        return getRoot().resolve(path);
    }

    @Override
    public boolean erase(Path path) {
        if (exists(path)) {
            term.eval("rm " + BashUtils.encode(getAbs(path)));
            return true;
        }

        return false;

    }

    @Override
    public Writer patch(Path path) {
        try {
            Path newFile = getAbs(path);
            Files.createDirectories(newFile.getParent());
            BashUtils.append(filesToAdd, path.toString());

            return new FileWriter(newFile.toFile(), true);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Reader get(Path path) {
        try {
            return new FileReader(getAbs(path).toFile());
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean exists(Path path) {
        return Files.exists(getAbs(path));
    }

    @Override
    public boolean isDir(Path path) {
        return getAbs(path).toFile().isDirectory();
    }

    @Override
    public int run(String cmd, Consumer<String> stdOut, Consumer<String> stdErr) {
        return BashUtils.lockDir(
                term,
                () -> {
                    term.eval("cd " + BashUtils.encode(getAbs(Path.of(""))));
                    return term.exec(cmd, stdOut, stdErr);
                }
        );
    }

    @Override
    public void find(Function<Path, Boolean> needContinue, Consumer<Path> sayBye, Comparator<Path> sorting) {
        Path root = getRoot();
        FileSystemUtils.tVerse(
                root,
                (p) -> needContinue.apply(root.relativize(p)),
                (p) -> sayBye.accept(root.relativize(p)),
                sorting
        );
    }

    @Override
    public void flush() {
        if (isReadOnly) {
            return;
        }

        if (!filesToAdd.toString().isBlank()) {
            BashUtils.lockDir(
                    term,
                    () -> {
                        term.eval("cd " + BashUtils.encode(getRoot()));
                        Arrays.stream(filesToAdd.toString().split(System.lineSeparator()))
                                .forEach(cFile -> GitBindings.add(term, Path.of(cFile)));
                        if (GitBindings.hasStatus(term)) {
                            GitBindings.commit(term);
                            term.eval("cd ..");
                            source.flush();
                        } else {
                            log.debug("Status has no changes, skip commit.");
                        }

                        return 0;
                    }
            );
        }

        filesToAdd = new StringBuffer();
    }

    @Override
    public FileSystem subTree(Path jump) {
        return new FileSystemSubDirectory(this, jump);
    }


    @Override
    public void close() throws Exception {
        source.close();
    }

    @Override
    public FileSystem subTree(PosixPath jump) {
        return new FileSystemSubDirectory(
                new VersionedFiles(false, getSource(), term),
                jump
        );
    }

    public Path getRoot() {
        return source.getRoot();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        VersionedFiles gitFiles = (VersionedFiles) o;
        return Objects.equals(source, gitFiles.source);
    }

    @Override
    public int hashCode() {
        return Objects.hash(source);
    }
}
