package io.github.andreyzebin.gitSql.pretty;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class RowRender implements Collector<Entry<String, Optional<String>>, List<String>, String> {

    private final Map<String, Integer> colNames;
    private final String delimiter;
    private final String cellSurround;
    private final boolean hasTabulation;
    private final Function<String, String> nullRender;
    private final Function<String, String> escape;

    public RowRender(
            Map<String, Integer> colNames,
            String delimiter,
            String cellSurround,
            boolean hasTabulation,
            Function<String, String> nullRender,
            Function<String, String> escape
    ) {
        this.colNames = colNames;
        this.delimiter = delimiter;
        this.cellSurround = cellSurround;
        this.hasTabulation = hasTabulation;
        this.nullRender = nullRender;
        this.escape = escape;
    }

    @Override
    public Supplier<List<String>> supplier() {
        return LinkedList::new;
    }

    @Override
    public BiConsumer<List<String>, Entry<String, Optional<String>>> accumulator() {
        return (a, b) -> {
            if (colNames.containsKey(b.getKey().toLowerCase())) {
                a.add(prepareCell(b));
            }
        };
    }


    @Override
    public BinaryOperator<List<String>> combiner() {
        return (a, b) -> {
            a.addAll(b);
            return a;
        };
    }

    @Override
    public Function<List<String>, String> finisher() {
        return (a) -> {
            final StringJoiner stringJoiner = new StringJoiner(delimiter);
            a.forEach(stringJoiner::add);

            return stringJoiner.toString();
        };
    }

    private String prepareCell(Entry<String, Optional<String>> cell) {
        Optional<String> value = cell.getValue();
        // render null
        String rValue = MyTableComposition.renderNullableValue(cell, nullRender);

        if (hasTabulation) {
            Integer maxWidth = colNames.get(cell.getKey().toLowerCase());
            value = cell.getValue();

            // trim to tab width
            if (rValue.length() > maxWidth - 1 - cellSurround.length() * 2) {
                rValue = rValue.substring(0, maxWidth - 4) + "...";
            }

            rValue = rValue + fill(maxWidth - rValue.length(), " ");
        }

        // escape
        rValue = escape.apply(rValue);

        rValue = cellSurround + rValue + cellSurround;
        return rValue;
    }

    public static String fill(int length, String brick) {
        return Stream.generate(() -> brick)
                .limit(length)
                .collect(Collectors.joining());
    }

    @Override
    public Set<Characteristics> characteristics() {
        return Set.of();
    }
}
