package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.git.VersionControl.Commit;

import java.nio.file.Path;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.function.Function;

import io.github.andreyzebin.gitSql.git.VersionedFiles;
import lombok.extern.slf4j.Slf4j;


@Slf4j
public final class TimeSeriesQuery extends PersistedIndex {

    public static final String COLUMN_NAME_POINT = "ts_point";
    public static final String COLUMN_NAME_HASH = "ts_hash";
    public static final String COLUMN_NAME_ORIGIN = "ts_origin";
    private final Collection<VersionedFiles> dataSetVersions;
    private final Map<String, Map<VersionedFiles, JdbcView>> indexes;

    // Sql queries per result table name
    Map<String, String> queries;
    private final Map<String, Function<VersionedFiles, JdbcView>> indexers;
    private final Instant from;
    private final Instant to;
    private final Map<String, List<String>> resultAliases;

    private final Map<String, AtomicBoolean> schemaStatuses;

    /**
     * Query few samples from data set {@link TimeSeriesQuery#dataSetVersions} with indexer
     * {@link TimeSeriesQuery#indexers} within given commit range
     * ({@link TimeSeriesQuery#from};{@link TimeSeriesQuery#to}) and merge to time series index. Provides a jdbc
     * connection {@link #getConnection()} to result index.
     */
    public TimeSeriesQuery(
            String sql,
            VersionedFiles dataSetVersions,
            Instant from,
            Instant to,
            BiFunction<? super VersionedFiles, Path, ? extends JdbcView> indexer
    ) {
        this(
                List.of(dataSetVersions),
                from,
                to,
                Map.of(
                        "result",
                        git -> indexer.apply(git, Path.of("run", "tsq_dataset", git.getRoot().toString()))
                ),
                Map.of(
                        "result",
                        sql
                )
        );
    }

    public TimeSeriesQuery(
            Collection<VersionedFiles> dataSetVersions,
            // Range
            Instant from,
            Instant to,
            // Indexer factories per result table alias
            Map<String, Function<VersionedFiles, JdbcView>> indexers,
            // Sql queries per result table name
            Map<String, String> queries
    ) {
        super(Path.of("run", "tsq_result"));
        this.dataSetVersions = dataSetVersions;
        this.indexes = new HashMap<>();
        this.indexers = indexers;
        this.queries = queries;
        this.from = from;
        this.to = to;
        this.schemaStatuses = new HashMap<>();
        this.resultAliases = new HashMap<>();
    }

    private JdbcView getView(VersionedFiles versionControl, String table) {
        return indexes.computeIfAbsent(table, f -> new HashMap<>())
                .computeIfAbsent(
                        versionControl,
                        f -> indexers.get(table).apply(versionControl)
                );
    }

    private boolean getSchemaStatus(String table) {
        return schemaStatuses.computeIfAbsent(table, f -> new AtomicBoolean(false)).get();
    }

    private List<String> getAliases(String table) {
        return resultAliases.computeIfAbsent(table, t -> new ArrayList<>());
    }

    private boolean setSchemaStatus(String table, boolean status) {
        return schemaStatuses.computeIfAbsent(table, f -> new AtomicBoolean(false)).getAndSet(status);
    }

    @Override
    protected void createSchema() throws SQLException {
        // schema is only known after some point within reindex()
    }

    @Override
    protected void reindex() {
        Connection connection = getConnection();
        dataSetVersions.forEach(
                cDataSet -> GitUtils.getPoints(cDataSet, from, to)
                        .forEach(cPoint -> reindexCommit(cDataSet, cPoint, connection))
        );

        try {
            super.createSchema();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }

    }

    private void reindexCommit(VersionedFiles dataSet, Commit cCommitPrev, Connection indexConnection) {
        indexers.keySet()
                .forEach(
                        cTable -> {
                            dataSet.getSource().seek(cCommitPrev.getHash());
                            JdbcView dataSetIndex = getView(dataSet, cTable).drop();

                            // query data set index and merge to time series index
                            SqlUtils.streamRows(
                                            (rs) -> prepareRow(
                                                    cTable,
                                                    dataSet.getSource().origin().orElse(dataSet.getRoot().toString()),
                                                    cCommitPrev,
                                                    rs
                                            ),
                                            SqlUtils.query(dataSetIndex.getConnection(), queries.get(cTable))
                                    )
                                    .forEach(
                                            cRow -> {
                                                // schema for index
                                                createSchema2(cTable);
                                                // save to index
                                                SqlUtils.merge(indexConnection, cTable, cRow.toArray(String[]::new));
                                            }
                                    );
                        }
                );
    }

    private LinkedList<String> prepareRow(String table, String origin, Commit cCommitPrev, ResultSet rs) {
        boolean needFillHeader = getAliases(table).isEmpty();

        // ALSO add ts_data_set VARCHAR(256),
        if (needFillHeader) {
            getAliases(table).add(COLUMN_NAME_POINT);
            getAliases(table).add(COLUMN_NAME_HASH);
            getAliases(table).add(COLUMN_NAME_ORIGIN);
        }

        List<String> row = new LinkedList<>();
        row.add(cCommitPrev.getTimestampInstant().toString());
        row.add(cCommitPrev.getHash());
        row.add(origin);
        SqlUtils.streamFields(rs)
                .forEach(
                        cColumn -> {
                            if (needFillHeader) {
                                getAliases(table).add(cColumn.getKey());
                            }

                            row.add(cColumn.getValue());
                        }
                );

        LinkedList<String> rov = new LinkedList<>(getAliases(table));
        rov.addAll(row);
        return rov;
    }


    private void createSchema2(String tableName) {
        if (getSchemaStatus(tableName)) {
            return;
        }
        if (storeExists()) {
            try (Statement dml = getConnection().createStatement();) {
                final StringBuilder sql1 = SqlUtils.renderSchema(tableName, getAliases(tableName));

                log.debug(CommitsIndex.renderSqlLog(sql1.toString()));

                dml.execute(sql1.toString());

                setSchemaStatus(tableName, true);
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public TimeSeriesQuery drop() {
        final TimeSeriesQuery drop = (TimeSeriesQuery) super.drop();

        schemaStatuses.forEach((t, v) -> v.set(false));
        return drop;
    }
}
