package io.github.andreyzebin.gitSql.pretty;

import static io.github.andreyzebin.gitSql.pretty.RowRender.fill;
import static io.github.andreyzebin.gitSql.pretty.TablePrinter.combine;

import java.io.BufferedWriter;
import java.io.IOException;
import java.util.AbstractMap.SimpleEntry;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import io.github.zebin.javabash.frontend.TerminalPalette;
import lombok.Builder;
import lombok.Data;
import lombok.With;

@Data
@Builder
@With
public class MyTableComposition implements TableComposition {

    private String horizontalSplitter;
    private String brick;
    private ToIntFunction<Entry<String, Optional<String>>> columnOrder;
    private ToIntFunction<Stream<Entry<String, String>>> indents;
    private boolean headingHorizontalBorders;
    private boolean totalsHorizontalBorders;
    private boolean needAfterRows;
    private String cellSurround;
    private boolean hasTabulation;
    private Function<Entry<String, String>, String> totalsRender;
    private final Function<String, String> nullRender;
    private final Function<String, String> escape;
    private final boolean skipHeading;


    public MyTableComposition(
            String horizontalSplitter,
            String verticalSplitter,
            ToIntFunction<Entry<String, Optional<String>>> columnOrder,
            ToIntFunction<Stream<Entry<String, String>>> indents,
            boolean headingHorizontalBorders,
            boolean totalsHorizontalBorders,
            boolean needAfterRows,
            String cellSurround,
            boolean hasTabulation,
            Function<Entry<String, String>, String> totalsRender,
            Function<String, String> nullRender,
            Function<String, String> escape, boolean skipHeading
    ) {

        this.horizontalSplitter = horizontalSplitter;
        this.brick = verticalSplitter;
        this.columnOrder = columnOrder;
        this.indents = indents;
        this.headingHorizontalBorders = headingHorizontalBorders;
        this.totalsHorizontalBorders = totalsHorizontalBorders;
        this.needAfterRows = needAfterRows;
        this.cellSurround = cellSurround;
        this.hasTabulation = hasTabulation;
        this.totalsRender = totalsRender;
        this.nullRender = nullRender;
        this.escape = escape;
        this.skipHeading = skipHeading;
    }

    public static MyTableComposition dumpCSV(
            List<String> rowNumber,
            String horizontalSplitter,
            String surround
    ) {
        return new MyTableComposition(
                horizontalSplitter,
                "",
                // column order
                constantOrder(rowNumber),
                // no limits
                value -> value.findFirst().get().getValue().length() + 1,
                false,
                false,
                false,
                surround,
                false,
                ResultRenders.renderValue(),
                (f) -> "null",
                (f) -> {
                    if (!surround.isEmpty()) {
                        f = f.replace(surround, surround + surround);
                    }
                    return f;
                },
                false
        );
    }

    public static MyTableComposition prettyPrint(int defaultWidth, List<SimpleEntry<String, Integer>> columns) {
        return new MyTableComposition(
                "",
                "-",
                // column order
                constantOrder(columns.stream().map(SimpleEntry::getKey).collect(Collectors.toList())),
                constantWidth(
                        columns.stream().collect(Collectors.toMap(SimpleEntry::getKey, SimpleEntry::getValue)),
                        defaultWidth
                ),
                true,
                true,
                true,
                "",
                true,
                ResultRenders.renderValue(),
                (f) -> "null",
                // no escape
                (f) -> f,
                false);
    }

    public static Consumer<String> println(BufferedWriter br) {
        return cLine -> {
            try {
                br.write(cLine);
                br.newLine();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        };
    }

    public static ToIntFunction<Stream<Entry<String, String>>> constantWidth(Map<String, Integer> rowNumber,
            int defaultValue) {
        return (a) -> {
            final String key = a.map(Entry::getKey).reduce((aa, bb) -> aa).get().toLowerCase();
            return rowNumber.getOrDefault(key, defaultValue);
        };
    }

    public static ToIntFunction<Entry<String, Optional<String>>> constantOrder(List<String> cols) {
        return aa -> {
            Map<String, Integer> colIndexes = new HashMap<>();

            cols.forEach((key) -> colIndexes.put(key.toLowerCase(), colIndexes.size()));
            return Optional.ofNullable(colIndexes.get(aa.getKey().toLowerCase()))
                    .orElse(Integer.MAX_VALUE);
        };
    }

    private Function<Map<String, Optional<String>>, String> formattedTable() {
        return (r) -> {
            final Stream<Entry<String, Optional<String>>> rowRAW = r.entrySet().stream();
            final Map<String, Integer> indents = r.entrySet()
                    .stream()
                    .map(stringStringEntry ->
                            new SimpleEntry<>(
                                    stringStringEntry.getKey().toLowerCase(),
                                    this.indents.applyAsInt(Stream.of(
                                            new SimpleEntry<>(
                                                    stringStringEntry.getKey(),
                                                    renderNullableValue(stringStringEntry, nullRender)
                                            )
                                    ))
                            )
                    )
                    .filter(se -> se.getValue() > 0)
                    .collect(Collectors.toMap(SimpleEntry::getKey,
                            SimpleEntry::getValue));

            // column indents
            // column horizontal delimiter
            RowRender rowRender = new RowRender(
                    indents,
                    horizontalSplitter,
                    cellSurround,
                    hasTabulation,
                    nullRender,
                    escape
            );
            return rowRAW
                    .sorted(Comparator.comparingInt(columnOrder))
                    .collect(rowRender);
        };
    }

    @Override
    public void getAfterRows(
            TablePrinter tableStream,
            LinkedList<String> strings
    ) {
        if (needAfterRows) {
            printTotals(tableStream, totalsRender, strings);
        }
    }

    @Override
    public Consumer<TableStreamRow> getRowConsumer(LinkedList<String> strings) {
        return row -> {
            final Map<String, Optional<String>> rowWithTotals = row.getRowWithTotals();
            if (!row.getTable().isHeaderPrinted() && !skipHeading) {

                // print heading
                int sumWidth = rowWithTotals.entrySet()
                        .stream()
                        .map(r -> indents.applyAsInt(
                                Stream.of(
                                        new SimpleEntry<>(
                                                r.getKey(),
                                                renderNullableValue(r, nullRender)
                                        )
                                )))
                        .reduce(Integer::sum)
                        .get();

                if (headingHorizontalBorders) {
                    strings.add(fill(sumWidth, brick));
                }
                printHeading(row, strings, formattedTable());
                if (headingHorizontalBorders) {
                    strings.add(fill(sumWidth, brick));
                }
            }

            // print row
            strings.add(formattedTable().apply(rowWithTotals));
        };
    }

    public static String renderNullableValue(Entry<String, Optional<String>> r, Function<String, String> nullRender1) {
        return r.getValue().isEmpty() ? nullRender1.apply(r.getKey())
                : r.getValue().get();
    }

    public void printTotals(
            TablePrinter tableStream,
            Function<Entry<String, String>, String> totalsRender,
            LinkedList<String> strings
    ) {

        final Integer sumWidth = tableStream.getLast().getRowWithTotals().entrySet()
                .stream().map(r -> indents.applyAsInt(Stream.of(new SimpleEntry<>(r.getKey(), renderNullableValue(r,
                        nullRender)))))
                .reduce(Integer::sum).get();

        if (totalsHorizontalBorders) {
            strings.add(fill(sumWidth, brick));
        }
        final Stream<Entry<String, Integer>> totals = tableStream.getLast()
                .getTotals()
                .entrySet()
                .stream();
        strings.add("Total: " + TerminalPalette.WHITE_BOLD_BRIGHT +
                totals.map(ce -> new SimpleEntry<>(ce.getKey(), ce.getValue().toString()))
                        .map(totalsRender)
                        .collect(Collectors.joining(brick))
                + TerminalPalette.RESET
        );
    }

    private static void printHeading(
            TableStreamRow row,
            LinkedList<String> strings,
            Function<Map<String, Optional<String>>, String> tableRender
    ) {
        Map<String, Optional<String>> heading = combine(
                row.getTable().getHeaders(),
                row.getTotals()
                        .entrySet()
                        .stream()
                        .collect(
                                Collectors.toMap(
                                        Entry::getKey,
                                        Entry::getKey
                                )
                        )
        ).entrySet()
                .stream()
                .collect(Collectors.toMap(Entry::getKey, vv -> Optional.ofNullable(vv.getValue())));
        strings.add(tableRender.apply(heading));
        row.getTable().setHeaderPrinted(true);
    }

}
