package io.github.andreyzebin.gitSql.cache;

import io.github.andreyzebin.gitSql.git.*;
import io.github.zebin.javabash.sandbox.DirectoryTree;

import java.nio.file.Path;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;

public class GitFsCacheProxy implements GitFs {
    private final AtomicReference<String> cacheControl;
    private final AtomicReference<String> commitCacheControl;
    private final Map<String, String> branchCache;
    private final Map<String, List<Commit>> commitCache;
    private final List<GitEventListener> listeners = new LinkedList<>();

    private final GitFs gfs;

    public GitFsCacheProxy(
            GitFs gfs,
            AtomicReference<String> cacheControl,
            AtomicReference<String> commitCacheControl,
            Map<String, String> branchCache,
            Map<String, List<Commit>> commitCache
    ) {
        this.gfs = gfs;
        this.cacheControl = cacheControl;
        this.commitCacheControl = commitCacheControl;
        this.branchCache = branchCache;
        this.commitCache = commitCache;
    }

    public static GitFsCacheProxy cachedProxy(
            GitFs gfs,
            AtomicReference<String> cacheControl
    ) {
        Map<String, List<Commit>> commitCache = new HashMap<>();
        AtomicReference<String> commitCacheControl = new AtomicReference<>(null);
        Map<String, String> branchCache = new HashMap<>();

        return new GitFsCacheProxy(
                gfs,
                cacheControl,
                commitCacheControl,
                branchCache,
                commitCache
        );
    }

    public void addListener(GitEventListener listener) {
        listeners.add(listener);
    }

    @Override
    public Versions seek(String commit) {
        cacheControl.set(commit);
        commitCacheControl.set(commit);
        branchCache.remove("branch");
        return gfs.seek(commit);
    }

    @Override
    public void setBranch(String branchName) {
        cacheControl.set(branchName);
        commitCacheControl.set(null);
        branchCache.put("branch", branchName);
        gfs.setBranch(branchName);
    }

    @Override
    public void merge(String hash) {
        listeners.forEach(cl -> cl.call("clear"));
        //cacheControl.set(branchName);
        commitCacheControl.set(null);
        //branchCache.put("branch", branchName);
        gfs.merge(hash);
    }

    @Override
    public Stream<BranchHead> listBranches() {
        return gfs.listBranches();
    }

    @Override
    public Optional<String> getBranch() {
        return Optional.ofNullable(
                branchCache.computeIfAbsent(
                        "branch",
                        br -> gfs.getBranch().orElse(null)
                )
        );
    }

    @Override
    public Stream<Commit> listCommits() {
        AtomicReference<List<Commit>> commits = new AtomicReference<>();
        if (commitCacheControl.get() == null) {
            commits.set((List<Commit>) gfs.listCommits().toList());
            commitCacheControl.set(commits.get().get(0).getHash());
        }

        return commitCache.computeIfAbsent(commitCacheControl.get(), cc -> {
                    if (commits.get() == null) {
                        return (List<Commit>) gfs.listCommits().toList();
                    }
                    return commits.get();
                })
                .stream();

    }

    @Override
    public boolean contains(String hash) {
        return gfs.contains(hash);
    }

    @Override
    public Stream<? extends Change> getChanges(String hashFrom) {
        return gfs.getChanges(hashFrom);
    }

    @Override
    public Stream<? extends Change> getChanges(String hashFrom, String hashTo) {
        return gfs.getChanges(hashFrom, hashTo);
    }

    @Override
    public Instant getTimestamp() {
        return gfs.getTimestamp();
    }

    @Override
    public void pull() {
        commitCacheControl.set(null);
        gfs.pull();
    }

    @Override
    public void push() {
        gfs.push();
    }

    @Override
    public void setOrigin(String origin) {
        gfs.setOrigin(origin);
    }

    @Override
    public Optional<String> getOrigin() {
        return gfs.getOrigin();
    }

    @Override
    public void fetch() {
        gfs.fetch();
    }

    @Override
    public void setUpstream(String local, String origin) {
        gfs.setUpstream(local, origin);
    }

    @Override
    public Optional<String> getUpstream(String local) {
        return gfs.getUpstream(local);
    }

    @Override
    public void reset() {
        commitCacheControl.set(null);
        gfs.reset();
    }

    @Override
    public void commit() {
        commitCacheControl.set(null);
        gfs.commit();
    }

    @Override
    public Path getRoot() {
        return gfs.getRoot();
    }

    @Override
    public DirectoryTree getDirectory() {
        DirectoryTreeCacheProxy dt = DirectoryTreeCacheProxy.cachedProxy(gfs.getDirectory(), cacheControl);
        addListener(dt);
        return dt;
    }

    @Override
    public void close() throws Exception {
        gfs.close();
    }
}
