package io.github.andreyzebin.gitSql.config;

import io.github.andreyzebin.gitSql.git.AbstractClient;
import io.github.andreyzebin.gitSql.git.BranchHead;
import io.github.andreyzebin.gitSql.git.GitAPI;
import io.github.andreyzebin.gitSql.git.GitFs;
import io.github.zebin.javabash.sandbox.AllFileManager;
import io.github.zebin.javabash.sandbox.PosixPath;
import lombok.extern.slf4j.Slf4j;

import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
public class RequestTree {
    private final Function<String, GitFs> originFactory;
    private final Function<GitFs, ConfigVersions> versionsFactory;
    private final String trunk;
    private final AllFileManager tt;

    public RequestTree(Function<String, GitFs> originFactory,
                       Function<GitFs, ConfigVersions> versionsFactory,
                       String trunk,
                       AllFileManager tt
    ) {
        this.originFactory = originFactory;
        this.versionsFactory = versionsFactory;
        this.trunk = trunk;
        this.tt = tt;
    }

    public Set<String> listBranches() {
        return originFactory.apply(trunk).listBranches().map(BranchHead::getName).collect(Collectors.toSet());
    }

    public ConfigVersions getBranch(String branch) {
        return versionsFactory.apply(originFactory.apply(branch));
    }

    public ConfigVersions getTrunk() {
        return getBranch(trunk);
    }

    public String getOffset(String branch) {
        return getBranch(branch).listVersions()
                .map(ConfigHistory.PropertiesVersion::getVersionHash)
                .filter(v -> {
                    GitAPI trunkControl = originFactory.apply(trunk);
                    return trunkControl.contains(v);
                })
                .findFirst().get();
    }

    public boolean merge(String branch, String hash) {
        GitFs trunkControl = originFactory.apply(trunk);
        PosixPath location = trunkControl.getLocation();

        tt.go(location);
        try {
            trunkControl.merge(hash);
            trunkControl.push();
        } catch (RuntimeException e) {
            log.error("Merge failed: ", e);
        }

        StringBuilder sb = new StringBuilder();
        tt.getTerminal().exec(inject("git diff --check"), sb::append, sb::append);
        // git diff --find-renames 11b072f3f4dd433ebf5fd931f81df4cf643bb26b a98be9580c3ed2f9a0280aaf744f0aa9d0d28473
        if (sb.toString().lines().findAny().isPresent()) {
            // wait till unlocked - then rebase and try again
            trunkControl.reset();
            trunkControl.setBranch(trunk);
            return false;
        }

        return true;
    }

    public static String inject(String gitMerge, Object... ff) {
        return String.format(gitMerge, ff);
    }
}
