package io.github.andreyzebin.gitSql.git;


import io.github.andreyzebin.gitSql.config.DirectoryTreeFactory;
import io.github.zebin.javabash.process.TextTerminal;
import io.github.zebin.javabash.sandbox.*;
import lombok.extern.slf4j.Slf4j;

import java.nio.file.Path;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Stream;

@Slf4j
public class RemoteOrigin extends AbstractClient {

    protected String gitUri;
    private final TextTerminal bash;
    private final AllFileManager fm;
    private final GitAuth authStrategy;
    protected Path root;

    private final String branch;
    private String defaultBranch;

    private final Path home;
    private final GitConfigurations configurations;
    private final DirectoryTreeFactory dtf;

    public RemoteOrigin(String gitUri, AllFileManager bash, GitAuth authStrategy) {
        this(gitUri, bash, authStrategy, null, new DefaultConfigurations(bash), WorkingDirectory::new);
    }

    public RemoteOrigin(String gitUri, AllFileManager bash, GitAuth authStrategy, String branch) {
        this(gitUri, bash, authStrategy, branch, new DefaultConfigurations(bash), WorkingDirectory::new);
    }

    public RemoteOrigin(
            String gitUri,
            AllFileManager fm,
            GitAuth authStrategy,
            String branch,
            GitConfigurations configurations,
            DirectoryTreeFactory dtf
    ) {
        super(fm, dtf);
        this.dtf = dtf;
        this.fm = fm;
        this.branch = branch;
        this.gitUri = gitUri;
        this.bash = fm.getTerminal();
        this.authStrategy = authStrategy;
        this.home = configurations.getHomeTemporaryDir();
        this.configurations = configurations;
    }

    public LocalSource getLocal() {
        return new LocalSource(getRoot(), fm);
    }

    @Override
    public void close() throws Exception {
        if (root != null) {
            setupDir(() -> {
                log.debug("Cleaning up temporary git root...");
                fm.goUp();
                bash.eval("rm -rf " + root.getFileName().toString());
            });
        }
    }

    public void withTerminal(Consumer<TextTerminal> evals) throws Exception {
        if (root != null) {
            setupDir(() -> evals.accept(bash));
        }
    }

    public Stream<BranchHead> listBranches() {
        authStrategy.beforeRemote(bash);
        Stream<BranchHead> remoteBranches = GitBindings.getRemoteBranches(
                authStrategy.injectAuth(gitUri),
                bash,
                authStrategy.useAuthHeader()
        );
        authStrategy.afterRemote(bash);

        return remoteBranches;
    }

    @Override
    public Path getRoot() {
        if (root == null) {
            log.debug("Initializing Git root...");

            return recoverState(() -> {
                String tempDirName = configurations.getTemporaryName(gitUri, branch);
                try {
                    PosixPath girRoot = PosixPath.of(home).climb(PosixPath.ofPosix(tempDirName));
                    fm.makeDir(girRoot);
                    fm.go(girRoot);

                    authStrategy.beforeRemote(bash);

                    GitBindings.clone(
                            ".",
                            bash,
                            authStrategy.injectAuth(gitUri),
                            false,
                            authStrategy.useAuthHeader(),
                            null
                    );
                    authStrategy.afterRemote(bash);

                    return root = girRoot.toPath();
                } catch (Exception e) {
                    log.error("Error while loading from remote: ", e);
                    throw new RuntimeException(e);
                }
            });
        }
        return root;
    }

    @Override
    public void push() {
        if (root != null) {
            BashUtils.lockDir(bash, () -> {
                bash.eval("cd " + BashUtils.encode(getRoot()));
                authStrategy.beforeRemote(bash);
                GitBindings.push(bash);
                authStrategy.afterRemote(bash);
                return 0;
            });
        }
    }

    @Override
    public void setOrigin(String origin) {

    }

    @Override
    public void pull() {
        BashUtils.lockDir(bash, () -> {
            bash.eval("cd " + BashUtils.encode(getRoot()));
            authStrategy.beforeRemote(bash);
            GitBindings.pullAll(bash);
            authStrategy.afterRemote(bash);
            return 0;
        });
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RemoteOrigin that = (RemoteOrigin) o;
        return Objects.equals(gitUri, that.gitUri) && Objects.equals(branch, that.branch);
    }

    @Override
    public int hashCode() {
        return Objects.hash(gitUri, branch);
    }

}
