package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.sql.SqlUtils.SchemaBuilder;
import java.nio.file.Path;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;


@Slf4j
public class SuperAggregation extends PersistedIndex {

    private final Collection<TableSource> sources;

    public SuperAggregation(
            Path storeDir,
            Collection<TableSource> sources
    ) {
        super(storeDir);
        this.sources = sources;
    }

    @Override
    protected void createSchema() throws SQLException {
        super.createSchema();
        log.info("Creating schema...");
        try (Statement dml = getConnection().createStatement();) {
            String dml2 = new SchemaBuilder("source_log")
                    .withColumn("hash")
                    .withColumn("timestamp")
                    .withColumn("version")
                    .withMerge("hash")
                    .withIndex("UNIQUE NULLS ALL DISTINCT (hash)")
                    .toDML();
            log.debug("Executing DML: " + CommitsIndex.renderSqlLog(dml2));
            dml.execute(dml2);

            sources.forEach(cSource -> {
                try {
                    String dml1 = cSource.createSchema().toDML();
                    log.debug("Executing DML: " + CommitsIndex.renderSqlLog(dml1));
                    dml.execute(dml1);
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            });
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void init() {
        super.init();

        try {
            createSchema();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        reindex();

    }

    @Override
    protected void reindex() {
        super.reindex();

        sources.forEach(
                cSource -> {

                    final Optional<Stream<Entry<String, String>>> hashLog = SqlUtils.streamRows(
                            SqlUtils::streamFields,
                            SqlUtils.query(
                                    getConnection(),
                                    "SELECT version FROM source_log WHERE hash = " + cSource.hashCode() + ";"
                            )
                    ).findFirst();

                    if (hashLog.isPresent()) {
                        String version = hashLog.get()
                                .filter(cEntr -> cEntr.getKey().equalsIgnoreCase("version"))
                                .findFirst().get().getValue();
                        cSource = cSource.updates(version);
                    }

                    SqlUtils.merge(
                            getConnection(),
                            "source_log",
                            List.of("hash"),
                            Map.of(
                                    "hash", String.valueOf(cSource.hashCode()),
                                    "timestamp", String.valueOf(LocalDateTime.now()),
                                    "version", cSource.version()
                            )
                    );

                    TableSource finalCSource = cSource;
                    cSource.rows()
                            .forEach(
                                    cRow -> {
                                        try {
                                            SqlUtils.merge(
                                                    getConnection(),
                                                    finalCSource.createSchema().getAlias(),
                                                    finalCSource.createSchema().getMergeColumns(),
                                                    cRow
                                            );
                                        } catch (SQLException e) {
                                            throw new RuntimeException(e);
                                        }
                                    }
                            );

                }
        );
    }

    @Override
    public SuperAggregation drop() {
        return (SuperAggregation) super.drop();
    }
}
