package io.github.cruisoring.tuple;

import io.github.cruisoring.Functions;
import io.github.cruisoring.TypeHelper;
import io.github.cruisoring.utility.ArrayHelper;
import io.github.cruisoring.utility.Logger;

import java.lang.reflect.Array;
import java.util.*;
import java.util.function.Predicate;

/**
 * This is a special data structure contains multiple immutable elements in fixed sequence. The AutoCloseable implementation
 * would close any elements if they are also AutoCloseable.
 */
public class Tuple implements AutoCloseable, Comparable<Tuple>, WithValues {

    public static final Tuple0 UNIT = new Tuple0();
    public static final Tuple1 TRUE = new Tuple1(true);
    public static final Tuple1 FALSE = new Tuple1(false);

    protected final Object[] values;
    private Integer _hashCode;
    private int[][] deepLength;
    private String _toString;

    /**
     * Protected constructor to keep the elements as a final array.
     * @param elements values to be persisted by the Tuple.
     */
    protected Tuple(Object... elements){
        if(elements == null){
            elements = new Object[]{elements};
        }
        int length = elements.length;
        values = new Object[length];
        for (int i=0; i<length; i++){
            values[i] = elements[i];
        }
    }

    @Override
    public Object getValueAt(int index) {
        if(index < 0 || index >= values.length)
            return null;
        return values[index];
    }


    /**
     * Get all Non-null elements matching the given class as an immutable <code>Set</code>
     * @param clazz Class to evaluate the saved values.
     * @param <T>   Type of the given Class.
     * @return      Immutable <code>Set</code> containing matched elements.
     */
    public <T> Set<T> getSetOf(Class<T> clazz){
        Objects.requireNonNull(clazz);

        try {
            Predicate<Class> predicate = TypeHelper.getClassEqualitor(clazz);
            Class equivalent = (TypeHelper.isPrimitive(clazz) && !clazz.isArray()) ? TypeHelper.getEquivalentClass(clazz) : clazz;
            Boolean isArray = clazz.isArray();
            Class<?> componentType = isArray ? clazz.getComponentType() : null;

            int length = getLength();
            List<T> list = new ArrayList<T>();
            for (int i = 0; i < length; i++) {
                Object v = values[i];
                if (v != null) {
                    Class vClass = v.getClass();
                    if(equivalent.equals(vClass)){
                        list.add((T)v);
                    } else if (predicate.test(vClass)) {
                        T t = (T) TypeHelper.getToEquivalentParallelConverter(vClass).apply(v);
                        list.add(t);
                    }
                }
            }
            Object[] array = list.toArray((T[])ArrayHelper.getNewArray(equivalent, list.size()));
            return setOf(equivalent, (T[])array);
        }catch (Exception ex){
            return null;
        }
    }

    /**
     * Get all Non-null elements matching the given class as an immutable <code>Set</code>
     * @param clazz Class to evaluate the saved values.
     * @param valuePredicate predicate to filter elements by their values
     * @param <T>   Type of the given Class.
     * @return      Immutable <code>Set</code> containing matched elements.
     */
    public <T> Set<T> getSetOf(Class<T> clazz, Predicate<T> valuePredicate){
        Objects.requireNonNull(clazz);
        Objects.requireNonNull(valuePredicate);
        List<T> matched = new ArrayList<>();

        Predicate<Class> classPredicate = TypeHelper.getClassEqualitor(clazz);

        int length = getLength();
        for (int i = 0; i < length; i++) {
            Object v = values[i];
            if(v != null){
                try {
                    if(classPredicate.test(v.getClass()) && valuePredicate.test((T)v))
                        matched.add((T)v);
                }catch (Exception ex){}
            }
        };
        T[] array = (T[]) matched.stream().toArray();
        return setOf(clazz, array);
    }

    /**
     * Get the length of the saved values.
     * @return length of the values.
     */
    public int getLength(){
        return values.length;
    }

    public int[][] getDeepLength() {
        if(deepLength == null){
            deepLength = TypeHelper.getDeepLength(values);
        }
        return deepLength;
    }

    /**
     * Compares this object with the specified object for order.  Returns a
     * negative integer, zero, or a positive integer as this object is less
     * than, equal to, or greater than the specified object.
     * <p>The implementor must ensure <tt>sgn(x.compareTo(y)) ==
     * -sgn(y.compareTo(x))</tt> for all <tt>x</tt> and <tt>y</tt>.  (This
     * implies that <tt>x.compareTo(y)</tt> must throw an exception iff
     * <tt>y.compareTo(x)</tt> throws an exception.)
     * </p>
     * <p>The implementor must also ensure that the relation is transitive:
     * <tt>(x.compareTo(y)&gt;0 &amp;&amp; y.compareTo(z)&gt;0)</tt> implies
     * <tt>x.compareTo(z)&gt;0</tt>.
     * </p>
     * <p>Finally, the implementor must ensure that <tt>x.compareTo(y)==0</tt>
     * implies that <tt>sgn(x.compareTo(z)) == sgn(y.compareTo(z))</tt>, for
     * all <tt>z</tt>.
     * </p>
     * <p>It is strongly recommended, but <i>not</i> strictly required that
     * <tt>(x.compareTo(y)==0) == (x.equals(y))</tt>.  Generally speaking, any
     * class that implements the <tt>Comparable</tt> interface and violates
     * this condition should clearly indicate this fact.  The recommended
     * language is "Note: this class has a natural ordering that is
     * inconsistent with equals."
     * </p>
     * <p>In the foregoing description, the notation
     * <tt>sgn(</tt><i>expression</i><tt>)</tt> designates the mathematical
     * <i>signum</i> function, which is defined to return one of <tt>-1</tt>,
     * <tt>0</tt>, or <tt>1</tt> according to whether the value of
     * <i>expression</i> is negative, zero or positive.
     * </p>
     *
     * @param o the object to be compared.
     * @return a negative integer, zero, or a positive integer as this object
     * is less than, equal to, or greater than the specified object.
     * @throws NullPointerException if the specified object is null
     * @throws ClassCastException   if the specified object's type prevents it
     *                              from being compared to this object.
     */
    @Override
    public int compareTo(Tuple o) {
        int length = o.getLength();
        int result = this.getLength() - length;
        if(result != 0)
            return result;

        String thisString = this.toString();
        String oString = o.toString();

        result = thisString.length()-oString.length();
        if(result != 0)
            return result;

        return thisString.compareTo(oString);
    }

    @Override
    public int hashCode() {
        if(_hashCode == null){
            _hashCode = Arrays.deepHashCode(values);
        }
        return _hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        int length = values.length;
        if(obj == null || !(obj instanceof Tuple) || length != ((Tuple) obj).getLength())
            return false;
        if (obj == this)
            return true;
        Tuple other = (Tuple)obj;
        if(!other.canEqual(this))
            return false;

        return TypeHelper.valueEquals(values, other.values, getDeepLength(), other.getDeepLength());
    }

    public boolean canEqual(Object obj) {
        return (obj instanceof Tuple);
    }

    @Override
    public String toString() {
        if(_toString==null){
            _toString = TypeHelper.deepToString(values);
        }
        return _toString;
    }

    private boolean closed = false;

    /**
     * Check the flag <code>closed</code> first, then close the saved values if they are AutoCloseable in reverse order.
     * @throws Exception any exception that could be thrown when closing AutoCloseable objects
     */
    @Override
    public void close() throws Exception {
        if(!closed) {
            //Close AutoCloseable object in reverse order
            for (int i = values.length - 1; i >= 0; i--) {
                Object value = values[i];
                if (value != null && value instanceof AutoCloseable) {
                    Functions.Default.run(() -> ((AutoCloseable) value).close());
                    Logger.L("%s closed()", value);
                }
            }
            closed = true;
            Logger.L("%s.close() run successfully!", this);
        }
    }

//    @Override
//    protected void finalize() throws Throwable {
//        if(!closed)
//        {
//            Logger.L( "******FORGOT TO CLOSE THE TUPLE!" );
//        }
//        super.finalize();
//    }

    //region Factories to create Strong-typed Tuple instances based on the number of given arguments

    /**
     * Create a Tuple instance to keep any number of elements without caring about their Type info.
     * @param elements  All elements to be persisted by the Tuple
     * @return          A <tt>Tuple</tt> instance with length of the elements
     */
    public static Tuple of(Object... elements){
        if (elements == null){
            return create(null);
        }
        int length = elements.length;
        switch (length){
            case 0: return UNIT;
            case 1: return new Tuple1(elements[0]);
            case 2: return new Tuple2(elements[0], elements[1]);
            case 3: return new Tuple3(elements[0], elements[1], elements[2]);
            case 4: return new Tuple4(elements[0], elements[1], elements[2], elements[3]);
            case 5: return new Tuple5(elements[0], elements[1], elements[2], elements[3], elements[4]);
            case 6: return new Tuple6(elements[0], elements[1], elements[2], elements[3], elements[4], elements[5]);
            case 7: return new Tuple7(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6]);
            case 8: return new Tuple8(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7]);
            case 9: return new Tuple9(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8]);
            case 10: return new Tuple10(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9]);
            case 11: return new Tuple11(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10]);
            case 12: return new Tuple12(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11]);
            case 13: return new Tuple13(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12]);
            case 14: return new Tuple14(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13]);
            case 15: return new Tuple15(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14]);
            case 16: return new Tuple16(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14],
                    elements[15]);
            case 17: return new Tuple17(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14],
                    elements[15], elements[16]);
            case 18: return new Tuple18(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14],
                    elements[15], elements[16], elements[17]);
            case 19: return new Tuple19(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14],
                    elements[15], elements[16], elements[17], elements[18]);
            case 20: return new Tuple20(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14],
                    elements[15], elements[16], elements[17], elements[18], elements[19]);

            default: return new TuplePlus(elements[0], elements[1], elements[2], elements[3], elements[4],
                    elements[5], elements[6], elements[7], elements[8], elements[9],
                    elements[10], elements[11], elements[12], elements[13], elements[14],
                    elements[15], elements[16], elements[17], elements[18], elements[19], Arrays.copyOfRange(elements, 20, length));
        }
    }

    /**
     * Crate an Tuple0 that has no value saved, shall not be called directly.
     * @return  The Tuple0 object.
     */
    protected static Tuple0 create(){
        return UNIT;
    }

    /**
     * Create a Tuple1 instance that contains only one value of Type T
     * @param t     Element to be persisted by the Tuple
     * @param <T>   Type of the element <code>t</code>
     * @return      Tuple containing 6 elements that could be accessed as their original types.
     */
    public static <T> Tuple1<T> create(T t){
        return new Tuple1(t);
    }

    /**
     * Create a Tuple2 instance that contains 2 elements of Types T/U respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @return      Tuple containing 2 elements that could be accessed as their original types.
     */
    public static <T,U> Tuple2<T,U> create(T t, U u){
        return new Tuple2(t, u);
    }

    /**
     * Create a Tuple3 instance that contains 3 elements of Types T/U/V respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @return      Tuple containing 3 elements that could be accessed as their original types.
     */
    public static <T,U,V> Tuple3<T,U,V> create(T t, U u, V v){
        return new Tuple3(t, u, v);
    }

    /**
     * Create a Tuple4 instance that contains 4 elements of Types T/U/V/W respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @return      Tuple containing 4 elements that could be accessed as their original types.
     */
    public static <T,U,V,W> Tuple4<T,U,V,W> create(T t, U u, V v, W w){
        return new Tuple4(t, u, v, w);
    }

    /**
     * Create a Tuple5 instance that contains 5 elements of Types T/U/V/W/X respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @return      Tuple containing 5 elements that could be accessed as their original types.
     */
    public static <T,U,V,W,X> Tuple5<T,U,V,W,X> create(T t, U u, V v, W w, X x){
        return new Tuple5(t, u, v, w, x);
    }

    /**
     * Create a Tuple6 instance that contains 6 elements of Types T/U/V/W/X/Y respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @return      Tuple containing 6 elements that could be accessed as their original types.
     */
    public static <T,U,V,W,X,Y> Tuple6<T,U,V,W,X,Y> create(T t, U u, V v, W w, X x, Y y){
        return new Tuple6(t, u, v, w, x, y);
    }

    /**
     * Create a Tuple7 instance that contains 7 elements of Types T/U/V/W/X/Y/Z respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @return      Tuple containing seven elements that could be accessed as their original types.
     */
    public static <T,U,V,W,X,Y,Z> Tuple7<T,U,V,W,X,Y,Z> create(T t, U u, V v, W w, X x, Y y, Z z){
        return new Tuple7(t, u, v, w, x, y, z);
    }

    /**
     * Create a Tuple8 instance that contains 8 elements of 8 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @return      Tuple containing 8 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A> Tuple8<T,U,V,W,X,Y,Z,A> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a){
        return new Tuple8(t, u, v, w, x, y, z, a);
    }

    /**
     * Create a Tuple9 instance that contains 9 elements of 9 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @return      Tuple containing 9 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B> Tuple9<T,U,V,W,X,Y,Z,A,B> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b){
        return new Tuple9(t, u, v, w, x, y, z, a, b);
    }
    
    /**
     * Create a Tuple10 instance that contains 10 elements of 10 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @return      Tuple containing 10 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C> Tuple10<T,U,V,W,X,Y,Z,A,B,C> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c){
        return new Tuple10(t, u, v, w, x, y, z, a, b, c);
    }
    
    /**
     * Create a Tuple11 instance that contains 11 elements of 11 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @return      Tuple containing 11 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D> Tuple11<T,U,V,W,X,Y,Z,A,B,C,D> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d){
        return new Tuple11(t, u, v, w, x, y, z, a, b, c, d);
    }

    /**
     * Create a Tuple12 instance that contains 12 elements of 12 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @return      Tuple containing 12 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E> Tuple12<T,U,V,W,X,Y,Z,A,B,C,D,E> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e){
        return new Tuple12(t, u, v, w, x, y, z, a, b, c, d, e);
    }

    /**
     * Create a Tuple13 instance that contains 13 elements of 13 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @return      Tuple containing 13 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F> Tuple13<T,U,V,W,X,Y,Z,A,B,C,D,E,F> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f){
        return new Tuple13(t, u, v, w, x, y, z, a, b, c, d, e, f);
    }
    
    /**
     * Create a Tuple14 instance that contains 14 elements of 14 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @return      Tuple containing 14 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G> Tuple14<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g){
        return new Tuple14(t, u, v, w, x, y, z, a, b, c, d, e, f, g);
    }

    /**
     * Create a Tuple15 instance that contains 15 elements of 15 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @return      Tuple containing 15 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H> Tuple15<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h){
        return new Tuple15(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h);
    }


    /**
     * Create a Tuple16 instance that contains 16 elements of 16 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param i     Sixteenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @param <I>   Type of the 16th element <code>i</code>
     * @return      Tuple containing 16 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I> Tuple16<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h, I i){
        return new Tuple16(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h, i);
    }

    /**
     * Create a Tuple17 instance that contains 17 elements of 17 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param i     Sixteenth element to be persisted by the Tuple
     * @param j     Seventeenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @param <I>   Type of the 16th element <code>i</code>
     * @param <J>   Type of the 17th element <code>j</code>
     * @return      Tuple containing 17 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J> Tuple17<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h, I i, J j){
        return new Tuple17(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h, i, j);
    }

    /**
     * Create a Tuple18 instance that contains 18 elements of 18 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param i     Sixteenth element to be persisted by the Tuple
     * @param j     Seventeenth element to be persisted by the Tuple
     * @param k     Eighteenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @param <I>   Type of the 16th element <code>i</code>
     * @param <J>   Type of the 17th element <code>j</code>
     * @param <K>   Type of the 18th element <code>k</code>
     * @return      Tuple containing 18 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K> Tuple18<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h, I i, J j, K k){
        return new Tuple18(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h, i, j, k);
    }

    /**
     * Create a Tuple19 instance that contains 19 elements of 19 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param i     Sixteenth element to be persisted by the Tuple
     * @param j     Seventeenth element to be persisted by the Tuple
     * @param k     Eighteenth element to be persisted by the Tuple
     * @param l     Nineteenth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @param <I>   Type of the 16th element <code>i</code>
     * @param <J>   Type of the 17th element <code>j</code>
     * @param <K>   Type of the 18th element <code>k</code>
     * @param <L>   Type of the 19th element <code>l</code>
     * @return      Tuple containing 19 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K,L> Tuple19<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K,L> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h, I i, J j, K k, L l){
        return new Tuple19(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h, i, j, k, l);
    }

    /**
     * Create a Tuple20 instance that contains 20 elements of 20 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param i     Sixteenth element to be persisted by the Tuple
     * @param j     Seventeenth element to be persisted by the Tuple
     * @param k     Eighteenth element to be persisted by the Tuple
     * @param l     Nineteenth element to be persisted by the Tuple
     * @param m     Twentieth element to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @param <I>   Type of the 16th element <code>i</code>
     * @param <J>   Type of the 17th element <code>j</code>
     * @param <K>   Type of the 18th element <code>k</code>
     * @param <L>   Type of the 19th element <code>l</code>
     * @param <M>   Type of the 20th element <code>m</code>
     * @return      Tuple containing 20 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K,L,M> Tuple20<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K,L,M> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h, I i, J j, K k, L l, M m){
        return new Tuple20(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h, i, j, k, l, m);
    }

    /**
     * Create a TuplePlus instance that contains more than 20 elements of 20 types respectively
     * @param t     First element to be persisted by the Tuple
     * @param u     Second element to be persisted by the Tuple
     * @param v     Third element to be persisted by the Tuple
     * @param w     Fourth element to be persisted by the Tuple
     * @param x     Fifth element to be persisted by the Tuple
     * @param y     Sixth element to be persisted by the Tuple
     * @param z     Seventh element to be persisted by the Tuple
     * @param a     Eighth element to be persisted by the Tuple
     * @param b     Ninth element to be persisted by the Tuple
     * @param c     Tenth element to be persisted by the Tuple
     * @param d     Eleventh element to be persisted by the Tuple
     * @param e     Twelfth element to be persisted by the Tuple
     * @param f     Thirteenth element to be persisted by the Tuple
     * @param g     Fourteenth element to be persisted by the Tuple
     * @param h     Fifteenth element to be persisted by the Tuple
     * @param i     Sixteenth element to be persisted by the Tuple
     * @param j     Seventeenth element to be persisted by the Tuple
     * @param k     Eighteenth element to be persisted by the Tuple
     * @param l     Nineteenth element to be persisted by the Tuple
     * @param m     Twentieth element to be persisted by the Tuple
     * @param others     All other elements to be persisted by the Tuple
     * @param <T>   Type of the first element <code>t</code>
     * @param <U>   Type of the second element <code>u</code>
     * @param <V>   Type of the third element <code>v</code>
     * @param <W>   Type of the fourth element <code>w</code>
     * @param <X>   Type of the fifth element <code>x</code>
     * @param <Y>   Type of the sixth element <code>y</code>
     * @param <Z>   Type of the seventh element <code>z</code>
     * @param <A>   Type of the 8th element <code>a</code>
     * @param <B>   Type of the 9th element <code>b</code>
     * @param <C>   Type of the 10th element <code>c</code>
     * @param <D>   Type of the 11th element <code>d</code>
     * @param <E>   Type of the 12th element <code>e</code>
     * @param <F>   Type of the 13th element <code>f</code>
     * @param <G>   Type of the 14th element <code>g</code>
     * @param <H>   Type of the 15th element <code>h</code>
     * @param <I>   Type of the 16th element <code>i</code>
     * @param <J>   Type of the 17th element <code>j</code>
     * @param <K>   Type of the 18th element <code>k</code>
     * @param <L>   Type of the 19th element <code>l</code>
     * @param <M>   Type of the 20th element <code>m</code>
     * @return      Tuple containing more than 20 elements that could be retrieved as their original types.
     */
    public static <T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K,L,M> TuplePlus<T,U,V,W,X,Y,Z,A,B,C,D,E,F,G,H,I,J,K,L,M> create(
            T t, U u, V v, W w, X x, Y y, Z z, A a, B b, C c, D d, E e, F f, G g, H h, I i, J j, K k, L l, M m, Object... others){
        return new TuplePlus(t, u, v, w, x, y, z, a, b, c, d, e, f, g, h, i, j, k, l, m, others);
    }

    /**
     * Factory to create a <tt>Set</tt> instance of type <tt>T</tt> with elements of the same type
     * @param elements  Elements of same type <tt>T</tt> to be persisted
     * @param <T>       Type of the given elements
     * @return          A strong-typed Set instance
     */
    public static <T> Set<T> setOf(T... elements) {
        if(elements == null)
            return (Set<T>) Functions.ReturnsDefaultValue.apply(Set::new, elements);
        Class<T> elementType = (Class<T>) elements.getClass().getComponentType();
        return new Set<T>(elementType, elements);
    }

    /**
     * Factory to create a <tt>Set</tt> instance of type <tt>T</tt> with elements of the same type, and their type to cope with Tpe Erasure
     * @param elementType   Class of the Type of the elements
     * @param elements      Elements of the same type T
     * @param <T>           Actually Type of the elements
     * @return              A strong-typed Set instance
     */
    public static <T> Set<T> setOf(Class<T> elementType, T[] elements){
        return new Set<T>(elementType, elements);
    }

    /**
     * Factory to create a <code>Set</code> of type <code>T</code>
     * @param collection    Collection of Elements of type <code>T</code> to be persisted
     * @param clazz         Class of the elements to cope with Java Type Erasure
     * @param <T>           Declared Type of the elements
     * @return              Strong-typed Set, with component type of <code>T</code>
     */
    public static <T> Set<T> setOf(Collection<T> collection, Class<? extends T> clazz){
        Objects.requireNonNull(collection);
        T[] array = (T[])collection.toArray((T[]) Array.newInstance(clazz, 0));
        return setOf(array);
    }

    //endregion Factories to create Strong-typed Tuple instances based on the number of given arguments

}
