package io.github.endreman0.javajson.nodes;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

/**
 * A JSON array node consists of a series of nodes separated by commas, enclosed in square brackets ({@code []}).
 * The enclosed nodes may be of any type, and are not necessarily the same type. You can have arrays within arrays as well.
 * You can access the contents of this node much like you would a {@link java.util.List List&lt;Node&gt;}.
 * @author endreman0
 */
public class ArrayNode extends ParentNode implements Iterable<Node>{
	private List<Node> children = new ArrayList<Node>();
	/**
	 * Create an empty array node.
	 */
	public ArrayNode(){super();}
	/**
	 * Create an array node with the given nodes as children.
	 * @param children The nodes to add as children
	 */
	public ArrayNode(Node... children){this(); addAll(children);}
	/**
	 * Create an array node with the given nodes as children.
	 * @param children The nodes to add as children
	 */
	public ArrayNode(Iterable<Node> children){this(); addAll(children);}
	/**
	 * Add the given node to this node's children.
	 * @param node The node to add as a child
	 * @return {@code this}, for chaining
	 */
	public ArrayNode add(Node node){
		if(node != null){
			children.add(node);
			node.parent = this;
		}
		return this;
	}
	/**
	 * Add the given node to this node's children at the specified index.
	 * The node currently at the given index, if any, will be shifted to the right (index will increase by 1).
	 * @param index The index to add the node at
	 * @param node The node to add as a child
	 * @return {@code this}, for chaining
	 */
	public ArrayNode add(int index, Node node){
		children.add(index, node);
		node.parent = this;
		return this;
	}
	/**
	 * Add the given nodes to this node's children.
	 * @param nodes The nodes to add as children
	 * @return {@code this}, for chaining
	 */
	public ArrayNode addAll(Node... nodes){
		for(Node node : nodes) add(node);
		return this;
	}
	/**
	 * Add the given nodes to this node's children.
	 * @param nodes The nodes to add as children
	 * @return {@code this}, for chaining
	 */
	public ArrayNode addAll(Iterable<Node> nodes){
		for(Node node : nodes) add(node);
		return this;
	}
	/**
	 * Add the given nodes to this node's children at the specified index.
	 * The nodes will occupy indices {@code index} to {@code index + nodes.length - 1}.
	 * The nodes currently at those indices, if any, will be shifted to the right (index will increase) until they are out of the way.
	 * @param index The index to add the node at
	 * @param nodes The nodes to add as children
	 * @return {@code this}, for chaining
	 */
	public ArrayNode addAll(int index, Node... nodes){
		for(Node node : nodes) add(index++, node);
		return this;
	}
	/**
	 * Add the given nodes to this node's children at the specified index.
	 * The nodes will occupy indices {@code index} to {@code index + nodes.length - 1}.
	 * The nodes currently at those indices, if any, will be shifted to the right (index will increase) until they are out of the way.
	 * @param index The index to add the node at
	 * @param nodes The nodes to add as children
	 * @return {@code this}, for chaining
	 */
	public ArrayNode addAll(int index, Iterable<Node> nodes){
		for(Node node : nodes) add(index++, node);
		return this;
	}
	/**
	 * Get the child node at the specified index.
	 * @param index The index to get from
	 * @return The node at the specified index.
	 */
	public Node get(int index){return children.get(index);}
	/**
	 * Get the child node at the specified index, if it is an array node.
	 * This method gets the node at the specified index and tests if it is an array node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param index The index to get from
	 * @return The node at the specified index, if it is an array node; {@code null} otherwise
	 */
	public ArrayNode getArray(int index){
		Node node = get(index);
		if(node instanceof ArrayNode) return (ArrayNode)node; else return null;
	}
	/**
	 * Get the child node at the specified index, if it is a boolean node.
	 * This method gets the node at the specified index and tests if it is a boolean node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param index The index to get from
	 * @return The node at the specified index, if it is a boolean node; {@code null} otherwise
	 */
	public BooleanNode getBoolean(int index){
		Node node = get(index);
		if(node instanceof BooleanNode) return (BooleanNode)node; else return null;
	}
	/**
	 * Get the child node at the specified index, if it is a number node.
	 * This method gets the node at the specified index and tests if it is a number node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param index The index to get from
	 * @return The node at the specified index, if it is a number node; {@code null} otherwise
	 */
	public NumberNode getNumber(int index){
		Node node = get(index);
		if(node instanceof NumberNode) return (NumberNode)node; else return null;
	}
	/**
	 * Get the child node at the specified index, if it is an object node.
	 * This method gets the node at the specified index and tests if it is an object node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param index The index to get from
	 * @return The node at the specified index, if it is an object node; {@code null} otherwise
	 */
	public ObjectNode getObject(int index){
		Node node = get(index);
		if(node instanceof ObjectNode) return (ObjectNode)node; else return null;
	}
	/**
	 * Get the child node at the specified index, if it is a string node.
	 * This method gets the node at the specified index and tests if it is a string node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param index The index to get from
	 * @return The node at the specified index, if it is a string node; {@code null} otherwise
	 */
	public StringNode getString(int index){
		Node node = get(index);
		if(node instanceof StringNode) return (StringNode)node; else return null;
	}
	/**
	 * Get the index of the given node in this node's children.
	 * If the node isn't a child of this node, {@code -1} is returned.
	 * @param node The node to search for
	 * @return The index of the node, or -1 if not found
	 */
	public int indexOf(Node node){return children.indexOf(node);}
	@Override public boolean contains(Node node){return children.contains(node);}
	/**
	 * Remove the node at the specified index from this node's children.
	 * @param index The index to remove from
	 * @return The removed node
	 */
	public Node remove(int index){return children.remove(index);}
	@Override
	public boolean remove(Node node){
		if(children.remove(node)){
			node.parent = null;
			return true;
		}else return false;
	}
	@Override public void removeAll(){children.clear();}
	@Override public int size(){return children.size();}
	@Override public Iterator<Node> iterator(){return children.iterator();}
	/**
	 * Return a {@link ListIterator} over this node's children.
	 * @return this node's children's {@link ListIterator}
	 */
	public ListIterator<Node> listIterator(){return children.listIterator();}
	@Override 
	public boolean equals(Object obj){
		if(!(obj instanceof ArrayNode)) return false;
		ArrayNode node = (ArrayNode)obj;
		if(size() != node.size()) return false;
		for(Iterator<Node> i1=iterator(), i2=node.iterator(); i1.hasNext() && i2.hasNext();)//For-each over both nodes
			if(!i1.next().equals(i2.next())) return false;
		return true;
	}
	@Override
	public int hashCode(){
		int hashCode = 0;
		for(ListIterator<Node> i=children.listIterator(); i.hasNext();)
			if(i.nextIndex() % 2 == 1) hashCode += i.next().hashCode(); else hashCode -= i.next().hashCode();
		//Alternate adding and subtracting to add randomness 
		return hashCode;
	}
	@Override
	public String toString(){
		StringBuilder sb = new StringBuilder().append("[\r\n");
		for(Iterator<Node> i = iterator();i.hasNext();){
			sb.append(" ").append(i.next().toString().replace("\n", "\n "));
			if(i.hasNext()) sb.append(",\r\n");//If this isn't the last node, add a comma and a new line
		}
		return sb.append("\r\n]").toString();
	}
}
