package io.github.endreman0.javajson.nodes;

import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

/**
 * A JSON object node consists of a series of key-value pairs separated by commas, enclosed in curly brackets (<code>{}</code>).
 * The key to each pair must be a string enclosed in double quotes ({@code ""}), and the value can be any node. 
 * @author endreman0
 */
public class ObjectNode extends ParentNode implements Iterable<Map.Entry<String, Node>>{
	private Map<String, Node> children = new TreeMap<String, Node>();
	/**
	 * Create an empty object node.
	 */
	public ObjectNode(){super();}
	/**
	 * Create an object node with the given key-value pairs as children.
	 * @param children The string-node pairs to add as children
	 */
	public ObjectNode(Iterable<Map.Entry<String, Node>> children){this(); putAll(children);}
	/**
	 * Assign the given node to the value of the given key.
	 * If another node was already assigned to the given key, it will be removed from this node. 
	 * @param key The key to assign to...
	 * @param value ...this node
	 * @return {@code this}, for chaining
	 */
	public ObjectNode put(String key, Node value){
		children.put(key, value);
		value.parent = this;
		return this;
	}
	/**
	 * Add all of the given key-value pairs to this node's children.
	 * @param children The string-node pairs to add
	 * @return {@code this}, for chaining
	 */
	public ObjectNode putAll(Iterable<Map.Entry<String, Node>> children){
		for(Map.Entry<String, Node> child : children) put(child.getKey(), child.getValue());
		return this;
	}
	/**
	 * Get the node assigned to the provided key.
	 * @param key The key to search for
	 * @return The node assigned to that key, or {@code null} if no node was found
	 */
	public Node get(String key){return children.get(key);}
	/**
	 * Get the node assigned to the provided key, if it is an array node.
	 * This method gets the node assigned to the provided key and tests if it is an array node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param key The key to search for
	 * @return The node assigned to the specified key, if it is an array node; {@code null} otherwise
	 */
	public ArrayNode getArray(String key){
		Node node = get(key);
		if(node instanceof ArrayNode) return (ArrayNode)node; else return null;
	}
	/**
	 * Get the node assigned to the provided key, if it is a boolean node.
	 * This method gets the node assigned to the provided key and tests if it is a boolean node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param key The key to search for
	 * @return The node assigned to the specified key, if it is a boolean node; {@code null} otherwise
	 */
	public BooleanNode getBoolean(String key){
		Node node = get(key);
		if(node instanceof BooleanNode) return (BooleanNode)node; else return null;
	}
	/**
	 * Get the node assigned to the provided key, if it is a number node.
	 * This method gets the node assigned to the provided key and tests if it is a number node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param key The key to search for
	 * @return The node assigned to the specified key, if it is a number node; {@code null} otherwise
	 */
	public NumberNode getNumber(String key){
		Node node = get(key);
		if(node instanceof NumberNode) return (NumberNode)node; else return null;
	}
	/**
	 * Get the node assigned to the provided key, if it is an object node.
	 * This method gets the node assigned to the provided key and tests if it is an object node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param key The key to search for
	 * @return The node assigned to the specified key, if it is an object node; {@code null} otherwise
	 */
	public ObjectNode getObject(String key){
		Node node = get(key);
		if(node instanceof ObjectNode) return (ObjectNode)node; else return null;
	}
	/**
	 * Get the node assigned to the provided key, if it is a string node.
	 * This method gets the node assigned to the provided key and tests if it is a string node.
	 * If it is, the node is returned. If not, {@code null} is returned.
	 * @param key The key to search for
	 * @return The node assigned to the specified key, if it is a string node; {@code null} otherwise
	 */
	public StringNode getString(String key){
		Node node = get(key);
		if(node instanceof StringNode) return (StringNode)node; else return null;
	}
	/**
	 * Get the key that the given node is assigned to.
	 * If the node is not assigned to any key, {@code null} is returned.
	 * @param value The node to search for
	 * @return The key of the specified node, or {@code null} if not found
	 */
	public String keyOf(Node value){
		for(Map.Entry<String, Node> entry : children.entrySet())
			if(entry.getValue().equals(value)) return entry.getKey();
		return null;
	}
	@Override public int size(){return children.size();}
	/**
	 * Get whether any node is assigned to the given key.
	 * @param key The key to check for
	 * @return {@code true} if the key has an assigned value, {@code false} otherwise
	 */
	public boolean contains(String key){return children.containsKey(key);}
	@Override public boolean contains(Node value){return children.containsValue(value);}
	/**
	 * Remove the key-value pair with the given key.
	 * @param key The key to remove
	 * @return The removed pair's value
	 */
	public Node remove(String key){return children.remove(key);}
	@Override
	public boolean remove(Node value){
		for(Map.Entry<String, Node> entry : children.entrySet())
			if(entry.getValue().equals(value)){
				children.remove(entry.getKey());
				value.parent = null;
				return true;
			}
		return false;
	}
	@Override public void removeAll(){children.clear();}
	@Override public Iterator<Map.Entry<String, Node>> iterator(){return children.entrySet().iterator();}
	@Override 
	public boolean equals(Object obj){
		if(!(obj instanceof ObjectNode)) return false;
		ObjectNode node = (ObjectNode)obj;
		if(size() != node.size()) return false;
		for(Iterator<Map.Entry<String, Node>> i1=iterator(), i2=node.iterator(); i1.hasNext() && i2.hasNext();)//For-each over both nodes
			if(!i1.next().equals(i2.next())) return false;
		return true;
	}
	@Override
	public int hashCode(){
		int hashCode = 0;
		boolean add = true;
		for(Map.Entry<String, Node> entry : this)
			if(add = !add) hashCode += entry.hashCode(); else hashCode -= entry.hashCode();//Add or subtract, and toggle for the next iteration
		return hashCode;
	}
	@Override
	public String toString(){
		StringBuilder sb = new StringBuilder().append("{\r\n");
		for(Iterator<Map.Entry<String, Node>> i=iterator();i.hasNext();){
			Map.Entry<String, Node> entry = i.next();
			sb.append(String.format(" \"%s\" : %s", entry.getKey(), entry.getValue()).replace("\n", "\n "));
			if(i.hasNext()) sb.append(",\r\n");//If this isn't the last entry, add a comma and a new line
		}
		return sb.append("\r\n}").toString();
	}
}
