package ch.framedev.javamysqlutils;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;

import java.sql.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;


/**
 * This Class is in use for MySQL Database Stuff
 * Please only use this class if you can.
 */
@SuppressWarnings("unused")
public class MySQL {

    public static String host;
    public static String user;
    public static String password;
    public static String database;
    public static String port;
    private static JsonConnection jsonConnection;

    public static Connection connection;

    private static boolean allowPublicKey;

    private static HikariDataSource ds;

    private static final ExecutorService executor = Executors.newFixedThreadPool(10);

    public static interface Callback<T> {
        void onResult(T result);

        void onError(Throwable t);
    }

    public static void setJsonConnection(JsonConnection jsonConnection) {
        MySQL.jsonConnection = jsonConnection;
    }

    public JsonConnection getJsonConnection() {
        return jsonConnection;
    }

    public static class Builder {
        private String host;
        private String user;
        private String password;
        private int port;
        private String database;
        private boolean allowPublicKey = false;

        public Builder host(String host) {
            this.host = host;
            return this;
        }

        public Builder user(String user) {
            this.user = user;
            return this;
        }

        public Builder password(String password) {
            this.password = password;
            return this;
        }

        public Builder port(int port) {
            this.port = port;
            return this;
        }

        public Builder database(String database) {
            this.database = database;
            return this;
        }

        public Builder allowPublicKey(boolean allowPublicKey) {
            this.allowPublicKey = allowPublicKey;
            return this;
        }

        public MySQL build() {
            MySQL.setHost(host);
            MySQL.setUser(user);
            MySQL.setPassword(password);
            MySQL.setPort(String.valueOf(port));
            MySQL.setDatabase(database);
            MySQL.setAllowPublicKey(allowPublicKey);
            return new MySQL(host, user, password, port, database);
        }
    }

    public MySQL(JsonConnection jsonConnection) {
        MySQL.jsonConnection = jsonConnection;
        host = jsonConnection.getHost();
        user = jsonConnection.getUser();
        password = jsonConnection.getPassword();
        database = jsonConnection.getDatabase();
        port = String.valueOf(jsonConnection.getPort());
    }

    public MySQL(String host, String user, String password, int port, String database) {
        MySQL.host = host;
        MySQL.user = user;
        MySQL.password = password;
        MySQL.port = String.valueOf(port);
        MySQL.database = database;
    }

    public static void setAllowPublicKey(boolean allowPublicKey) {
        MySQL.allowPublicKey = allowPublicKey;
    }

    public static boolean isAllowPublicKey() {
        return allowPublicKey;
    }

    public static String getHost() {
        return host;
    }

    public static void setHost(String host) {
        MySQL.host = host;
    }

    public static String getUser() {
        return user;
    }

    public static void setUser(String user) {
        MySQL.user = user;
    }

    public static String getPassword() {
        return password;
    }

    public static void setPassword(String password) {
        MySQL.password = password;
    }

    public static String getDatabase() {
        return database;
    }

    public static void setDatabase(String database) {
        MySQL.database = database;
    }

    public static String getPort() {
        return port;
    }

    public static void setPort(String port) {
        MySQL.port = port;
    }

    public static void connect() {
        if (ds == null) {
            try {
                Class.forName("com.mysql.cj.jdbc.Driver");
                System.out.println("Driver loaded manually.");
            } catch (ClassNotFoundException e) {
                try {
                    Class.forName("com.mysql.jdbc.Driver");
                    System.out.println("Driver loaded manually.");
                } catch (ClassNotFoundException ex) {
                    throw new RuntimeException(ex);
                }
            }
            HikariConfig config = new HikariConfig();
            config.setJdbcUrl("jdbc:mysql://" + host + ":" + port + "/" + database);
            config.setUsername(user);
            config.setPassword(password);

// Optional but recommended:
            config.addDataSourceProperty("cachePrepStmts", "true");
            config.addDataSourceProperty("prepStmtCacheSize", "250");
            config.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
            config.addDataSourceProperty("useServerPrepStmts", "true");
            config.addDataSourceProperty("useSSL", "false");  // If you're not using SSL
            config.addDataSourceProperty("serverTimezone", "UTC");  // To avoid timezone issues

// Recommended for MariaDB newer versions:
            config.addDataSourceProperty("allowPublicKeyRetrieval", "true");  // If you're using password auth without SSL

            ds = new HikariDataSource(config);
        }
    }

    public static Connection getConnection() throws SQLException {
        if (ds == null) {
            connect();
        }
        return ds.getConnection();
    }

    public static void close() {
        if (ds != null && !ds.isClosed()) {
            ds.close();  // Properly close the pool
        }
    }

    public static boolean isTableExists(String table) {
        String sql = "SHOW TABLES LIKE ?";
        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {
            stmt.setString(1, table);
            try (ResultSet rs = stmt.executeQuery()) {
                return rs.next();
            }
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return false;
    }

    public static void isTableExistsAsync(String table, Callback<Boolean> callback) {
        executor.execute(() -> {
            String sql = "SHOW TABLES LIKE ?";
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {
                stmt.setString(1, table);
                try (ResultSet rs = stmt.executeQuery()) {
                    callback.onResult(rs.next());
                }
            } catch (SQLException ex) {
                callback.onError(ex);
            }
        });
    }

    public static boolean createTable(String tableName, String... columns) {
        throwErrorOnLength(columns == null || columns.length == 0, "At least one column must be specified.");

        String columnDefinitions = getColumnDefinitions(columns);

        String sql = "CREATE TABLE IF NOT EXISTS " + tableName + " ("
                + "ID INT PRIMARY KEY AUTO_INCREMENT, "
                + columnDefinitions
                + ", created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP);";
        try (PreparedStatement stmt = MySQL.getConnection().prepareStatement(sql)) {
            return stmt.executeUpdate() > 0;
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void createTableAsync(String tableName, Callback<Boolean> callback, String... columns) {
        throwErrorOnLength(columns == null || columns.length == 0, "At least one column must be specified.");

        String columnDefinitions = getColumnDefinitions(columns);

        String sql = "CREATE TABLE IF NOT EXISTS " + tableName + " ("
                + "ID INT PRIMARY KEY AUTO_INCREMENT, "
                + columnDefinitions
                + ", created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP);";

        executor.execute(new Runnable() {
            @Override
            public void run() {
                try (PreparedStatement stmt = MySQL.getConnection().prepareStatement(sql)) {
                    stmt.execute();  // execute() is used here instead of executeUpdate()
                    callback.onResult(true);
                } catch (SQLException e) {
                    callback.onError(e);
                }
            }
        });
    }

    private static String getColumnDefinitions(String[] columns) {
        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            stringBuilder.append(columns[i]);
            if (i < columns.length - 1) {
                stringBuilder.append(", ");
            }
        }
        return stringBuilder.toString();
    }

    public static boolean insertData(String table, Object[] data, String[] columns) {
        throwErrorOnLength(columns == null || columns.length == 0 || data == null || data.length == 0, "Columns and data must be provided.");

        throwErrorOnLength(columns.length != data.length, "Number of columns and data values must match.");

        String sql = createInsertSQL(table, columns);

        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            for (int i = 0; i < data.length; i++) {
                stmt.setObject(i + 1, data[i]);
            }

            return stmt.executeUpdate() > 0;
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void insertDataAsync(String table, Object[] data, String[] columns, Callback<Boolean> callback) {
        throwErrorOnLength(columns == null || columns.length == 0 || data == null || data.length == 0, "Columns and data must be provided.");

        throwErrorOnLength(columns.length != data.length, "Number of columns and data values must match.");

        String sql = createInsertSQL(table, columns);

        executor.execute(() -> {
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                for (int i = 0; i < data.length; i++) {
                    stmt.setObject(i + 1, data[i]);
                }

                callback.onResult(stmt.executeUpdate() > 0);
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    private static String createInsertSQL(String table, String[] columns) {
        StringBuilder columnsBuilder = new StringBuilder();
        StringBuilder placeholdersBuilder = new StringBuilder();

        for (int i = 0; i < columns.length; i++) {
            columnsBuilder.append(columns[i]);
            placeholdersBuilder.append("?");
            if (i < columns.length - 1) {
                columnsBuilder.append(", ");
                placeholdersBuilder.append(", ");
            }
        }

        return "INSERT INTO " + table +
                " (" + columnsBuilder + ") VALUES (" + placeholdersBuilder + ")";
    }

    private static void throwErrorOnLength(boolean length, String s) {
        if (length) {
            throw new IllegalArgumentException(s);
        }
    }

    public static boolean updateData(String table, String column, Object value, String whereClause) {
        String sql = "UPDATE " + table + " SET " + column + " = ? WHERE " + whereClause;

        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            stmt.setObject(1, value);

            int result = stmt.executeUpdate();
            return result > 0;
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void updateDataAsync(String table, String column, Object value, String whereClause, Callback<Boolean> callback) {
        String sql = "UPDATE " + table + " SET " + column + " = ? WHERE " + whereClause;

        executor.execute(() -> {
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                stmt.setObject(1, value);

                int result = stmt.executeUpdate();
                callback.onResult(result > 0);
            } catch (SQLException e) {
                callback.onError(e);
            }

        });
    }

    public static boolean updateData(String table, String[] columns, Object[] values, String where) {
        if (columns.length == 0 || values.length == 0 || columns.length != values.length) {
            throw new IllegalArgumentException("Columns and values arrays must not be empty and must have the same length.");
        }

        // Construct the SET clause dynamically
        StringBuilder setClause = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            setClause.append(columns[i]).append(" = ?");
            if (i < columns.length - 1) {
                setClause.append(", ");
            }
        }

        String sql = "UPDATE " + table + " SET " + setClause.toString() + " WHERE " + where;
        try (Connection conn = getConnection();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Set the values for the placeholders
            for (int i = 0; i < values.length; i++) {
                pstmt.setObject(i + 1, values[i]);
            }

            // Execute the update
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void updateDataAsync(String table, String[] columns, Object[] values, String where, Callback<Boolean> callback) {
        if (columns.length == 0 || values.length == 0 || columns.length != values.length) {
            throw new IllegalArgumentException("Columns and values arrays must not be empty and must have the same length.");
        }

        // Construct the SET clause dynamically
        StringBuilder setClause = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            setClause.append(columns[i]).append(" = ?");
            if (i < columns.length - 1) {
                setClause.append(", ");
            }
        }

        String sql = "UPDATE " + table + " SET " + setClause.toString() + " WHERE " + where;

        executor.execute(() -> {
                    try (Connection conn = getConnection();
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Set the values for the placeholders
                        for (int i = 0; i < values.length; i++) {
                            pstmt.setObject(i + 1, values[i]);
                        }

                        // Execute the update
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
        });
    }

    public static boolean deleteDataInTable(String table, String whereClause) {
        String sql = "DELETE FROM " + table + " WHERE " + whereClause;

        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            int result = stmt.executeUpdate();
            return result > 0;
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void deleteDataInTableAsync(String table, String whereClause, Callback<Boolean> callback) {
        String sql = "DELETE FROM " + table + " WHERE " + whereClause;

        executor.execute(() -> {
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                int result = stmt.executeUpdate();
                callback.onResult(result > 0);
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    public static boolean deleteDataInTable(String table, String whereColumn, String whereValue, String andColumn, String andValue) {
        String sql = "DELETE FROM " + table + " WHERE " + whereColumn + " = ? AND " + andColumn + " = ?";

        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            stmt.setString(1, whereValue);
            stmt.setString(2, andValue);

            int result = stmt.executeUpdate();
            return result > 0;
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void deleteDataInTableAsync(String table, String whereColumn, String whereValue, String andColumn, String andValue, Callback<Boolean> callback) {
        String sql = "DELETE FROM " + table + " WHERE " + whereColumn + " = ? AND " + andColumn + " = ?";

        executor.execute(() -> {

            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                stmt.setString(1, whereValue);
                stmt.setString(2, andValue);

                int result = stmt.executeUpdate();
                callback.onResult(result > 0);
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    public static boolean exists(String table, String column, String data) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ?";

        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            stmt.setString(1, data);

            try (ResultSet res = stmt.executeQuery()) {
                return res.next();
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void existsAsync(String table, String column, String data, Callback<Boolean> callback) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                stmt.setString(1, data);

                try (ResultSet res = stmt.executeQuery()) {
                    callback.onResult(res.next());
                }
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    public static boolean exists(String table, String column, String data, String andCondition) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ? AND " + andCondition;

        try (Connection conn = MySQL.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            stmt.setString(1, data);

            try (ResultSet res = stmt.executeQuery()) {
                return res.next();
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void existsAsync(String table, String column, String data, String andCondition, Callback<Boolean> callback) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ? AND " + andCondition;

        executor.execute(() -> {
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                stmt.setString(1, data);

                try (ResultSet res = stmt.executeQuery()) {
                    callback.onResult(res.next());
                }
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    public static void getAsync(String table, String selected, String column, String data, Callback<Object> callback) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                stmt.setString(1, data);

                try (ResultSet res = stmt.executeQuery()) {
                    if (res.next()) {
                        callback.onResult(res.getObject(selected));
                    }
                }
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    public static Object get(String table, String selected, String column, String data) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";

            try (Connection conn = MySQL.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(sql)) {

                stmt.setString(1, data);

                try (ResultSet res = stmt.executeQuery()) {
                    if (res.next()) {
                        return res.getObject(selected);
                    }
                }
            } catch (SQLException e) {
                e.printStackTrace();
            }
            return null;
    }

    public static List<Object> get(String table, String[] selectedColumns, String column, Object data, String andColumn, Object andValue) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ? AND " + andColumn + " = ?";
        List<Object> results = new ArrayList<>();

        try (Connection conn = getConnection();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);
            pstmt.setObject(2, andValue);

            try (ResultSet res = pstmt.executeQuery()) {
                while (res.next()) {
                    Object row = null;
                    for (String selectedColumn : selectedColumns) {
                        row = res.getObject(selectedColumn); // Retrieve each column value
                        results.add(row);
                    }
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }

        return results;
    }

    public static void getAsync(String table, String[] selectedColumns, String column, Object data, String andColumn, Object andValue, Callback<List<Object>> callback) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ? AND " + andColumn + " = ?";
        List<Object> results = new ArrayList<>();

        executor.execute(() -> {
            try (Connection conn = getConnection();
                 PreparedStatement pstmt = conn.prepareStatement(sql)) {

                pstmt.setObject(1, data);
                pstmt.setObject(2, andValue);

                try (ResultSet res = pstmt.executeQuery()) {
                    while (res.next()) {
                        Object row = null;
                        for (String selectedColumn : selectedColumns) {
                            row = res.getObject(selectedColumn); // Retrieve each column value
                            results.add(row);
                        }
                    }
                    callback.onResult(results);
                }
            } catch (SQLException e) {
                callback.onError(e);
            }
        });
    }

    public static List<Object> get(String table, String[] selectedColumns, String column, Object data) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ?";
        List<Object> results = new ArrayList<>();

        try (Connection conn = getConnection();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                while (res.next()) {
                    Object row = null;
                    for (String selectedColumn : selectedColumns) {
                        row = res.getObject(selectedColumn); // Retrieve each column value
                        results.add(row);
                    }
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }

        return results;
    }

    public static void getAsync(String table, String[] selectedColumns, String column, Object data, Callback<List<Object>> callback) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ?";
        List<Object> results = new ArrayList<>();

        executor.execute(() -> {
                    try (Connection conn = getConnection();
                         PreparedStatement pstmt = conn.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            while (res.next()) {
                                Object row = null;
                                for (String selectedColumn : selectedColumns) {
                                    row = res.getObject(selectedColumn); // Retrieve each column value
                                    results.add(row);
                                }
                            }
                            callback.onResult(results);
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                });
    }

    public static void addColumn(String table, String column, Callback<Boolean> callback) {
        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (result) {
                    String sql = "ALTER TABLE " + table + " ADD COLUMN " + column;
                    try (Connection conn = MySQL.getConnection();
                         Statement stmt = conn.createStatement()) {
                        stmt.executeUpdate(sql);
                        callback.onResult(true);
                    } catch (SQLException ex) {
                        callback.onError(ex);
                    }
                }
            }

            @Override
            public void onError(Throwable t) {
                callback.onError(t);
            }
        });
    }

    public static void removeColumn(String table, String column, Callback<Boolean> callback) {
        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (result) {
                    String sql = "ALTER TABLE " + table + " DROP COLUMN " + column;
                    try (Connection conn = MySQL.getConnection();
                         Statement stmt = conn.createStatement()) {
                        stmt.executeUpdate(sql);
                        callback.onResult(true);
                    } catch (SQLException ex) {
                        callback.onError(ex);
                    }
                }
            }

            @Override
            public void onError(Throwable t) {
                callback.onError(t);
            }
        });
    }

    public static void showColumns(String table, Callback<List<String>> callback) {
        List<String> columns = new ArrayList<>();
        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (result) {
                    String sql = "SHOW COLUMNS FROM " + table;
                    try (Connection conn = MySQL.getConnection();
                         Statement stmt = conn.createStatement();
                         ResultSet resultSet = stmt.executeQuery(sql)) {
                        while (resultSet.next()) {
                            columns.add(resultSet.getString("Field"));
                        }
                        callback.onResult(columns);
                    } catch (SQLException ex) {
                        callback.onError(ex);
                    }
                }
            }

            @Override
            public void onError(Throwable t) {
                callback.onError(t);
            }
        });
    }

}

