package ch.framedev.javasqliteutils;

import java.io.File;
import java.sql.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This Plugin was Created by FrameDev
 * Package : mysql
 * Date: 07.03.21
 * Project: untitled
 * Copyrighted by FrameDev
 */
@SuppressWarnings({"unused", "ResultOfMethodCallIgnored"})
public class SQLite {
    private static final Logger LOGGER = Logger.getLogger(SQLite.class.getName());

    public static Connection connection;
    private static String fileName;
    private static String path;

    private static final ExecutorService executor = Executors.newFixedThreadPool(10);

    public SQLite(String path, String fileName) {
        if (!new File(path).exists())
            new File(path).mkdirs();
        SQLite.fileName = fileName;
        SQLite.path = path;
    }

    public SQLite(JsonConnection connection) {
        SQLite.fileName = connection.getFileName();
        SQLite.path = connection.getPath();
        if (!new File(SQLite.path).exists())
            new File(SQLite.path).mkdirs();
    }

    public static void connectAsync(Callback<Connection> callback) {
        executor.execute(() -> {
            try {
                if (connection != null && !connection.isClosed()) {
                    callback.onResult(connection);
                }
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
            try {
                Class.forName("org.sqlite.JDBC");
                String url = "jdbc:sqlite:" + path + "/" + fileName;
                connection = DriverManager.getConnection(url);
                callback.onResult(connection);
            } catch (SQLException | ClassNotFoundException e) {
                callback.onError(e);
            }
        });
    }

    public static Connection connect() {
        try {
            if (connection != null && !connection.isClosed()) {
                return connection;
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        try {
            Class.forName("org.sqlite.JDBC");
            String url = "jdbc:sqlite:" + path + "/" + fileName;
            connection = DriverManager.getConnection(url);
            return connection;
        } catch (SQLException | ClassNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static void close() {
        if (connection != null) {
            try {
                if (!connection.isClosed()) {
                    connection.close();
                    connection = null; // Ensure the connection reference is nullified after closing
                }
            } catch (SQLException ex) {
                // Consider logging the exception or handling it in a way that is more robust than just printing the stack trace
                ex.printStackTrace();
            }
        }
    }

    public static boolean isTableExists(String table) {
        String sql = "SELECT name FROM sqlite_master WHERE type = 'table' AND name = ? AND name NOT LIKE 'sqlite_%';";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, table);
            try (ResultSet rs = pstmt.executeQuery()) {
                return rs.next(); // Returns true if the table exists, false otherwise
            }

        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return false;
    }

    public static void isTableExistsAsync(String table, Callback<Boolean> callback) {
        String sql = "SELECT name FROM sqlite_master WHERE type = 'table' AND name = ? AND name NOT LIKE 'sqlite_%';";

        executor.execute(() -> connectAsync(new Callback<Connection>() {
            @Override
            public void onResult(Connection result) {
                try (Connection conn = result;
                     PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                    pstmt.setString(1, table);
                    try (ResultSet rs = pstmt.executeQuery()) {
                        callback.onResult(rs.next()); // Returns true if the table exists, false otherwise
                    }

                } catch (SQLException ex) {
                    callback.onError(ex);
                }
            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        }));
    }

    public static boolean createTable(String tableName, boolean date, String[] columns) {
        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            stringBuilder.append(columns[i]);
            if (i < columns.length - 1) {
                stringBuilder.append(",");
            }
        }

        String sql;
        if (date) {
            sql = "CREATE TABLE IF NOT EXISTS " + tableName +
                    " (ID INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    stringBuilder.toString() +
                    ", created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP);";
        } else {
            sql = "CREATE TABLE IF NOT EXISTS " + tableName +
                    " (ID INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    stringBuilder.toString() +
                    ");";
        }
        try (Connection conn = connect();
             Statement stmt = conn.createStatement()) {
            return stmt.execute(sql);
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * Erstelle einen Table mit einem Table Name und verschiedene Column
     *
     * @param tableName TableName der erstellt wird
     * @param columns   Kolumm die erstellt werden
     */
    public static void createTableAsync(String tableName, boolean date, String[] columns, Callback<Boolean> callback) {
        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            stringBuilder.append(columns[i]);
            if (i < columns.length - 1) {
                stringBuilder.append(",");
            }
        }

        String sql;
        if (date) {
            sql = "CREATE TABLE IF NOT EXISTS " + tableName +
                    " (ID INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    stringBuilder.toString() +
                    ", created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP);";
        } else {
            sql = "CREATE TABLE IF NOT EXISTS " + tableName +
                    " (ID INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    stringBuilder.toString() +
                    ");";
        }

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         Statement stmt = conn.createStatement()) {
                        stmt.execute(sql);
                        callback.onResult(true);
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean insertData(String table, Object[] data, String[] columns) {
        if (columns.length == 0 || data.length == 0 || columns.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Join the column names with commas
        String columnNames = String.join(",", columns);

        // Create a string of placeholders (e.g., "?, ?, ?")
        String placeholders = String.join(",", Collections.nCopies(columns.length, "?"));

        // Construct the SQL insert statement
        String sql = "INSERT INTO " + table + " (" + columnNames + ") VALUES (" + placeholders + ")";
        try (Connection conn = connect();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            // Set the values for each placeholder
            for (int i = 0; i < data.length; i++) {
                stmt.setObject(i + 1, data[i]);
            }

            // Execute the statement
            return stmt.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void insertDataAsync(String table, Object[] data, String[] columns, Callback<Boolean> callback) {
        if (columns.length == 0 || data.length == 0 || columns.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Join the column names with commas
        String columnNames = String.join(",", columns);

        // Create a string of placeholders (e.g., "?, ?, ?")
        String placeholders = String.join(",", Collections.nCopies(columns.length, "?"));

        // Construct the SQL insert statement
        String sql = "INSERT INTO " + table + " (" + columnNames + ") VALUES (" + placeholders + ")";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement stmt = conn.prepareStatement(sql)) {

                        // Set the values for each placeholder
                        for (int i = 0; i < data.length; i++) {
                            stmt.setObject(i + 1, data[i]);
                        }

                        // Execute the statement
                        stmt.executeUpdate();
                        callback.onResult(true);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean updateData(String table, String selected, Object data, String where) {
        String sql = "UPDATE " + table + " SET " + selected + " = ? WHERE " + where;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Set the value for the placeholder
            pstmt.setObject(1, data);

            // Execute the update
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void updateDataAsync(String table, String selected, Object data, String where, Callback<Boolean> callback) {
        String sql = "UPDATE " + table + " SET " + selected + " = ? WHERE " + where;

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Set the value for the placeholder
                        pstmt.setObject(1, data);

                        // Execute the update
                        pstmt.executeUpdate();
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean updateData(String table, String[] columns, Object[] values, String where) {
        if (columns.length == 0 || values.length == 0 || columns.length != values.length) {
            throw new IllegalArgumentException("Columns and values arrays must not be empty and must have the same length.");
        }

        // Construct the SET clause dynamically
        StringBuilder setClause = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            setClause.append(columns[i]).append(" = ?");
            if (i < columns.length - 1) {
                setClause.append(", ");
            }
        }

        String sql = "UPDATE " + table + " SET " + setClause.toString() + " WHERE " + where;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Set the values for the placeholders
            for (int i = 0; i < values.length; i++) {
                pstmt.setObject(i + 1, values[i]);
            }

            // Execute the update
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void updateDataAsync(String table, String[] columns, Object[] values, String where, Callback<Boolean> callback) {
        if (columns.length == 0 || values.length == 0 || columns.length != values.length) {
            throw new IllegalArgumentException("Columns and values arrays must not be empty and must have the same length.");
        }

        // Construct the SET clause dynamically
        StringBuilder setClause = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            setClause.append(columns[i]).append(" = ?");
            if (i < columns.length - 1) {
                setClause.append(", ");
            }
        }

        String sql = "UPDATE " + table + " SET " + setClause.toString() + " WHERE " + where;

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Set the values for the placeholders
                        for (int i = 0; i < values.length; i++) {
                            pstmt.setObject(i + 1, values[i]);
                        }

                        // Execute the update
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean deleteDataInTable(String table, String where) {
        String sql = "DELETE FROM " + table + " WHERE " + where;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Execute the delete
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void deleteDataInTableAsync(String table, String where, Callback<Boolean> callback) {
        String sql = "DELETE FROM " + table + " WHERE " + where;

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Execute the delete
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean deleteDataInTable(String table, String whereColumn, String whereValue, String andColumn, String andValue) {
        String sql = "DELETE FROM " + table + " WHERE " + whereColumn + " = ? AND " + andColumn + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, whereValue);
            pstmt.setString(2, andValue);

            // Execute the delete
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void deleteDataInTableAsync(String table, String whereColumn, String whereValue, String andColumn, String andValue, Callback<Boolean> callback) {
        String sql = "DELETE FROM " + table + " WHERE " + whereColumn + " = ? AND " + andColumn + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        pstmt.setString(1, whereValue);
                        pstmt.setString(2, andValue);

                        // Execute the delete
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean exists(String table, String column, String data) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, data);
            try (ResultSet res = pstmt.executeQuery()) {
                return res.next(); // Returns true if the record exists
            }

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void existsAsync(String table, String column, String data, Callback<Boolean> callback) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        pstmt.setString(1, data);
                        try (ResultSet res = pstmt.executeQuery()) {
                            callback.onResult(res.next()); // Returns true if the record exists
                        }

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean exists(String table, String column, String data, String andColumn, String andValue) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ? AND " + andColumn + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, data);
            pstmt.setString(2, andValue);

            try (ResultSet res = pstmt.executeQuery()) {
                return res.next(); // Returns true if the record exists
            }

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void existsAsync(String table, String column, String data, String andColumn, String andValue, Callback<Boolean> callback) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ? AND " + andColumn + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        pstmt.setString(1, data);
                        pstmt.setString(2, andValue);

                        try (ResultSet res = pstmt.executeQuery()) {
                            callback.onResult(res.next()); // Returns true if the record exists
                        }

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static Object get(String table, String selected, String column, Object data) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    return res.getObject(selected);
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static void getAsync(String table, String selected, String column, Object data, Callback<Object> callback) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    Connection conn = result;
                    try (PreparedStatement pstmt = conn.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            if (res.next()) {
                                callback.onResult(res.getObject(selected));
                            }
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static List<Object> get(String table, String[] selectedColumns, String column, Object data) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ?";
        List<Object> results = new ArrayList<>();

        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                while (res.next()) {
                    Object row = null;
                    for (String selectedColumn : selectedColumns) {
                        row = res.getObject(selectedColumn); // Retrieve each column value
                        results.add(row);
                    }
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }

        return results;
    }

    public static void getAsync(String table, String[] selectedColumns, String column, Object data, Callback<List<Object>> callback) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ?";
        List<Object> results = new ArrayList<>();

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    Connection conn = result;
                    try (PreparedStatement pstmt = conn.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            while (res.next()) {
                                Object row = null;
                                for (String selectedColumn : selectedColumns) {
                                    row = res.getObject(selectedColumn); // Retrieve each column value
                                    results.add(row);
                                }
                            }
                            callback.onResult(results);
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static <T> T get(String table, String selected, String column, Object data, Class<T> type) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    Object value = res.getObject(selected);
                    if (value != null && type.isInstance(value)) {
                        return type.cast(value);
                    } else {
                        LOGGER.warning("Type mismatch: Expected " + type.getSimpleName() + " but got " +
                                (value != null ? value.getClass().getSimpleName() : "null"));
                    }
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static <T> void getAsync(String table, String selected, String column, Object data, Class<T> type, Callback<T> callback) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    Connection conn = result;
                    try (PreparedStatement pstmt = conn.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            if (res.next()) {
                                Object value = res.getObject(selected);
                                if (value != null && type.isInstance(value)) {
                                    callback.onResult(type.cast(value));
                                } else {
                                    LOGGER.warning("Type mismatch: Expected " + type.getSimpleName() + " but got " +
                                            (value != null ? value.getClass().getSimpleName() : "null"));
                                }
                            }
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    public static boolean addColumn(String table, String column) {
        String sql = "ALTER TABLE " + table + " ADD COLUMN " + column;
        if (!isTableExists(table))
            return false;
        try (Connection conn = connect();
             PreparedStatement statement = Objects.requireNonNull(conn).prepareStatement(sql)) {
            return statement.executeUpdate() > 0;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return false;
    }

    public static void addColumnAsync(String table, String column, Callback<Boolean> callback) {
        String sql = "ALTER TABLE " + table + " ADD COLUMN " + column;

        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (!result) {
                    callback.onResult(false);
                    return;
                }
                connectAsync(new Callback<Connection>() {
                    @Override
                    public void onResult(Connection result) {
                        try (Connection conn = result;
                             PreparedStatement statement = Objects.requireNonNull(conn).prepareStatement(sql)) {

                            callback.onResult(statement.executeUpdate() > 0);

                        } catch (SQLException e) {
                            callback.onError(e);
                        }
                    }

                    @Override
                    public void onError(Exception e) {
                        callback.onError(e);
                    }
                });
            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        });
    }

    public static boolean removeColumn(String table, String column) {
        if (!isTableExists(table))
            return false;
        try (Connection conn = connect()) {
            conn.setAutoCommit(false);

            // Step 1: Get the table's current structure
            String getColumnsSql = "PRAGMA table_info(" + table + ")";
            StringBuilder columns = new StringBuilder();
            try (PreparedStatement getColumnsStmt = conn.prepareStatement(getColumnsSql);
                 ResultSet rs = getColumnsStmt.executeQuery()) {

                while (rs.next()) {
                    String colName = rs.getString("name");
                    if (!colName.equalsIgnoreCase(column)) {
                        if (columns.length() > 0) {
                            columns.append(", ");
                        }
                        columns.append(colName);
                    }
                }
                if (columns.length() == 0) {
                    return false;
                }
                String columnsStr = columns.toString();

                // Step 2: Rename the old table
                String renameSql = "ALTER TABLE " + table + " RENAME TO " + table + "_old";
                try (PreparedStatement renameStmt = conn.prepareStatement(renameSql)) {
                    renameStmt.executeUpdate();
                }

                // Step 3: Create a new table without the column to be removed
                String createSql = "CREATE TABLE " + table + " AS SELECT " + columnsStr + " FROM " + table + "_old";
                try (PreparedStatement createStmt = conn.prepareStatement(createSql)) {
                    createStmt.executeUpdate();
                }

                // Step 4: Drop the old table
                String dropSql = "DROP TABLE " + table + "_old";
                try (PreparedStatement dropStmt = conn.prepareStatement(dropSql)) {
                    dropStmt.executeUpdate();
                }

                conn.commit();

            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return true;
    }

    public static void removeColumnAsync(String table, String column, Callback<Boolean> callback) {

        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (!result) {
                    callback.onResult(false);
                    return;
                }

                connectAsync(new Callback<Connection>() {
                    @Override
                    public void onResult(Connection result) {
                        try (Connection conn = result) {
                            conn.setAutoCommit(false);

                            // Step 1: Get the table's current structure
                            String getColumnsSql = "PRAGMA table_info(" + table + ")";
                            StringBuilder columns = new StringBuilder();
                            try (PreparedStatement getColumnsStmt = conn.prepareStatement(getColumnsSql);
                                 ResultSet rs = getColumnsStmt.executeQuery()) {

                                while (rs.next()) {
                                    String colName = rs.getString("name");
                                    if (!colName.equalsIgnoreCase(column)) {
                                        if (columns.length() > 0) {
                                            columns.append(", ");
                                        }
                                        columns.append(colName);
                                    }
                                }
                                if (columns.length() == 0) {
                                    callback.onResult(false);
                                    return;
                                }
                                String columnsStr = columns.toString();

                                // Step 2: Rename the old table
                                String renameSql = "ALTER TABLE " + table + " RENAME TO " + table + "_old";
                                try (PreparedStatement renameStmt = conn.prepareStatement(renameSql)) {
                                    renameStmt.executeUpdate();
                                }

                                // Step 3: Create a new table without the column to be removed
                                String createSql = "CREATE TABLE " + table + " AS SELECT " + columnsStr + " FROM " + table + "_old";
                                try (PreparedStatement createStmt = conn.prepareStatement(createSql)) {
                                    createStmt.executeUpdate();
                                }

                                // Step 4: Drop the old table
                                String dropSql = "DROP TABLE " + table + "_old";
                                try (PreparedStatement dropStmt = conn.prepareStatement(dropSql)) {
                                    dropStmt.executeUpdate();
                                }

                                conn.commit();

                            }
                        } catch (SQLException e) {
                            callback.onError(e);
                        }
                    }

                    @Override
                    public void onError(Exception e) {
                        callback.onError(e);
                    }
                });

            }

            @Override
            public void onError(Exception e) {

            }
        });
    }

    public static List<String> showColumns(String table) {
        List<String> columns = new ArrayList<>();
        String sql = "SELECT * FROM " + table;
        try (Connection conn = connect();
             Statement statement = Objects.requireNonNull(conn).createStatement();
             ResultSet rs = statement.executeQuery(sql)) {

            ResultSetMetaData mrs = rs.getMetaData();
            for (int i = 1; i <= mrs.getColumnCount(); i++) {
                columns.add(mrs.getColumnLabel(i));
            }
            return columns;

        } catch (SQLException e) {
            e.printStackTrace();
        }
        return columns;
    }

    public static void showColumnsAsync(String table, Callback<List<String>> callback) {
        List<String> columns = new ArrayList<>();
        String sql = "SELECT * FROM " + table;

        connectAsync(new Callback<Connection>() {
            @Override
            public void onResult(Connection result) {
                executor.execute(() -> {
                    try (Connection conn = result;
                         Statement statement = Objects.requireNonNull(conn).createStatement();
                         ResultSet rs = statement.executeQuery(sql)) {

                        ResultSetMetaData mrs = rs.getMetaData();
                        for (int i = 1; i <= mrs.getColumnCount(); i++) {
                            columns.add(mrs.getColumnLabel(i));
                        }
                        callback.onResult(columns);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                });
            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        });
    }
}
