

package com.wechat.pay.java.service.transferbills;

import com.wechat.pay.java.core.Config;
import com.wechat.pay.java.core.cipher.PrivacyDecryptor;
import com.wechat.pay.java.core.cipher.PrivacyEncryptor;
import com.wechat.pay.java.core.exception.HttpException;
import com.wechat.pay.java.core.exception.MalformedMessageException;
import com.wechat.pay.java.core.exception.ServiceException;
import com.wechat.pay.java.core.exception.ValidationException;
import com.wechat.pay.java.core.http.*;
import com.wechat.pay.java.core.util.GsonUtil;
import com.wechat.pay.java.service.transferbills.model.InitiateBillsTransferRequest;
import com.wechat.pay.java.service.transferbills.model.InitiateBillsTransferResponse;

import java.util.Objects;

public class TransferBillsService {
    private final HttpClient httpClient;
    private final HostName hostName;
    private final PrivacyEncryptor encryptor;
    private final PrivacyDecryptor decryptor;

    private TransferBillsService(
            HttpClient httpClient,
            HostName hostName,
            PrivacyEncryptor encryptor,
            PrivacyDecryptor decryptor) {
        this.httpClient = Objects.requireNonNull(httpClient);
        this.hostName = hostName;
        this.encryptor = Objects.requireNonNull(encryptor);
        this.decryptor = Objects.requireNonNull(decryptor);
    }

    /**
     * TransferBillsService构造器
     */
    public static class Builder {

        private HttpClient httpClient;
        private HostName hostName;
        private PrivacyEncryptor encryptor;
        private PrivacyDecryptor decryptor;

        public TransferBillsService.Builder config(Config config) {
            this.httpClient = new DefaultHttpClientBuilder().config(config).build();
            this.encryptor = config.createEncryptor();
            this.decryptor = config.createDecryptor();
            return this;
        }

        public TransferBillsService.Builder hostName(HostName hostName) {
            this.hostName = hostName;
            return this;
        }

        public TransferBillsService.Builder httpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }

        public TransferBillsService.Builder encryptor(PrivacyEncryptor encryptor) {
            this.encryptor = encryptor;
            return this;
        }

        public TransferBillsService.Builder decryptor(PrivacyDecryptor decryptor) {
            this.decryptor = decryptor;
            return this;
        }

        public TransferBillsService build() {
            return new TransferBillsService(httpClient, hostName, encryptor, decryptor);
        }
    }

    /**
     * 发起商家转账
     *
     * @param request 请求参数
     * @return InitiateBillsTransferResponse
     * @throws HttpException             发送HTTP请求失败。例如构建请求参数失败、发送请求失败、I/O错误等。包含请求信息。
     * @throws ValidationException       发送HTTP请求成功，验证微信支付返回签名失败。
     * @throws ServiceException          发送HTTP请求成功，服务返回异常。例如返回状态码小于200或大于等于300。
     * @throws MalformedMessageException 服务返回成功，content-type不为application/json、解析返回体失败。
     */
    public InitiateBillsTransferResponse initiateBillsTransfer(InitiateBillsTransferRequest request) {
        String requestPath = "https://api.mch.weixin.qq.com/v3/fund-app/mch-transfer/transfer-bills";
        // 加密敏感信息
        InitiateBillsTransferRequest realRequest = request.cloneWithCipher(encryptor::encrypt);
        if (this.hostName != null) {
            requestPath = requestPath.replaceFirst(HostName.API.getValue(), hostName.getValue());
        }
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.WECHAT_PAY_SERIAL, encryptor.getWechatpaySerial());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.POST)
                        .url(requestPath)
                        .headers(headers)
                        .body(createRequestBody(realRequest))
                        .build();
        HttpResponse<InitiateBillsTransferResponse> httpResponse =
                httpClient.execute(httpRequest, InitiateBillsTransferResponse.class);
        return httpResponse.getServiceResponse();
    }


    private RequestBody createRequestBody(Object request) {
        return (new JsonRequestBody.Builder()).body(GsonUtil.toJson(request)).build();
    }

}