package com.rosan.app_process;

import android.app.ActivityThread;
import android.content.ComponentName;
import android.content.Context;
import android.os.Binder;
import android.os.IBinder;
import android.os.Parcel;
import android.os.RemoteException;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;

public abstract class AppProcess implements Closeable {
    protected Context mContext = null;
    protected IProcessManager mManager = null;

    // 优化 1: 使用 ConcurrentHashMap 确保线程安全，防止多线程同时 isolatedServiceBinder 时崩溃
    protected final Map<String, IBinder> mChildProcess = new ConcurrentHashMap<>();

    // 优化 2: 提取常量，方便后续维护
    private static final String APP_PROCESS_PATH = "/system/bin/app_process";
    private static final String SYSTEM_BIN = "/system/bin";

    public static ProcessParams generateProcessParams(@NonNull String classPath, @NonNull String entryClassName, @NonNull List<String> args) {
        List<String> cmdList = new ArrayList<>();
        cmdList.add(APP_PROCESS_PATH);
        cmdList.add("-Djava.class.path=" + classPath);
        cmdList.add(SYSTEM_BIN);
        cmdList.add(entryClassName);
        cmdList.addAll(args);
        return new ProcessParams(cmdList, null, null);
    }

    // 这里的泛型方法保留，对 Kotlin 非常友好
    public static <T> T binderWithCleanCallingIdentity(Callable<T> action) throws Exception {
        final long callingIdentity = Binder.clearCallingIdentity();
        try {
            return action.call();
        } finally {
            Binder.restoreCallingIdentity(callingIdentity);
        }
    }

    public static IBinder binderWrapper(IProcessManager manager, IBinder binder) {
        return new BinderWrapper(manager, binder);
    }

    public static boolean remoteTransact(IProcessManager manager, IBinder binder, int code, Parcel data, Parcel reply, int flags) {
        IBinder managerBinder = manager.asBinder();
        Parcel processData = Parcel.obtain();
        try {
            // 优化 3: 增加 descriptor 为空的容错
            String descriptor = managerBinder.getInterfaceDescriptor();
            processData.writeInterfaceToken(descriptor != null ? descriptor : "");
            processData.writeStrongBinder(binder);
            processData.writeInt(code);
            processData.writeInt(flags);
            processData.appendFrom(data, 0, data.dataSize());
            // 使用 FIRST_CALL_TRANSACTION + 2 记得在 AIDL 中对应好
            return managerBinder.transact(IBinder.FIRST_CALL_TRANSACTION + 2, processData, reply, 0);
        } catch (RemoteException e) {
            throw new RuntimeException(e);
        } finally {
            processData.recycle();
        }
    }

    public static void linkDeathToThis(IProcessManager serviceManager) {
        linkDeathTo(serviceManager, new ClientManager().asBinder());
    }

    public static void linkDeathTo(IProcessManager serviceManager, @Nullable IBinder binder) {
        try {
            ParcelableBinder pBinder = binder == null ? null : new ParcelableBinder(binder);
            serviceManager.linkDeathTo(pBinder);
        } catch (RemoteException e) {
            throw new RuntimeException(e);
        }
    }

    public @NonNull Process start(@NonNull String classPath, @NonNull String entryClassName, @NonNull List<String> args) throws IOException {
        return startProcess(generateProcessParams(classPath, entryClassName, args));
    }

    public @NonNull Process start(@NonNull String classPath, @NonNull String entryClassName, @NonNull String[] args) throws IOException {
        return start(classPath, entryClassName, Arrays.asList(args));
    }

    public <T> @NonNull Process start(@NonNull String classPath, @NonNull Class<T> entryClass, @NonNull List<String> args) throws IOException {
        return start(classPath, entryClass.getName(), args);
    }

    public <T> @NonNull Process start(@NonNull String classPath, @NonNull Class<T> entryClass, @NonNull String[] args) throws IOException {
        return start(classPath, entryClass, Arrays.asList(args));
    }

    public boolean init() {
        return init(ActivityThread.currentActivityThread().getApplication());
    }

    public synchronized boolean init(@NonNull Context context) {
        if (initialized()) return true;
        mContext = context.getApplicationContext(); // 优化 4: 使用 ApplicationContext 防止 Activity 内存泄漏

        IProcessManager manager = newManager();
        if (manager == null) return false;
        mManager = manager;

        try {
            IBinder binder = manager.asBinder();
            binder.linkToDeath(() -> {
                // 使用内部局部变量引用，防止竞态条件
                if (mManager != null && binder == mManager.asBinder()) {
                    mManager = null;
                }
            }, 0);
        } catch (RemoteException e) {
            return false;
        }
        return initialized();
    }

    protected @Nullable IProcessManager newManager() {
        IBinder binder = isolatedServiceBinder(new ComponentName(mContext.getPackageName(), ProcessManager.class.getName()));
        if (binder == null) return null;

        IProcessManager serviceManager = IProcessManager.Stub.asInterface(binder);
        linkDeathToThis(serviceManager);
        return serviceManager;
    }

    public boolean initialized() {
        return mContext != null && mManager != null && mManager.asBinder().isBinderAlive();
    }

    @Override
    public void close() {
        mContext = null;
        if (mManager != null && mManager.asBinder().pingBinder()) {
            try {
                mManager.exit(0);
            } catch (Exception ignored) {
            }
        }
        mManager = null; // 优化 5: 显式置空，有助于 GC
    }

    protected @NonNull Process newProcess(@NonNull ProcessParams params) throws IOException {
        ProcessBuilder builder = new ProcessBuilder().command(params.mCmdList);
        if (params.mDirectory != null) builder.directory(new File(params.mDirectory));
        if (params.mEnv != null) builder.environment().putAll(params.mEnv);
        return builder.start();
    }

    private @NonNull Process startProcess(@NonNull ProcessParams params) throws IOException {
        if (!initialized()) return newProcess(params);
        return remoteProcess(params.mCmdList, params.mEnv, params.mDirectory);
    }

    private @NonNull IProcessManager requireManager() {
        IProcessManager manager = mManager; // 优化 6: 局部变量锁定，防止多线程环境下在判空后被 close 置空
        if (manager == null || !manager.asBinder().isBinderAlive()) {
            throw new IllegalStateException("ProcessManager not initialized or binder dead.");
        }
        return manager;
    }

    public boolean remoteTransact(IBinder binder, int code, Parcel data, Parcel reply, int flags) {
        return remoteTransact(requireManager(), binder, code, data, reply, flags);
    }

    public IBinder binderWrapper(IBinder binder) {
        return binderWrapper(requireManager(), binder);
    }

    public Process remoteProcess(@NonNull List<String> cmdList, @Nullable Map<String, String> env, @Nullable String directory) {
        try {
            return new RemoteProcess(requireManager().remoteProcess(cmdList, env, directory));
        } catch (RemoteException e) {
            throw new RuntimeException(e);
        }
    }

    public IBinder serviceBinder(ComponentName componentName) {
        try {
            return requireManager().serviceBinder(componentName).getBinder();
        } catch (RemoteException e) {
            throw new RuntimeException(e);
        }
    }

    public void linkDeathToThis() {
        linkDeathTo(new ClientManager().asBinder());
    }

    public void linkDeathTo(@Nullable IBinder binder) {
        linkDeathTo(requireManager(), binder);
    }

    // 优化 7: 使用 ConcurrentHashMap 后，简单的 computeIfAbsent 即可替代繁琐的 buildLock
    private final Map<String, Object> locks = new ConcurrentHashMap<>();

    public IBinder isolatedServiceBinder(@NonNull ComponentName componentName, boolean useCache) {
        if (!useCache) {
            String token = componentName.flattenToString();
            mChildProcess.remove(token);
        }
        return isolatedServiceBinder(componentName);
    }

    public IBinder isolatedServiceBinder(@NonNull ComponentName componentName) {
        String token = componentName.flattenToString();
        // 优化 8: 使用 computeIfAbsent 保证同 token 的初始化是原子性的
        synchronized (locks.computeIfAbsent(token, k -> new Object())) {
            IBinder existsBinder = mChildProcess.get(token);
            if (existsBinder != null && existsBinder.isBinderAlive()) return existsBinder;

            final IBinder binder = NewProcessReceiver.start(mContext, this, componentName);
            if (binder == null) return null;

            mChildProcess.put(token, binder);
            try {
                binder.linkToDeath(() -> mChildProcess.remove(token, binder), 0);
            } catch (RemoteException e) {
                mChildProcess.remove(token);
            }
            return binder;
        }
    }

    public static class Default extends AppProcess {
    }

    public static class None extends AppProcess {
        @Override
        protected IProcessManager newManager() {
            return new ProcessManager();
        }

        @Override
        public void close() {
            mManager = null;
        }
    }

    public abstract static class Terminal extends Default {
        protected abstract @NonNull List<String> newTerminal();

        @NonNull
        @Override
        protected Process newProcess(@NonNull ProcessParams params) throws IOException {
            Process process = innerProcess(new ProcessParams(params).setCmdList(newTerminal()));
            try (PrintWriter writer = new PrintWriter(process.getOutputStream(), true)) {
                writer.println(String.join(" ", params.mCmdList)); // 优化 9: 使用 String.join 替代 StringBuilder 循环
                writer.println("exit $?");
            }
            return process;
        }

        @NonNull
        protected Process innerProcess(@NonNull ProcessParams params) throws IOException {
            return super.newProcess(params);
        }
    }

    public static class Root extends Terminal {
        @NonNull
        @Override
        protected List<String> newTerminal() {
            return new ArrayList<>(Arrays.asList("su"));
        }
    }

    public static class RootSystem extends Terminal {
        @NonNull
        @Override
        protected List<String> newTerminal() {
            return new ArrayList<>(Arrays.asList("su", "1000"));
        }
    }

    // 优化 10: 成员变量改为包级可见，方便内部类直接访问，减少生成的 Accessor 方法
    public static class ProcessParams {
        @NonNull
        List<String> mCmdList;
        @Nullable
        Map<String, String> mEnv;
        @Nullable
        String mDirectory;

        public ProcessParams(@NonNull List<String> cmdList, @Nullable Map<String, String> env, @Nullable String directory) {
            this.mCmdList = cmdList;
            this.mEnv = env;
            this.mDirectory = directory;
        }

        public ProcessParams(@NonNull ProcessParams params) {
            this.mCmdList = new ArrayList<>(params.mCmdList);
            this.mEnv = params.mEnv != null ? new HashMap<>(params.mEnv) : null;
            this.mDirectory = params.mDirectory;
        }

        @NonNull
        public List<String> getCmdList() {
            return mCmdList;
        }

        @Nullable
        public Map<String, String> getEnv() {
            return mEnv;
        }

        @Nullable
        public String getDirectory() {
            return mDirectory;
        }

        public ProcessParams setCmdList(@NonNull List<String> cmdList) {
            this.mCmdList = cmdList;
            return this;
        }

        public ProcessParams setEnv(@Nullable Map<String, String> env) {
            this.mEnv = env;
            return this;
        }

        public ProcessParams setDirectory(@Nullable String directory) {
            this.mDirectory = directory;
            return this;
        }
    }
}