package io.github.kaposke.http.interceptors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import lombok.Getter;
import lombok.Setter;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class RefreshTokenInterceptor implements Interceptor {

  private final String endpoint;
  private final String refreshInputKey;
  private final String bearerKey;
  private final String refreshKey;

  private final ObjectMapper objectMapper = new ObjectMapper();

  @Getter
  @Setter
  private String bearerToken;

  @Getter
  @Setter
  private String refreshToken;

  public RefreshTokenInterceptor(
    String endpoint,
    String refreshInputKey,
    String bearerKey,
    String refreshKey
  ) {
    if (endpoint == null || endpoint.trim().isEmpty()) {
      throw new IllegalArgumentException("Endpoint is null or blank.");
    } else if (refreshInputKey == null || refreshInputKey.trim().isEmpty()) {
      throw new IllegalArgumentException("Refresh input key is null or blank.");
    } else if (bearerKey == null || bearerKey.trim().isEmpty()) {
      throw new IllegalArgumentException("Bearer key is null or blank.");
    }

    this.endpoint = endpoint;
    this.refreshInputKey = refreshInputKey;
    this.bearerKey = bearerKey;
    this.refreshKey = refreshKey;
  }

  @Override
  public Response intercept(Chain chain) throws IOException {
    Response response = chain.proceed(addBearerTokenToRequest(chain.request()));

    if (response.code() != 401 || this.refreshToken == null || this.refreshToken.isEmpty()) {
      return response;
    }

    response.close();

    if (!this.updateAuthTokens(chain)) {
      return response;
    }

    return chain.proceed(addBearerTokenToRequest(chain.request()));
  }

  Request addBearerTokenToRequest(Request request) {
    return this.bearerToken != null && !this.bearerToken.isEmpty()
      ? request.newBuilder().addHeader("Authorization", "Bearer " + this.bearerToken).build()
      : request;
  }

  private boolean updateAuthTokens(Chain chain) throws IOException {
    Map<String, String> requestMap = new HashMap<String, String>();
    requestMap.put(this.refreshInputKey, this.refreshToken);

    Response response = chain.proceed(
      chain
        .request()
        .newBuilder()
        .url(endpoint)
        .post(
          RequestBody.create(
            this.objectMapper.writeValueAsString(requestMap),
            MediaType.parse("application/json; charset=utf-8")
          )
        )
        .build()
    );

    if (!response.isSuccessful()) {
      return false;
    }

    Map<String, Object> responseMap =
      this.objectMapper.readValue(
          response.body().string(),
          new TypeReference<Map<String, Object>>() {}
        );

    if (this.refreshKey != null && responseMap.get(this.refreshKey) instanceof String) {
      this.refreshToken = (String) responseMap.get(refreshKey);
    }
    if (this.bearerKey != null && responseMap.get(this.bearerKey) instanceof String) {
      this.bearerToken = (String) responseMap.get(bearerKey);
    }

    return true;
  }
}
