package com.liecoder.framework.crop

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapShader
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Matrix
import android.graphics.Paint
import android.graphics.Path
import android.graphics.PointF
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.graphics.RectF
import android.graphics.Shader
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.Drawable
import android.graphics.drawable.ShapeDrawable
import android.graphics.drawable.shapes.OvalShape
import android.util.AttributeSet
import android.view.MotionEvent
import androidx.appcompat.widget.AppCompatImageView
import com.bumptech.glide.Glide
import com.bumptech.glide.Priority
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.liecoder.framework.R
import com.liecoder.framework.ktx.dpFloat
import com.liecoder.framework.ktx.dpInt
import com.liecoder.framework.ktx.w
import com.liecoder.framework.toast.TipsToast.Infos
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min
import kotlin.math.pow
import kotlin.math.roundToInt
import kotlin.math.sqrt

class CropWindowView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : AppCompatImageView(context, attrs, defStyleAttr), RequestListener<Drawable> {

    private val mAnchorBorderPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mAnchorPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mCropWindowPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mMaskPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mGuideLinePaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mMagnifierPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mMagnifierCrossPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val mPointDraggingPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }

    var mScaleX: Float = 1f
    var mScaleY: Float = 1f
    private var mActWidth: Float = 0f
    private var mActHeight: Float = 0f
    private var mActLeft: Float = 0f
    private var mActTop: Float = 0f
    private var mDraggingAnchorPoint: PointF? = null
    private var mDraggingWindowPoint: PointF? = null
    private var mMagnifierDrawable: ShapeDrawable? = null

    private val mMatrixValue = FloatArray(9)
    private val mMaskXfermode = PorterDuffXfermode(PorterDuff.Mode.DST_OUT)
    private val mAnchorLinePath = Path()
    private val mMagnifierMatrix = Matrix()

    private var mCornerAnchorPoints: Array<PointF>? = null
    private var mEdgeAnchorPoints: Array<PointF>? = null
    private var mCropWindowLineWidth = 2.dpFloat
    private var mAnchorPointBorderColor: Int = Color.WHITE
    private var mAnchorPointBorderWidth = 1.dpFloat
    private var mGuideLineWidth = 1.dpFloat
    private var mAnchorPointColor: Int = Color.parseColor("#FFF34D55")
    private var mCropWindowColor: Int = Color.parseColor("#FFF34D55")
    private var mMagnifierCrossColor: Int = Color.parseColor("#FFF34D55")
    private var mPointDraggingColor: Int = Color.parseColor("#4DF34D55")
    private var mGuideLineColor = Color.WHITE
    private var mMaskAlpha = 77
    private var mShowGuideLine = false
    private var mShowMagnifier = false
    private var mShowEdgeAnchorPoint = true
    private var mDragLimit = true

    private var mCropService: ICropWindowChangeLister? = null

    init {
        // 获取当前 ImageView 的缩放类型。
        val scaleType = scaleType

        // 检查缩放类型是否为 FIT_END、FIT_START 或 MATRIX。
        // 如果是，则抛出异常，因为这些缩放类型不适合裁剪操作。
        if (scaleType == ScaleType.FIT_END || scaleType == ScaleType.FIT_START || scaleType == ScaleType.MATRIX) {
            throw RuntimeException("Image in CropImageView must be in center")
        }

        // 初始化从 XML 属性传递的自定义属性。
        initAttrs(context, attrs)

        // 初始化画笔和其他绘制相关的资源。
        initPaints()
    }

    /**
     * 初始化从 XML 属性传递的自定义属性。
     *
     * @param context 用于创建此视图的上下文。
     * @param attrs 从 XML 属性传递的属性集。
     */
    private fun initAttrs(context: Context, attrs: AttributeSet?) {
        // 获取自定义属性的 TypedArray。
        val typedArray = context.obtainStyledAttributes(attrs, R.styleable.CropWindowView)

        // 初始化蒙版透明度，默认值为 86。
        // 值被限制在 0 到 255 之间。
        mMaskAlpha =
            typedArray.getInt(R.styleable.CropWindowView_cropMaskAlpha, 77).coerceIn(0, 255)

        // 初始化是否显示辅助线，默认值为 true。
        mShowGuideLine = typedArray.getBoolean(R.styleable.CropWindowView_showCropGuideLine, false)

        // 初始化选区线的颜色，默认值为 #FF2D71FF。
        mCropWindowColor = typedArray.getColor(
            R.styleable.CropWindowView_cropWindowColor, Color.parseColor("#FFF34D55")
        )

        // 初始化选区线的宽度，默认值为 2dp 单位转换为像素。
        mCropWindowLineWidth = typedArray.getDimension(
            R.styleable.CropWindowView_cropWindowLineWidth, 2.dpFloat
        )

        // 初始化锚点的颜色，默认值为 #FF2D71FF。
        mAnchorPointBorderColor = typedArray.getColor(
            R.styleable.CropWindowView_anchorPointBorderColor, Color.WHITE
        )

        // 初始化锚点的宽度，默认值为 2dp 单位转换为像素。
        mAnchorPointBorderWidth = typedArray.getDimension(
            R.styleable.CropWindowView_anchorPointBorderWidth, 1.dpFloat
        )

        // 初始化放大镜十字线的颜色，默认值为 #FF2D71FF。
        mMagnifierCrossColor = typedArray.getColor(
            R.styleable.CropWindowView_cropMagnifierCrossColor, Color.parseColor("#FFF34D55")
        )

        // 初始化是否显示放大镜，默认值为 true。
        mShowMagnifier = typedArray.getBoolean(R.styleable.CropWindowView_showCropMagnifier, false)

        // 初始化辅助线的宽度，默认值为 dp 单位转换为像素的 0.3F。
        mGuideLineWidth = typedArray.getDimension(
            R.styleable.CropWindowView_cropGuideLineWidth, 1.dpFloat
        )

        // 初始化辅助线的颜色，默认值为 Color.WHITE。
        mGuideLineColor = typedArray.getColor(
            R.styleable.CropWindowView_cropGuideLineColor, Color.WHITE
        )

        // 初始化锚点填充颜色，默认值为 #FF2D71FF。
        mAnchorPointColor = typedArray.getColor(
            R.styleable.CropWindowView_anchorPointColor, Color.parseColor("#FFF34D55")
        )

        // 初始化是否显示边锚点，默认值为 true。
        mShowEdgeAnchorPoint =
            typedArray.getBoolean(R.styleable.CropWindowView_showEdgeAnchorPoint, true)

        // 回收 TypedArray 对象，避免内存泄漏。
        typedArray.recycle()
    }

    /**
     * 初始化用于绘制的各种画笔。
     */
    private fun initPaints() {
        // 初始化锚点的边框画笔，设置为抗锯齿并指定颜色和宽度。
        mAnchorBorderPaint.apply {
            color = mAnchorPointBorderColor
            strokeWidth = mAnchorPointBorderWidth
            style = Paint.Style.STROKE
        }

        // 初始化锚点的填充画笔，设置为抗锯齿并指定颜色和透明度。
        mAnchorPaint.apply {
            color = mAnchorPointColor
            style = Paint.Style.FILL
        }

        // 初始化选区线的画笔，设置为抗锯齿并指定颜色和宽度。
        mCropWindowPaint.apply {
            color = mCropWindowColor
            strokeWidth = mCropWindowLineWidth
            style = Paint.Style.STROKE
        }

        // 初始化蒙版画笔，设置为抗锯齿并使用黑色填充。
        mMaskPaint.apply {
            color = Color.BLACK
            style = Paint.Style.FILL
        }

        // 初始化辅助线的画笔，设置为抗锯齿并指定颜色和宽度。
        mGuideLinePaint.apply {
            color = mGuideLineColor
            style = Paint.Style.FILL
            strokeWidth = mGuideLineWidth
        }

        // 初始化放大镜的画笔，设置为抗锯齿并使用白色填充。
        mMagnifierPaint.apply {
            color = Color.WHITE
            style = Paint.Style.FILL
        }

        // 初始化放大镜十字线的画笔，设置为抗锯齿并指定颜色和宽度。
        mMagnifierCrossPaint.apply {
            color = mMagnifierCrossColor
            style = Paint.Style.FILL
            strokeWidth = MAGNIFIER_CROSS_LINE_WIDTH.dpFloat
        }

        mPointDraggingPaint.apply {
            color = mPointDraggingColor
            style = Paint.Style.FILL
        }
    }

    /**
     * 在画布上绘制视图的内容。
     *
     * @param canvas 用于绘制的画布。
     */
    override fun onDraw(canvas: Canvas) {
        try {
            super.onDraw(canvas)

            // 初始化图片位置信息
            getDrawablePosition()

            // 开始绘制选区
            onDrawCropPoint(canvas)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * 获取Drawable的位置和尺寸
     */
    private fun getDrawablePosition() {
        // 获取Drawable对象
        val drawable = getDrawable()
        if (drawable != null) {
            // 获取图像矩阵并获取其值
            imageMatrix.getValues(mMatrixValue)
            // 获取缩放因子
            mScaleX = mMatrixValue[Matrix.MSCALE_X]
            mScaleY = mMatrixValue[Matrix.MSCALE_Y]
            // 获取Drawable的原始宽高
            val origW = drawable.intrinsicWidth
            val origH = drawable.intrinsicHeight
            // 计算缩放后的宽高
            mActWidth = (origW * mScaleX).roundToInt().toFloat()
            mActHeight = (origH * mScaleY).roundToInt().toFloat()
            // 计算Drawable在视图中的位置
            mActLeft = (width - mActWidth) / 2
            mActTop = (height - mActHeight) / 2
        }
    }

    /**
     * 在画布上绘制裁剪区域的各个组成部分。
     *
     * @param canvas 用于绘制的画布。
     */
    private fun onDrawCropPoint(canvas: Canvas) {
        // 绘制蒙版，覆盖不需要的区域。
        onDrawMask(canvas)

        // 绘制辅助线，帮助用户对齐裁剪区域。
        onDrawGuideLine(canvas)

        // 绘制选区线，显示裁剪框。
        onDrawCropWindow(canvas)

        // 绘制锚点，允许用户调整裁剪区域的大小和位置。
        onDrawAnchorPoints(canvas)

        // 绘制放大镜，提供裁剪区域的放大视图。
        onDrawMagnifier(canvas)
    }


    /**
     * 在画布上绘制遮罩效果
     * @param canvas 画布对象，用于绘制图形
     */
    private fun onDrawMask(canvas: Canvas) {
        // 如果遮罩透明度小于等于0，则不绘制
        if (mMaskAlpha <= 0) {
            return
        }

        // 获取路径
        val path = resetAnchorLinePath()
        if (path != null) {
            // 保存画布状态
            val sc = canvas.saveLayer(
                mActLeft,
                mActTop,
                mActLeft + mActWidth,
                mActTop + mActHeight,
                mMaskPaint,
                Canvas.ALL_SAVE_FLAG
            )

            // 设置遮罩的透明度
            mMaskPaint.alpha = mMaskAlpha

            // 绘制矩形遮罩
            canvas.drawRect(
                mActLeft, mActTop, mActLeft + mActWidth, mActTop + mActHeight, mMaskPaint
            )

            // 设置混合模式
            mMaskPaint.xfermode = mMaskXfermode

            // 重置透明度为不透明
            mMaskPaint.alpha = 255

            // 绘制路径
            canvas.drawPath(path, mMaskPaint)

            // 清除混合模式
            mMaskPaint.xfermode = null

            // 恢复画布状态
            canvas.restoreToCount(sc)
        }
    }

    /**
     * 在画布上绘制辅助线
     * @param canvas 画布对象，用于绘制图形
     */
    private fun onDrawGuideLine(canvas: Canvas) {
        // 如果不显示辅助线，则直接返回
        if (!mShowGuideLine) {
            return
        }

        // 计算辅助线的宽度和高度步长
        val widthStep = mActWidth / 3
        val heightStep = mActHeight / 3

        // 绘制水平辅助线
        canvas.drawLine(
            mActLeft + widthStep,
            mActTop,
            mActLeft + widthStep,
            mActTop + mActHeight,
            mGuideLinePaint
        )
        canvas.drawLine(
            mActLeft + widthStep * 2,
            mActTop,
            mActLeft + widthStep * 2,
            mActTop + mActHeight,
            mGuideLinePaint
        )

        // 绘制垂直辅助线
        canvas.drawLine(
            mActLeft,
            mActTop + heightStep,
            mActLeft + mActWidth,
            mActTop + heightStep,
            mGuideLinePaint
        )
        canvas.drawLine(
            mActLeft,
            mActTop + heightStep * 2,
            mActLeft + mActWidth,
            mActTop + heightStep * 2,
            mGuideLinePaint
        )
    }

    /**
     * 在画布上绘制线条
     * @param canvas 画布对象，用于绘制图形
     */
    private fun onDrawCropWindow(canvas: Canvas) {
        // 获取路径
        val path = resetAnchorLinePath()
        // 如果路径不为空，则在画布上绘制该路径
        if (path != null) {
            canvas.drawPath(path, mCropWindowPaint)
        }
    }

    /**
     * 在画布上绘制点
     * @param canvas 画布对象，用于绘制图形
     */
    private fun onDrawAnchorPoints(canvas: Canvas) {
        // 检查点是否有效
        if (!checkAnchorPoints(mCornerAnchorPoints)) return

        // 绘制角锚点
        for (point in mCornerAnchorPoints!!) {
            canvas.drawCircle(
                point.viewPointX, point.viewPointY, ANCHOR_RADIUS.dpFloat, mAnchorPaint
            )
            canvas.drawCircle(
                point.viewPointX, point.viewPointY, ANCHOR_RADIUS.dpFloat, mAnchorBorderPaint
            )
        }

        // 绘制边锚点
        if (mShowEdgeAnchorPoint) {
            if (mEdgeAnchorPoints.isNullOrEmpty()) return
            for (point in mEdgeAnchorPoints!!) {
                canvas.drawCircle(
                    point.viewPointX, point.viewPointY, ANCHOR_RADIUS.dpFloat, mAnchorPaint
                )
                canvas.drawCircle(
                    point.viewPointX, point.viewPointY, ANCHOR_RADIUS.dpFloat, mAnchorBorderPaint
                )
            }
        }
        if (mDraggingAnchorPoint != null) {
            canvas.drawCircle(
                mDraggingAnchorPoint!!.viewPointX,
                mDraggingAnchorPoint!!.viewPointY,
                DRAGGING_ANCHOR_RADIUS.dpFloat,
                mPointDraggingPaint
            )
        }
    }


    /**
     * 在画布上绘制放大镜效果
     * @param canvas 画布对象，用于绘制图形
     */
    private fun onDrawMagnifier(canvas: Canvas) {
        // 如果显示放大镜且有拖动点，则进行绘制
        if (mShowMagnifier && mDraggingAnchorPoint != null) {
            // 如果放大镜Drawable为空，则初始化
            if (mMagnifierDrawable == null) {
                initMagnifier()
            }

            // 获取拖动点的视图坐标
            val draggingX = mDraggingAnchorPoint?.viewPointX ?: return
            val draggingY = mDraggingAnchorPoint?.viewPointY ?: return

            // 计算放大镜的半径
            val radius = width / 8f
            var cx = radius
            val lineOffset = MAGNIFIER_BORDER_WIDTH.dpInt

            // 设置放大镜Drawable的边界
            mMagnifierDrawable?.setBounds(
                lineOffset,
                lineOffset,
                (radius * 2 - lineOffset).toInt(),
                (radius * 2 - lineOffset).toInt()
            )

            // 计算拖动点与放大镜中心的距离
            val pointsDistance =
                sqrt((draggingX - 0F).toDouble().pow(2) + (draggingY - 0F).toDouble().pow(2))
            if (pointsDistance < (radius * 2.5)) {
                mMagnifierDrawable?.setBounds(
                    (width - (radius * 2 - lineOffset)).toInt(),
                    lineOffset,
                    width - lineOffset,
                    (radius * 2 - lineOffset).toInt()
                )
                cx = width - radius
            }

            // 绘制放大镜圆形
            canvas.drawCircle(cx, radius, radius, mMagnifierPaint)

            // 设置放大镜Drawable的平移矩阵
            mMagnifierMatrix.setTranslate(radius - draggingX, radius - draggingY)
            mMagnifierDrawable?.paint?.shader?.setLocalMatrix(mMagnifierMatrix)

            // 绘制放大镜Drawable
            mMagnifierDrawable?.draw(canvas)

            // 绘制放大镜的十字线
            val crossLength = MAGNIFIER_CROSS_LINE_LENGTH.dpInt
            canvas.drawLine(
                cx, radius - crossLength, cx, radius + crossLength, mMagnifierCrossPaint
            )
            canvas.drawLine(
                cx - crossLength, radius, cx + crossLength, radius, mMagnifierCrossPaint
            )
        }
    }

    /**
     * 检查裁剪点是否有效。
     *
     * @param points 裁剪点数组。
     * @return 如果裁剪点数组不为空且所有点都非空，则返回 true，否则返回 false。
     */
    private fun checkAnchorPoints(points: Array<PointF>?): Boolean =
        points?.isNotEmpty() == true && points.size == 4

    /**
     * 重置点的路径
     * @return 路径对象，如果点无效则返回null
     */
    private fun resetAnchorLinePath(): Path? {
        // 检查点是否有效
        if (!checkAnchorPoints(mCornerAnchorPoints)) return null

        // 重置路径
        mAnchorLinePath.reset()

        // 获取四个角点
        val anchorLT = mCornerAnchorPoints?.getOrNull(Anchor.LT.position) ?: return null
        val anchorRT = mCornerAnchorPoints?.getOrNull(Anchor.RT.position) ?: return null
        val anchorRB = mCornerAnchorPoints?.getOrNull(Anchor.RB.position) ?: return null
        val anchorLB = mCornerAnchorPoints?.getOrNull(Anchor.LB.position) ?: return null

        // 将路径移动到左上角点
        mAnchorLinePath.moveTo(anchorLT.viewPointX, anchorLT.viewPointY)

        // 绘制路径到右上角点
        mAnchorLinePath.lineTo(anchorRT.viewPointX, anchorRT.viewPointY)

        // 绘制路径到右下角点
        mAnchorLinePath.lineTo(anchorRB.viewPointX, anchorRB.viewPointY)

        // 绘制路径到左下角点
        mAnchorLinePath.lineTo(anchorLB.viewPointX, anchorLB.viewPointY)

        // 关闭路径形成封闭区域
        mAnchorLinePath.close()

        // 返回路径对象
        return mAnchorLinePath
    }

    /**
     * 初始化放大镜效果。
     */
    private fun initMagnifier() {
        // 创建一个新的 Bitmap，尺寸与 ImageView 相同。
        val bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.RGB_565)
        val canvas = Canvas(bitmap)

        // 在 Bitmap 上绘制黑色背景。
        canvas.drawColor(Color.BLACK)

        // 获取与 ImageView 关联的 Bitmap。
        val bmp = (drawable as? BitmapDrawable)?.bitmap ?: return

        // 在黑色背景上绘制图片的指定区域。
        canvas.drawBitmap(
            bmp, null, RectF(mActLeft, mActTop, mActLeft + mActWidth, mActTop + mActHeight), null
        )

        // 创建一个 BitmapShader，用于在 ShapeDrawable 上绘制放大镜效果。
        val magnifierShader = BitmapShader(bitmap, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP)

        // 创建一个椭圆形状的 ShapeDrawable 作为放大镜的图形。
        mMagnifierDrawable = ShapeDrawable(OvalShape())
        mMagnifierDrawable?.paint?.shader = magnifierShader
    }

    fun getCropEdge(): RectF? {
        if (drawable == null) return null
        val left = mCornerAnchorPoints?.getOrNull(Anchor.LT.position)?.x ?: 0f
        val top = mCornerAnchorPoints?.getOrNull(Anchor.LT.position)?.y ?: 0f
        val right = mCornerAnchorPoints?.getOrNull(Anchor.RB.position)?.x
            ?: drawable.intrinsicWidth.toFloat()
        val bottom = mCornerAnchorPoints?.getOrNull(Anchor.LB.position)?.y
            ?: drawable.intrinsicHeight.toFloat()
        return RectF(left, top, right, bottom)
    }

    fun getFullImageEdge(): RectF? {
        if (drawable == null) return null
        return RectF(0f, 0f, drawable.intrinsicWidth.toFloat(), drawable.intrinsicHeight.toFloat())
    }

    fun setCropEdge(
        left: Float? = null,
        top: Float? = null,
        right: Float? = null,
        bottom: Float? = null,
        edgeMargin: Int = 8
    ) {
        if (drawable == null) return w("should call after set drawable")
        val maxWidth = drawable.intrinsicWidth.toFloat()
        val maxHeight = drawable.intrinsicHeight.toFloat()
        val points = Array(4) { PointF(0f, 0f) }
        points[0] = PointF(
            (left ?: 0f).coerceAtLeast(0f),
            (top ?: 0f).coerceAtLeast(0f)
        )
        points[1] = PointF(
            (right ?: maxWidth).coerceAtMost(maxWidth),
            (top ?: 0f).coerceAtLeast(0f)
        )
        points[2] = PointF(
            (right ?: maxWidth).coerceAtMost(maxWidth),
            (bottom ?: maxHeight).coerceAtMost(maxHeight)
        )
        points[3] = PointF(
            (left ?: 0f).coerceAtLeast(0f),
            (bottom ?: maxHeight).coerceAtMost(maxHeight)
        )
        if (edgeMargin > 0) {
            val margin = edgeMargin.dpFloat
            points[0] = PointF(points[0].x + margin, points[0].y + margin)
            points[1] = PointF(points[1].x - margin, points[1].y + margin)
            points[2] = PointF(points[2].x - margin, points[2].y - margin)
            points[3] = PointF(points[3].x + margin, points[3].y - margin)
        }
        mCornerAnchorPoints = points
        setEdgeMidPoints()
        postInvalidate()
    }


    /**
     *  获取裁剪比例
     */
    fun getCropRatio(): RectF? {
        if (drawable == null || !checkAnchorPoints(mCornerAnchorPoints)) return null
        val anchorPoints = mCornerAnchorPoints!!
        val left = anchorPoints[Anchor.LT.position].x
        val top = anchorPoints[Anchor.LT.position].y
        val right = anchorPoints[Anchor.RB.position].x
        val bottom = anchorPoints[Anchor.LB.position].y
        val totalWidth = drawable.intrinsicWidth.toFloat()
        val totalHeight = drawable.intrinsicHeight.toFloat()
        val leftRatio = left / totalWidth
        val rightRatio = right / totalWidth
        val topRatio = top / totalHeight
        val bottomRatio = bottom / totalHeight
        return RectF(leftRatio, topRatio, rightRatio, bottomRatio)
    }

    /**
     * 设置全图裁剪，即裁剪区域为整个图片。
     */
    private fun setFullImageCrop() {
        // 检查是否有图片被设置到 ImageView 中。
        if (drawable == null) {
            w("should call after set drawable")
            return
        }

        // 获取全图裁剪点。
        mCornerAnchorPoints = getFullImageWindow()

        // 计算边缘中点。
        setEdgeMidPoints()

        // 通知视图需要重绘。
        invalidate()
    }

    /**
     * 设置边缘中点，这些中点用于辅助线显示。
     */
    private fun setEdgeMidPoints() {
        // 如果 mEdgeMidPoints 为空，则初始化为4个 Point 对象的数组。
        mEdgeAnchorPoints = mEdgeAnchorPoints?.clone() ?: Array(4) { PointF() }

        // 如果裁剪点无效，则设置为全图裁剪。
        if (!checkAnchorPoints(mCornerAnchorPoints)) {
            setFullImageCrop()
            return
        }

        // 计算每两个裁剪点之间的中点。
        mCornerAnchorPoints?.forEachIndexed { i, point ->
            val nextIndex = (i + 1) % (mCornerAnchorPoints?.size ?: 0)
            val nextPoint = mCornerAnchorPoints?.getOrNull(nextIndex)
            mEdgeAnchorPoints?.getOrNull(i)?.set(
                (point.x.plus(nextPoint?.x ?: 0f)).div(2), (point.y.plus(nextPoint?.y ?: 0f)).div(2)
            )
        }
    }

    fun loadImageResource(
        imageResource: Any?,
        skipMemoryCache: Boolean = true
    ) {
        val requestManager = Glide.with(context)
        when (imageResource) {
            is Int -> requestManager.load(imageResource)
            is String -> requestManager.load(imageResource)
            is Bitmap -> requestManager.load(imageResource)
            is Drawable -> requestManager.load(imageResource)
            else -> requestManager.load(imageResource)
        }
            .skipMemoryCache(skipMemoryCache)
            .diskCacheStrategy(if (skipMemoryCache) DiskCacheStrategy.NONE else DiskCacheStrategy.RESOURCE)
            .priority(Priority.HIGH)
            .addListener(this)
            .into(this)

    }

    override fun onLoadFailed(
        e: GlideException?,
        model: Any?,
        target: Target<Drawable>?,
        isFirstResource: Boolean
    ): Boolean {
        Infos.show("Invalid resource")
        return false
    }


    override fun onResourceReady(
        resource: Drawable?,
        model: Any?,
        target: Target<Drawable>?,
        dataSource: DataSource?,
        isFirstResource: Boolean
    ): Boolean {
        post {
            if (checkAnchorPoints(mCornerAnchorPoints)) {
                setEdgeMidPoints()// 计算边缘中点。
                invalidate() // 通知视图需要重绘。
            } else {
                setCropEdge() // 如果裁剪点无效，则设置为全图裁剪。
            }
        }
        return false
    }


    override fun setImageBitmap(bm: Bitmap?) {
        super.setImageBitmap(bm)
        mMagnifierDrawable = null
    }

    /**
     * 获取选区
     * @return 选区顶点
     */
    fun getCropWindow(): Array<PointF>? {
        return mCornerAnchorPoints
    }

    /**
     * 获取全图裁剪点，即图片的四个角作为裁剪点。
     * @return 裁剪点数组。
     */
    fun getFullImageWindow(): Array<PointF> {
        val drawable = drawable ?: return emptyArray()
        val points = Array(4) { PointF(0f, 0f) }
        points[0] = PointF(0f, 0f)
        points[1] = PointF(drawable.intrinsicWidth.toFloat(), 0f)
        points[2] = PointF(drawable.intrinsicWidth.toFloat(), drawable.intrinsicHeight.toFloat())
        points[3] = PointF(0f, drawable.intrinsicHeight.toFloat())
        return points
    }

    /**
     * 设置锚点填充颜色
     * @param pointFillColor 颜色
     */
    fun setAnchorPointColor(pointFillColor: Int) {
        this.mAnchorPointColor = pointFillColor
    }

    /**
     * 蒙版透明度
     * @param maskAlpha 透明度
     */
    fun setMaskAlpha(maskAlpha: Int) {
        this.mMaskAlpha = min(max(0.0, maskAlpha.toDouble()), 255.0).toInt()
        invalidate()
    }

    /**
     * 是否显示辅助线
     * @param showGuideLine 是否
     */
    fun setShowGuideLine(showGuideLine: Boolean) {
        this.mShowGuideLine = showGuideLine
        invalidate()
    }

    /**
     * 设置辅助线颜色
     * @param guideLineColor 颜色
     */
    fun setGuideLineColor(guideLineColor: Int) {
        this.mGuideLineColor = guideLineColor
    }

    /**
     * 设置辅助线宽度
     * @param guideLineWidth 宽度 px
     */
    fun setGuideLineWidth(guideLineWidth: Float) {
        this.mGuideLineWidth = guideLineWidth
    }

    /**
     * 设置选区线的颜色
     * @param lineColor 颜色
     */
    fun setCropWindowColor(lineColor: Int) {
        this.mCropWindowColor = lineColor
        invalidate()
    }

    /**
     * 设置放大镜准心颜色
     * @param magnifierCrossColor 准心颜色
     */
    fun setMagnifierCrossColor(magnifierCrossColor: Int) {
        this.mMagnifierCrossColor = magnifierCrossColor
    }

    /**
     * 设置选区线宽度
     * @param lineWidth 线宽度，px
     */
    fun setCropWindowLineWidth(lineWidth: Int) {
        this.mCropWindowLineWidth = lineWidth.toFloat()
        invalidate()
    }

    /**
     * 设置锚点的颜色。
     *
     * @param pointColor 新的锚点颜色。
     */
    fun setAnchorPointBorderColor(pointColor: Int, invalidate: Boolean = true) {
        mAnchorPointBorderColor = pointColor
        if (invalidate) {
            invalidate()
        }
    }

    /**
     * 设置锚点的宽度。
     *
     * @param pointWidth 新的锚点宽度。
     */
    fun setAnchorPointBorderWidth(pointWidth: Float, invalidate: Boolean = true) {
        mAnchorPointBorderWidth = pointWidth
        if (invalidate) {
            invalidate()
        }
    }

    /**
     * 设置是否显示放大镜
     * @param showMagnifier 是否
     */
    fun setShowMagnifier(showMagnifier: Boolean) {
        this.mShowMagnifier = showMagnifier
    }

    /**
     * 设置是否限制拖动为凸四边形
     * @param dragLimit 是否
     */
    fun setDragLimit(dragLimit: Boolean) {
        this.mDragLimit = dragLimit
    }

    /**
     * 检查裁剪区域是否为凸四边形，即裁剪区域是否正确。
     * 此方法通过检查四边形的边是否正确交叉来确定裁剪区域是否有效。
     *
     * @return 如果裁剪区域有效，则返回 true，否则返回 false。
     */
    fun canRightCrop(): Boolean {
        // 首先检查裁剪点是否有效。
        if (!checkAnchorPoints(mCornerAnchorPoints)) {
            return false
        }

        // 获取四个裁剪点。
        val anchorLT = mCornerAnchorPoints?.getOrNull(Anchor.LT.position) ?: return false
        val anchorRT = mCornerAnchorPoints?.getOrNull(Anchor.RT.position) ?: return false
        val anchorRB = mCornerAnchorPoints?.getOrNull(Anchor.RB.position) ?: return false
        val anchorLB = mCornerAnchorPoints?.getOrNull(Anchor.LB.position) ?: return false

        // 使用向量叉积来判断四边形是否为凸四边形。
        // 如果相邻边的叉积符号相同，则它们在同侧，形成凸四边形。
        return (pointSideLine(
            pointStart = anchorLT, pointEnd = anchorRB, pointTarget = anchorLB
        ) * pointSideLine(
            pointStart = anchorLT, pointEnd = anchorRB, pointTarget = anchorRT
        ) < 0) && (pointSideLine(
            pointStart = anchorLB, pointEnd = anchorRT, pointTarget = anchorLT
        ) * pointSideLine(
            pointStart = anchorLB, pointEnd = anchorRT, pointTarget = anchorRB
        ) < 0)
    }

    /**
     * 计算点到线段的叉积，用于判断点与线段的相对位置。
     *
     * @param pointStart 线段的起点。
     * @param pointEnd 线段的终点。
     * @param point 需要判断的点。
     * @return 叉积结果。
     */
    private fun pointSideLine(pointStart: PointF, pointEnd: PointF, pointTarget: PointF): Float =
        pointSideLine(
            startX = pointStart.x,
            startY = pointStart.y,
            endX = pointEnd.x,
            endY = pointEnd.y,
            targetX = pointTarget.x,
            targetY = pointTarget.y
        )

    /**
     * 计算点到线段的叉积，用于判断点与线段的相对位置。
     *
     * @param startX 线段起点的 X 坐标。
     * @param startY 线段起点的 Y 坐标。
     * @param endX 线段终点的 X 坐标。
     * @param endY 线段终点的 Y 坐标。
     * @param targetX 需要判断的点的 X 坐标。
     * @param targetY 需要判断的点的 Y 坐标。
     * @return 叉积结果。
     */
    private fun pointSideLine(
        startX: Float, startY: Float, endX: Float, endY: Float, targetX: Float, targetY: Float
    ): Float = (targetX - startX) * (endY - startY) - (targetY - startY) * (endX - startX)


    /**
     * 处理触摸事件
     * @param event 触摸事件对象
     * @return 事件是否被处理
     */
    override fun onTouchEvent(event: MotionEvent): Boolean {
        // 获取触摸事件的动作类型
        val action = event.action
        var handle = true

        // 根据动作类型进行处理
        when (action) {
            MotionEvent.ACTION_DOWN -> {
                // 当触摸按下时，获取最近的点
                mDraggingAnchorPoint = getNearbyPoint(event)
                if (mDraggingAnchorPoint == null) {
                    mDraggingWindowPoint = if (isTouchInsideCropWindow(event)) PointF(
                        event.x, event.y
                    ) else null
                    if (mDraggingWindowPoint == null) {
                        handle = false
                    }
                }
                parent?.requestDisallowInterceptTouchEvent(
                    mDraggingAnchorPoint != null || mDraggingWindowPoint != null
                )
            }

            MotionEvent.ACTION_MOVE -> {
                // 当触摸移动时，更新点的位置
                toImagePointSize(mDraggingAnchorPoint, event)
                // 更新边中点
                updateMidEdgePoints()
            }

            MotionEvent.ACTION_UP, MotionEvent.ACTION_CANCEL -> {
                // 通知裁剪窗口变化事件
                if (mDraggingAnchorPoint != null || mDraggingWindowPoint != null) {
                    mCropService?.onCropWindowChanged(this, getCropEdge())
                }
                // 当触摸抬起时，清除拖动的点
                mDraggingAnchorPoint = null
                mDraggingWindowPoint = null
                parent?.requestDisallowInterceptTouchEvent(false)
            }
        }
        // 重绘视图
        invalidate()
        // 返回是否处理了事件或父类的事件处理
        return handle || super.onTouchEvent(event)
    }

    /**
     * 获取触摸事件最近的点
     * @param event 触摸事件对象
     * @return 最近的点，如果没有找到则返回null
     */
    private fun getNearbyPoint(event: MotionEvent): PointF? {
        // 检查点是否有效
        if (!checkAnchorPoints(mCornerAnchorPoints)) {
            return null
        }

        // 遍历主要的锚点，找到最近的点
        for (p in mCornerAnchorPoints!!) {
            if (isTouchOnAnchorPoint(p, event)) return p
        }

        // 如果不显示边锚点，则直接返回null
        if (!mShowEdgeAnchorPoint) {
            return null
        }

        // 如果边锚点列表为空，则返回null
        if (mEdgeAnchorPoints.isNullOrEmpty()) {
            return null
        }

        // 遍历边锚点，找到最近的点
        for (p in mEdgeAnchorPoints ?: return null) {
            if (isTouchOnAnchorPoint(p, event)) return p
        }

        // 如果没有找到任何点，则返回null
        return null
    }

    /**
     * 检查触摸点是否在指定点附近
     * @param anchor 指定的点
     * @param event 触摸事件对象
     * @return 如果触摸点在指定点附近则返回true，否则返回false
     */
    private fun isTouchOnAnchorPoint(anchor: PointF, event: MotionEvent): Boolean {
        // 获取触摸事件的X和Y坐标
        val x = event.x
        val y = event.y
        // 获取指定点在视图中的X和Y坐标
        val px = anchor.viewPointX
        val py = anchor.viewPointY
        // 计算触摸点与指定点之间的距离
        val distance = sqrt((x - px).toDouble().pow(2) + (y - py).toDouble().pow(2))
        // 如果距离小于设定的触摸捕捉距离，则认为触摸点在指定点附近
        return distance < TOUCH_POINT_CATCH_DISTANCE.dpFloat
    }

    /**
     * 检查触摸点是否在窗口内
     * @param event 触摸事件
     * @return 如果触摸点在窗口内则返回true，否则返回false
     */
    private fun isTouchInsideCropWindow(event: MotionEvent): Boolean {
        // 获取触摸事件的X和Y坐标
        val x = event.x
        val y = event.y

        // 安全地获取边中点，如果不存在则返回false
        val pointLeft = mEdgeAnchorPoints?.get(Anchor.L.position - 4) ?: return false
        val pointTop = mEdgeAnchorPoints?.get(Anchor.T.position - 4) ?: return false
        val pointRight = mEdgeAnchorPoints?.get(Anchor.R.position - 4) ?: return false
        val pointBottom = mEdgeAnchorPoints?.get(Anchor.B.position - 4) ?: return false

        // 获取窗口的边界
        val leftEdge = pointLeft.viewPointX + TOUCH_POINT_CATCH_DISTANCE.dpFloat
        val topEdge = pointTop.viewPointY + TOUCH_POINT_CATCH_DISTANCE.dpFloat
        val rightEdge = pointRight.viewPointX - TOUCH_POINT_CATCH_DISTANCE.dpFloat
        val bottomEdge = pointBottom.viewPointY - TOUCH_POINT_CATCH_DISTANCE.dpFloat
        // 检查触摸点是否在窗口内
        return x in leftEdge..rightEdge && y in topEdge..bottomEdge
    }

    /**
     * 将触摸点转换为图像上的大小并更新拖动的点
     * @param dragPoint 拖动的点
     * @param event 触摸事件对象
     */
    private fun toImagePointSize(dragPoint: PointF?, event: MotionEvent) {
        // 如果拖动的点为空，则直接返回
        if (dragPoint == null) {
            moveWindow(event)
            return
        }

        // 获取点的类型
        val pointType = getPointType(dragPoint)

        // 计算触摸点在图像上的位置
        val x = getActualX(event.x)
        val y = getActualY(event.y)

        // 如果设置了拖动限制并且点的类型有效，则检查是否允许移动
        if (mDragLimit && pointType != null) {
            val canMoveLeftTop = canMoveLeftTop(x, y)
            val canMoveRightTop = canMoveRightTop(x, y)
            val canMoveRightBottom = canMoveRightBottom(x, y)
            val canMoveLeftBottom = canMoveLeftBottom(x, y)
            when (pointType) {
                Anchor.LT -> if (!canMoveLeftTop) return
                Anchor.RT -> if (!canMoveRightTop) return
                Anchor.RB -> if (!canMoveRightBottom) return
                Anchor.LB -> if (!canMoveLeftBottom) return
                Anchor.T -> if (!canMoveLeftTop || !canMoveRightTop) return
                Anchor.R -> if (!canMoveRightTop || !canMoveRightBottom) return
                Anchor.B -> if (!canMoveLeftBottom || !canMoveRightBottom) return
                Anchor.L -> if (!canMoveLeftBottom || !canMoveLeftTop) return
            }
        }
        val offsetX = x - dragPoint.x
        val offsetY = y - dragPoint.y
        moveEdge(pointType, offsetX, offsetY)
    }

    private fun getActualX(x: Float): Float =
        (x.coerceAtLeast(mActLeft).coerceAtMost(mActLeft + mActWidth) - mActLeft) / mScaleX

    private fun getActualY(y: Float): Float =
        (y.coerceAtLeast(mActTop).coerceAtMost(mActTop + mActHeight) - mActTop) / mScaleY

    /**
     * 根据拖动的边缘点类型移动相应的点
     * @param type 拖动的边缘点类型
     * @param offsetX x方向上的偏移量
     * @param offsetY y方向上的偏移量
     */
    private fun moveEdge(type: Anchor?, offsetX: Float, offsetY: Float) {
        when (type) {
            Anchor.T -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.LT.position), 0F, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.RT.position), 0F, offsetY)
            }

            Anchor.R -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.RT.position), offsetX, 0F)
                movePoint(mCornerAnchorPoints?.get(Anchor.RB.position), offsetX, 0F)
            }

            Anchor.B -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.LB.position), 0F, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.RB.position), 0F, offsetY)
            }

            Anchor.L -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.LT.position), offsetX, 0F)
                movePoint(mCornerAnchorPoints?.get(Anchor.LB.position), offsetX, 0F)
            }

            Anchor.LT -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.LT.position), offsetX, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.RT.position), 0F, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.LB.position), offsetX, 0F)
            }

            Anchor.LB -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.LB.position), offsetX, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.LT.position), offsetX, 0F)
                movePoint(mCornerAnchorPoints?.get(Anchor.RB.position), 0F, offsetY)
            }

            Anchor.RT -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.RT.position), offsetX, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.LT.position), 0F, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.RB.position), offsetX, 0F)
            }

            Anchor.RB -> {
                movePoint(mCornerAnchorPoints?.get(Anchor.RB.position), offsetX, offsetY)
                movePoint(mCornerAnchorPoints?.get(Anchor.RT.position), offsetX, 0F)
                movePoint(mCornerAnchorPoints?.get(Anchor.LB.position), 0F, offsetY)
            }

            else -> {}
        }
    }

    /**
     * 根据触摸事件移动裁剪窗口
     * @param event 触摸事件
     */
    private fun moveWindow(event: MotionEvent) {
        if (mDraggingWindowPoint == null) return
        // 计算触摸点在图像上的位置
        val x = (event.x
            .coerceAtLeast(mActLeft)
            .coerceAtMost(mActLeft + mActWidth) - mActLeft) / mScaleX
        val y =
            (event.y.coerceAtLeast(mActTop).coerceAtMost(mActTop + mActHeight) - mActTop) / mScaleY
        val offsetX = x - mDraggingWindowPoint!!.x
        val offsetY = y - mDraggingWindowPoint!!.y
        val anchorLT = mCornerAnchorPoints?.getOrNull(Anchor.LT.position) ?: return
        val anchorRT = mCornerAnchorPoints?.getOrNull(Anchor.RT.position) ?: return
        val anchorRB = mCornerAnchorPoints?.getOrNull(Anchor.RB.position) ?: return
        val anchorLB = mCornerAnchorPoints?.getOrNull(Anchor.LB.position) ?: return

        // 确保点的新位置不会超出Drawable的边界
        val drawableWidth = getDrawable()?.intrinsicWidth?.toFloat() ?: 0F
        val drawableHeight = getDrawable()?.intrinsicHeight?.toFloat() ?: 0F
        if ((anchorLT.x + offsetX) in 0F..drawableWidth && // 确保左边不出界
            anchorRT.x + offsetX in 0F..drawableWidth && // 确保右边不出界
            anchorRB.x + offsetX in 0F..drawableWidth &&// 确保右边不出界
            anchorLB.x + offsetX in 0F..drawableWidth// 确保左边不出界
        ) {
            anchorLT.x += offsetX
            anchorRT.x += offsetX
            anchorRB.x += offsetX
            anchorLB.x += offsetX
        }
        if ((anchorLT.y + offsetY) in 0F..drawableHeight && // 确保上边不出界
            anchorRT.y + offsetY in 0F..drawableHeight && // 确保上边不出界
            anchorRB.y + offsetY in 0F..drawableHeight && // 确保下边不出界
            anchorLB.y + offsetY in 0F..drawableHeight // 确保下边不出
        ) {
            anchorLT.y += offsetY
            anchorRT.y += offsetY
            anchorRB.y += offsetY
            anchorLB.y += offsetY
        }
        mDraggingWindowPoint!!.set(x, y)
    }


    /**
     * 移动点
     * @param point 要移动的点
     * @param offsetX x方向上的偏移量
     * @param offsetY y方向上的偏移量
     */
    private fun movePoint(point: PointF?, offsetX: Float, offsetY: Float) {
        if (point == null) return
        val x = point.x + offsetX
        val y = point.y + offsetY
        // 确保点的新位置不会超出Drawable的边界
        val drawableWidth = getDrawable()?.intrinsicWidth ?: 0
        val drawableHeight = getDrawable()?.intrinsicHeight ?: 0
        if (x >= 0 && x <= drawableWidth) point.x = x
        if (y >= 0 && y <= drawableHeight) point.y = y
    }


    /**
     * 检查左上角的点是否可以移动到指定的位置
     * @param targetX 新的x坐标
     * @param targetY 新的y坐标
     * @return 如果可以移动则返回true，否则返回false
     */
    private fun canMoveLeftTop(targetX: Float, targetY: Float): Boolean {
        // 检查点(x, y)是否在由P_RT, P_LB和P_RB形成的三角形内部
        val anchorRT = mCornerAnchorPoints?.getOrNull(Anchor.RT.position) ?: return false
        val anchorLB = mCornerAnchorPoints?.getOrNull(Anchor.LB.position) ?: return false
        val anchorRB = mCornerAnchorPoints?.getOrNull(Anchor.RB.position) ?: return false
        if (pointSideLine(
                startX = anchorRT.x,
                startY = anchorRT.y,
                endX = anchorLB.x,
                endY = anchorLB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorRT, pointEnd = anchorLB, pointTarget = anchorRB
            ) > 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_RT, P_RB和P_LB形成的三角形内部
        if (pointSideLine(
                startX = anchorRT.x,
                startY = anchorRT.y,
                endX = anchorRB.x,
                endY = anchorRB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorRT, pointEnd = anchorRB, pointTarget = anchorLB
            ) < 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_LB, P_RB和P_RT形成的三角形内部
        if (pointSideLine(
                startX = anchorLB.x,
                startY = anchorLB.y,
                endX = anchorRB.x,
                endY = anchorRB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLB, pointEnd = anchorRB, pointTarget = anchorRT
            ) < 0
        ) {
            return false
        }
        // 如果点(x, y)通过了所有的检查，则可以移动
        return true
    }

    /**
     * 检查右上角的点是否可以移动到指定的位置
     * @param targetX 新的x坐标
     * @param targetY 新的y坐标
     * @return 如果可以移动则返回true，否则返回false
     */
    private fun canMoveRightTop(targetX: Float, targetY: Float): Boolean {
        // 检查点(x, y)是否在由P_LT, P_RB和P_LB形成的三角形内部
        val anchorLT = mCornerAnchorPoints?.get(Anchor.LT.position) ?: return false
        val anchorLB = mCornerAnchorPoints?.get(Anchor.LB.position) ?: return false
        val anchorRB = mCornerAnchorPoints?.get(Anchor.RB.position) ?: return false
        if (pointSideLine(
                startX = anchorLT.x,
                startY = anchorLT.y,
                endX = anchorRB.x,
                endY = anchorRB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLT, pointEnd = anchorRB, pointTarget = anchorLB
            ) > 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_LT, P_LB和P_RB形成的三角形内部
        if (pointSideLine(
                startX = anchorLT.x,
                startY = anchorLT.y,
                endX = anchorLB.x,
                endY = anchorLB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLT, pointEnd = anchorLB, pointTarget = anchorRB
            ) < 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_LB, P_RB和P_LT形成的三角形内部
        if (pointSideLine(
                startX = anchorLB.x,
                startY = anchorLB.y,
                endX = anchorRB.x,
                endY = anchorRB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLB, pointEnd = anchorRB, pointTarget = anchorLT
            ) < 0
        ) {
            return false
        }
        // 如果点(x, y)通过了所有的检查，则可以移动
        return true
    }

    /**
     * 检查右下角的点是否可以移动到指定的位置
     * @param targetX 新的x坐标
     * @param targetY 新的y坐标
     * @return 如果可以移动则返回true，否则返回false
     */
    private fun canMoveRightBottom(targetX: Float, targetY: Float): Boolean {
        // 检查点(x, y)是否在由P_RT, P_LB和P_LT形成的三角形内部
        val anchorLT = mCornerAnchorPoints?.get(Anchor.LT.position) ?: return false
        val anchorRT = mCornerAnchorPoints?.get(Anchor.RT.position) ?: return false
        val anchorLB = mCornerAnchorPoints?.get(Anchor.LB.position) ?: return false
        if (pointSideLine(
                startX = anchorRT.x,
                startY = anchorRT.y,
                endX = anchorLB.x,
                endY = anchorLB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorRT, pointEnd = anchorLB, pointTarget = anchorLT
            ) > 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_LT, P_RT和P_LB形成的三角形内部
        if (pointSideLine(
                startX = anchorLT.x,
                startY = anchorLT.y,
                endX = anchorRT.x,
                endY = anchorRT.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLT, pointEnd = anchorRT, pointTarget = anchorLB
            ) < 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_LT, P_LB和P_RT形成的三角形内部
        if (pointSideLine(
                startX = anchorLT.x,
                startY = anchorLT.y,
                endX = anchorLB.x,
                endY = anchorLB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLT, pointEnd = anchorLB, pointTarget = anchorRT
            ) < 0
        ) {
            return false
        }
        // 如果点(x, y)通过了所有的检查，则可以移动
        return true
    }

    /**
     * 检查左下角的点是否可以移动到指定的位置
     * @param targetX 新的x坐标
     * @param targetY 新的y坐标
     * @return 如果可以移动则返回true，否则返回false
     */
    private fun canMoveLeftBottom(targetX: Float, targetY: Float): Boolean {
        // 检查点(x, y)是否在由P_LT, P_RB和P_RT形成的三角形内部
        val anchorLT = mCornerAnchorPoints?.get(Anchor.LT.position) ?: return false
        val anchorRT = mCornerAnchorPoints?.get(Anchor.RT.position) ?: return false
        val anchorRB = mCornerAnchorPoints?.get(Anchor.RB.position) ?: return false
        if (pointSideLine(
                startX = anchorLT.x,
                startY = anchorLT.y,
                endX = anchorRB.x,
                endY = anchorRB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLT, pointEnd = anchorRB, pointTarget = anchorRT
            ) > 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_LT, P_RT和P_RB形成的三角形内部
        if (pointSideLine(
                startX = anchorLT.x,
                startY = anchorLT.y,
                endX = anchorRT.x,
                endY = anchorRT.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorLT, pointEnd = anchorRT, pointTarget = anchorRB
            ) < 0
        ) {
            return false
        }
        // 检查点(x, y)是否在由P_RT, P_RB和P_LT形成的三角形内部
        if (pointSideLine(
                startX = anchorRT.x,
                startY = anchorRT.y,
                endX = anchorRB.x,
                endY = anchorRB.y,
                targetX = targetX,
                targetY = targetY
            ) * pointSideLine(
                pointStart = anchorRT, pointEnd = anchorRB, pointTarget = anchorLT
            ) < 0
        ) {
            return false
        }
        // 如果点(x, y)通过了所有的检查，则可以移动
        return true
    }

    /**
     * 获取拖动点的类型
     * @param dragPoint 拖动的点
     * @return 点的类型，如果未找到则返回null
     */
    private fun getPointType(dragPoint: PointF?): Anchor? {
        // 如果相关数组或拖动点为空，则返回null
        if (dragPoint == null) return null

        if (!checkAnchorPoints(mCornerAnchorPoints) && !checkAnchorPoints(mEdgeAnchorPoints)) return null
        // 遍历角点数组，查找拖动的点
        for (i in mCornerAnchorPoints!!.indices) {
            if (dragPoint == mCornerAnchorPoints!!.getOrNull(i)) {
                return Anchor.entries[i]
            }
        }
        if (!checkAnchorPoints(mEdgeAnchorPoints)) return null
        // 遍历边中点数组，查找拖动的点
        for (i in mEdgeAnchorPoints!!.indices) {
            if (dragPoint == mEdgeAnchorPoints!!.getOrNull(i)) {
                return Anchor.entries[4 + i]
            }
        }
        // 如果没有找到匹配的点，则返回null
        return null
    }

    /**
     * 更新边中点的位置
     */
    private fun updateMidEdgePoints() {
        // 如果显示边中点，则设置边中点
        if (mShowEdgeAnchorPoint) setEdgeMidPoints()
    }

    /**
     * 获取视图中点的X坐标
     * @return 点在视图中的X坐标
     */
    private val PointF.viewPointX: Float
        get() = x * mScaleX + mActLeft

    /**
     * 获取视图中点的Y坐标
     * @return 点在视图中的Y坐标
     */
    private val PointF.viewPointY: Float
        get() = y * mScaleY + mActTop

    /**
     * 使用默认点或根据提供的点进行裁剪
     * @param points 提供的点数组
     * @return 裁剪使用的点数组
     */
    public fun useDefaultPoint(points: Array<PointF>): Array<PointF> {
        // 计算Drawable区域的绝对值
        val drawableArea = abs(mActHeight * mActWidth)
        // 计算点的宽度
        val pointWidth = abs(points[0].viewPointX - points[1].viewPointX)
        // 计算点的高度
        val pointHeight = abs(points[0].viewPointY - points[3].viewPointY)
        // 计算点的面积
        val pointArea = pointHeight * pointWidth
        // 如果Drawable面积与点面积的比值大于8，则使用全图裁剪点
        if (drawableArea / pointArea > 8.0) {
            return getFullImageWindow()
        }
        // 否则，使用提供的点
        return points
    }

    fun setCropWindowChangeLister(service: ICropWindowChangeLister) {
        this.mCropService = service
    }

    companion object {
        /**
         * 触摸点捕捉到锚点的最小距离(dp)
         */
        private const val TOUCH_POINT_CATCH_DISTANCE = 15

        /**
         * 锚点绘制b半径(dp)
         */
        private const val ANCHOR_RADIUS = 6

        /**
         * 锚点点按效果半径(dp)
         */
        private const val DRAGGING_ANCHOR_RADIUS = 45

        /**
         * 放大镜十字宽度(dp)
         */
        private const val MAGNIFIER_CROSS_LINE_WIDTH = 0.8f

        /**
         * 放大镜十字长度(dp)
         */
        private const val MAGNIFIER_CROSS_LINE_LENGTH = 3

        /**
         * 放大镜边框宽度(dp)
         */
        private const val MAGNIFIER_BORDER_WIDTH = 2

    }


}