package com.liecoder.framework.global

import android.app.Activity
import android.app.Application
import android.content.SharedPreferences
import com.liecoder.framework.model.BuildType
import com.liecoder.framework.manager.LifecycleManager
import com.liecoder.framework.model.AppStatus
import com.liecoder.framework.service.IAppStatusService
import com.liecoder.framework.service.ISharedPreferencesService

/**
 * 应用程序助手对象，提供应用程序级别的辅助功能。
 * 实现了 [IAppStatusService] 接口，用以管理应用程序的状态。
 */
object AppHelper : IAppStatusService {

    // 私有属性，存储应用程序实例。
    private lateinit var _application: Application


    // 私有属性，存储当前的构建类型。
    private var _buildType: BuildType? = null


    // 私有属性，存储应用程序的当前状态。
    private var _appStatus: AppStatus? = null


    // 公开的只读属性，提供对应用程序实例的访问。
    val application: Application
        get() = _application


    // 公开的只读属性，提供对当前构建类型的访问。
    val envMode: BuildType?
        get() = _buildType


    // 公开的只读属性，判断当前是否为开发环境。
    val isDevelopment: Boolean
        get() = _buildType == BuildType.DEVELOPMENT


    // 公开的只读属性，判断当前是否为测试环境。
    val isTesting: Boolean
        get() = _buildType == BuildType.TESTING


    // 公开的只读属性，判断当前是否为生产环境。
    val isProduction: Boolean
        get() = _buildType == BuildType.PRODUCTION


    // 公开的只读属性，判断当前是否为前台状态。
    val isAppForeground: Boolean
        get() = _appStatus == AppStatus.Foreground


    // 公开的只读属性，判断当前是否为后台状态。
    val isAppBackground: Boolean
        get() = _appStatus == AppStatus.Background


    /**
     * 初始化应用程序助手。
     * @param application 应用程序实例。
     * @param buildType 应用程序的构建类型。
     */
    fun init(application: Application, buildType: BuildType) {
        _application = application
        _buildType = buildType
        LifecycleManager.register(application = application, mService = this)
    }


    /**
     * [IAppStatusService] 接口实现，用于处理应用程序状态变化。
     * @param appStatus 应用程序的新状态。
     * @param activity 触发状态变化的活动。
     */
    override fun onAppStatusChanged(appStatus: AppStatus, activity: Activity) {
        _appStatus = appStatus
    }


    /**
     * 创建并返回应用程序的 SharedPreferences 对象。
     * @return SharedPreferences 实例。
     */
    fun createSharedPreferences(): SharedPreferences = application.getSharedPreferences(
        (application as ISharedPreferencesService).getSharedPreferencesFileName(),
        (application as ISharedPreferencesService).getSharedPreferencesMode()
    )

}