package com.liecoder.framework.ktx

import android.animation.ObjectAnimator
import android.animation.ValueAnimator
import android.view.View
import android.view.animation.AlphaAnimation
import android.view.animation.Animation
import android.view.animation.CycleInterpolator
import android.view.animation.LinearInterpolator
import android.view.animation.TranslateAnimation

/**
 * Applies a shaking animation to a view.
 *
 * This function creates a horizontal shaking animation by translating the view back and forth.
 *
 * @param count The number of shake cycles the view should perform.
 * @return The View instance itself, allowing for chaining of method calls.
 */
fun <T : View> T.shaking(count: Int = 8): T {
    val animation = TranslateAnimation(
            0f,
            10f,
            0f,
            0f
    ).apply {
        duration = 1000
        interpolator = CycleInterpolator(count.toFloat())
    }
    startAnimation(animation)
    return this
}

/**
 * Applies a flickering animation to a view, making it fade in and out.
 *
 * This function creates an alpha animation that continuously fades the view in and out, creating a flickering effect.
 *
 * @return The View instance itself, allowing for chaining of method calls.
 */
fun <T : View> T.flickering(): T {
    val animation = AlphaAnimation(
            1f,
            0f
    ).apply {
        duration = 500
        interpolator = LinearInterpolator()
        repeatCount = Animation.INFINITE
        repeatMode = Animation.REVERSE
    }
    startAnimation(animation)
    return this
}

/**
 * Applies a beating animation to a view, making it move up and down.
 *
 * This function creates a 'beating' effect by repeatedly moving the view up and down along the Y-axis.
 *
 * @return The View instance itself, allowing for chaining of method calls.
 */
fun <T : View> T.beating(): T {
    val animator = ObjectAnimator.ofFloat(
            this,
            "translationY",
            0f,
            -100f
    )
    animator.repeatMode = ValueAnimator.REVERSE
    animator.repeatCount = ValueAnimator.INFINITE
    animator.interpolator = LinearInterpolator()
    animator.duration = 800
    animator.start()
    return this
}