package com.liecoder.framework.ktx

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onCompletion
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.onStart

/**
 * 启动一个倒计时协程，用于在给定的协程作用域内进行倒计时。
 *
 * @param scope [CoroutineScope] 指定在哪个协程作用域内启动倒计时。
 * @param onTick (Int) -> Unit 每次倒计时时调用的回调函数，参数为当前倒计时的值。
 * @param onStart (() -> Unit)? 可选的回调函数，当倒计时开始时调用。
 * @param onFinish (() -> Unit)? 可选的回调函数，当倒计时结束时调用。
 * @return Job 返回一个Job对象，用于控制倒计时协程的取消等操作。
 *
 * 该函数创建了一个Flow，用于每秒发射一次倒计时的当前值。通过flowOn将Flow的操作绑定到主线程，
 * 并通过onStart、onCompletion和onEach方法分别在倒计时开始、完成和每次发射值时调用相应的回调函数。
 * 最后，使用launchIn方法在指定的协程作用域内启动Flow。
 */
fun Int.countDown(
    scope: CoroutineScope,
    onTick: (Int) -> Unit,
    onStart: (() -> Unit)? = null,
    onFinish: (() -> Unit)? = null
): Job {
    // 创建一个 Flow，用于发射倒计时的值
    return flow {
        for (i in this@countDown downTo 0) {
            // 每隔一秒发射一个值
            emit(i)
            delay(1000)
        }
    }
        // 在主线程上执行 Flow 操作
        .flowOn(Dispatchers.Main)
        // 在 Flow 启动时执行 onStart 回调函数（如果有）
        .onStart { onStart?.invoke() }
        // 在 Flow 完成时执行 onFinish 回调函数（如果有）
        .onCompletion { onFinish?.invoke() }
        // 每次发射一个倒计时值时执行 onTick 回调函数
        .onEach { onTick.invoke(it) }
        // 在指定的协程作用域中启动 Flow
        .launchIn(scope)
}
