@file:JvmName("LiecoderGlide")

package com.liecoder.framework.ktx

import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.Drawable
import android.widget.ImageView
import com.bumptech.glide.Glide
import com.bumptech.glide.Priority
import com.bumptech.glide.RequestBuilder
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.load.engine.bitmap_recycle.BitmapPool
import com.bumptech.glide.load.resource.bitmap.CenterCrop
import com.bumptech.glide.load.resource.bitmap.CircleCrop
import com.bumptech.glide.load.resource.bitmap.RoundedCorners
import com.bumptech.glide.request.RequestOptions
import com.liecoder.framework.R

/**
 * 使用Glide库加载图片到ImageView，并提供一系列自定义选项。
 * 这个扩展函数支持多种图片资源类型，包括资源ID、URL字符串、Bitmap和Drawable对象。
 * 同时，提供了内存缓存跳过、占位图、错误图、加载优先级、圆角和边框等高级功能。
 *
 * @param imageResource 要加载的图片资源，可以是资源ID、URL、Bitmap或Drawable。
 * @param skipMemoryCache 是否跳过内存缓存，默认为true。
 * @param placeholder 加载过程中显示的占位图资源ID，默认为R.drawable.ic_glide_placeholder。
 * @param error 加载失败时显示的错误图资源ID，默认为R.drawable.ic_glide_error。
 * @param priority 加载优先级，默认为Priority.HIGH。
 * @param circleCrop 是否将图片裁剪为圆形，默认为false。
 * @param borderWidth 边框宽度，默认为0F。
 * @param borderColor 边框颜色，默认为Color.TRANSPARENT。
 * @param cornerRadius 圆角半径，默认为0。
 */
@JvmOverloads
fun ImageView.glide(
    imageResource: Any?,
    skipMemoryCache: Boolean = true,
    circleCrop: Boolean = false,
    borderWidth: Float = 0F,
    borderColor: Int = Color.TRANSPARENT,
    cornerRadius: Int = 0,
    priority: Priority = Priority.HIGH,
    dontAnimate: Boolean = false,
    placeholder: Any? = R.drawable.ic_glide_placeholder,
    error: Any? = R.drawable.ic_glide_error
) = try {
    val requestManager = Glide.with(context)
    when (imageResource) {
        is Int      -> requestManager.load(imageResource)
        is String   -> requestManager.load(imageResource)
        is Bitmap   -> requestManager.load(imageResource)
        is Drawable -> requestManager.load(imageResource)
        else        -> requestManager.load(imageResource)
    }
        .skipMemoryCache(skipMemoryCache)
        .diskCacheStrategy(if (skipMemoryCache) DiskCacheStrategy.NONE else DiskCacheStrategy.RESOURCE)
        .applyPlaceHolder(placeholder)
        .applyError(error)
        .priority(priority)
        .circleCrop(transform = circleCrop, borderWidth = borderWidth, borderColor = borderColor)
        .roundedCrop(transform = !circleCrop && cornerRadius > 0, radius = cornerRadius)
        .applyDontAnimate(dontAnimate)
        .into(this)
} catch (e: Exception) {
    setImageResource(R.drawable.ic_glide_error)
    e.printStackTrace()
}

/**
 * 应用圆角变换到Glide的请求构建器。
 * 如果不应用变换，则返回原始的请求构建器。
 *
 * @param transform 是否应用变换。
 * @param radius 圆角的半径。
 * @return 应用了变换的请求构建器或原始请求构建器。
 */
private fun RequestBuilder<*>.roundedCrop(transform: Boolean, radius: Int): RequestBuilder<*> =
    if (transform) transform(CenterCrop(), RoundedCorners(radius.dpInt)) else this

/**
 * 应用圆形裁剪和边框到Glide的请求构建器。
 * 如果不应用变换或者边框宽度小于等于0或颜色为透明，则返回原始的请求构建器。
 *
 * @param transform 是否应用变换。
 * @param borderWidth 边框的宽度。
 * @param borderColor 边框的颜色。
 * @return 应用了圆形裁剪和边框的请求构建器或原始请求构建器。
 */
private fun RequestBuilder<*>.circleCrop(
    transform: Boolean,
    borderWidth: Float,
    borderColor: Int,
): RequestBuilder<*> {
    if (!transform) return this
    if (borderWidth <= 0 || borderColor == Color.TRANSPARENT) {
        return apply(RequestOptions.circleCropTransform())
    }
    return transform(CircleBorderTransform(borderWidth, borderColor))
}

private fun RequestBuilder<*>.applyDontAnimate(dontAnimate: Boolean): RequestBuilder<*> {
    return if (dontAnimate) this.dontAnimate() else this
}

private fun RequestBuilder<*>.applyPlaceHolder(placeHolder: Any?): RequestBuilder<*> {
    if (placeHolder is Int) return this.placeholder(placeHolder)
    if (placeHolder is Drawable) return this.placeholder(placeHolder)
    if (placeHolder is Bitmap) {
        val placeholderDrawable = try {
            BitmapDrawable(Resources.getSystem(), placeHolder.ensureBitmapCanDraw())
        } catch (e: Exception) {
            ColorDrawable(Color.WHITE)
        }
        return this.placeholder(placeholderDrawable)
    }
    return this
}

private fun RequestBuilder<*>.applyError(error: Any?): RequestBuilder<*> {
    if (error is Int) return this.error(error)
    if (error is Drawable) return this.error(error)
    if (error is Bitmap) {
        val placeholderDrawable = try {
            BitmapDrawable(Resources.getSystem(), error.ensureBitmapCanDraw())
        } catch (e: Exception) {
            ColorDrawable(Color.WHITE)
        }
        return this.error(placeholderDrawable)
    }
    return this
}

/**
 * 用于CircleCrop的自定义变换，添加边框到圆形图片。
 */
internal class CircleBorderTransform(
    private val borderWidth: Float, private val borderColor: Int
) : CircleCrop() {

    private var borderPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        color = borderColor
        style = Paint.Style.STROKE
        strokeWidth = borderWidth
    }

    override fun transform(
        pool: BitmapPool, toTransform: Bitmap, outWidth: Int, outHeight: Int
    ): Bitmap {
        //因为继承自CircleCrop，并且CircleCrop是圆形，在这里获取的bitmap是裁剪后的圆形bitmap
        val transform = super.transform(pool, toTransform, outWidth, outHeight)
        val canvas = Canvas(transform)
        val halfWidth = (outWidth / 2).toFloat()
        val halfHeight = (outHeight / 2).toFloat()
        canvas.drawCircle(
            halfWidth, halfHeight, halfWidth.coerceAtMost(halfHeight) - borderWidth / 2, borderPaint
        )
        canvas.setBitmap(null)
        return transform
    }

}
