package com.liecoder.framework.ktx

import android.graphics.Color
import android.graphics.Typeface
import android.text.method.LinkMovementMethod
import android.widget.TextView

/**
 * 设置 TextView 的文字大小，并应用给定范围的 Span。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param scale 相对于当前文本大小的比例。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.sizeSpan(
        str: String = "",
        range: IntRange,
        scale: Float = 1.5f
): TextView {
    text = (str.ifEmpty { text }).toSizeSpan(
            range,
            scale
    )
    return this
}

/**
 * 设置 TextView 的文字大小，并应用给定范围的 Span。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param scale 相对于当前文本大小的比例。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.sizeSpan(
        str: String = "",
        range: IntRange,
        textSize: Int = 16
): TextView {
    text = (str.ifEmpty { text }).toSizeSpan(
            range,
            textSize
    )
    return this
}

/**
 * 设置 TextView 的文字大小，并应用给定范围的 Span。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param scale 相对于当前文本大小的比例。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendSizeSpan(
        str: String = "",
        scale: Float = 1.5f
): TextView {
    append(
            str.toSizeSpan(
                    0..str.length,
                    scale
            )
    )
    return this
}

/**
 * 设置 TextView 的文字大小，并应用给定范围的 Span。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param scale 相对于当前文本大小的比例。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendSizeSpan(
        str: String = "",
        textSize: Int
): TextView {
    append(
            str.toSizeSpan(
                    0..str.length,
                    textSize
            )
    )
    return this
}

/**
 * 向 TextView 追加文字，并设置追加文本的颜色。
 * @param str 要追加的文本，为空时追加当前文本。
 * @param range 应用 Span 的文本范围。
 * @param color 文本颜色，默认为红色。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.colorSpan(
        str: String = "",
        range: IntRange,
        color: Int = Color.RED
): TextView {
    text = (str.ifEmpty { text }).toColorSpan(
            range,
            color
    )
    return this
}

/**
 * 向 TextView 追加文字，并设置追加文本的颜色。
 * @param str 要追加的文本，为空时追加当前文本。
 * @param range 应用 Span 的文本范围。
 * @param color 文本颜色，默认为红色。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendColorSpan(
        str: String = "",
        color: Int = Color.RED
): TextView {
    append(
            str.toColorSpan(
                    0..str.length,
                    color
            )
    )
    return this
}

/**
 * 设置 TextView 的背景颜色，并应用给定范围的 Span。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param color 背景颜色，默认为红色。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.backgroundColorSpan(
        str: String = "",
        range: IntRange,
        color: Int = Color.RED
): TextView {
    text = (str.ifEmpty { text }).toBackgroundColorSpan(
            range,
            color
    )
    return this
}

/**
 * 设置 TextView 的背景颜色，并应用给定范围的 Span。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param color 背景颜色，默认为红色。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendBackgroundColorSpan(
        str: String = "",
        color: Int = Color.RED
): TextView {
    append(
            str.toBackgroundColorSpan(
                    0..str.length,
                    color
            )
    )
    return this
}

/**
 * 设置 TextView 的文本并添加删除线样式。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.strikeThroughtSpan(
        str: String = "",
        range: IntRange
): TextView {
    text = (str.ifEmpty { text }).toStrikeThroughtSpan(range)
    return this
}

/**
 * 设置 TextView 的文本并添加删除线样式。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendStrikeThroughtSpan(str: String = ""): TextView {
    append(str.toStrikeThroughtSpan(0..str.length))
    return this
}

/**
 * 向 TextView 追加文本，并为追加的文本添加点击事件。
 * @param str 要追加的文本，为空时追加当前文本。
 * @param range 应用 Span 的文本范围。
 * @param color 文本颜色，默认为红色。
 * @param isUnderlineText 是否显示下划线，默认为 false。
 * @param clickAction 点击文本时执行的 lambda 表达式。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.clickSpan(
        str: String = "",
        range: IntRange,
        color: Int = Color.RED,
        isUnderlineText: Boolean = false,
        clickAction: () -> Unit
): TextView {
    movementMethod = LinkMovementMethod.getInstance()
    highlightColor = Color.TRANSPARENT  // remove click bg color
    text = (str.ifEmpty { text }).toClickSpan(
            range,
            color,
            isUnderlineText,
            clickAction
    )
    return this
}

/**
 * 向 TextView 追加文本，并为追加的文本添加点击事件。
 * @param str 要追加的文本，为空时追加当前文本。
 * @param range 应用 Span 的文本范围。
 * @param color 文本颜色，默认为红色。
 * @param isUnderlineText 是否显示下划线，默认为 false。
 * @param clickAction 点击文本时执行的 lambda 表达式。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendClickSpan(
        str: String = "",
        color: Int = Color.RED,
        isUnderlineText: Boolean = false,
        clickAction: () -> Unit
): TextView {
    movementMethod = LinkMovementMethod.getInstance()
    highlightColor = Color.TRANSPARENT  // remove click bg color
    append(
            str.toClickSpan(
                    0..str.length,
                    color,
                    isUnderlineText,
                    clickAction
            )
    )
    return this
}

/**
 * 设置 TextView 的文本样式。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param style 文本样式，如粗体、斜体等，默认为粗体。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.styleSpan(
        str: String = "",
        range: IntRange,
        style: Int = Typeface.BOLD
): TextView {
    text = (str.ifEmpty { text }).toStyleSpan(
            range,
            style
    )
    return this
}

/**
 * 向 TextView 追加文本，并为追加的文本设置文本样式。
 * @param str 要设置的文本，为空时使用当前文本。
 * @param range 应用 Span 的文本范围。
 * @param style 文本样式，如粗体、斜体等，默认为粗体。
 * @return TextView 实例，方便链式调用。
 */
fun TextView.appendStyleSpan(
        str: String = "",
        style: Int = Typeface.BOLD
): TextView {
    append(
            str.toStyleSpan(
                    0..str.length,
                    style
            )
    )
    return this
}