package com.liecoder.framework.titlebar

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.text.TextUtils.TruncateAt
import android.view.Gravity
import android.view.View
import android.widget.TextView
import com.liecoder.framework.ktx.asTypeface
import com.liecoder.framework.ktx.dpInt
import com.liecoder.framework.ktx.spFloat

/**
 * 标题栏样式统一接口。
 *
 * 此接口定义了一组方法，用于创建标题栏相关的视图和获取标题栏的样式属性。
 */
interface ITitleBarStyleService {

    /**
     * 创建标题栏的中间标题文本视图。
     *
     * @param context 上下文对象。
     * @return 创建的 TextView 实例。
     */
    fun createTitleView(context: Context): TextView = TextView(context).apply {
        gravity = Gravity.CENTER_VERTICAL
        isFocusable = true
        isSingleLine = true
    }

    /**
     * 创建标题栏左侧视图。
     *
     * @param context 上下文对象。
     * @return 创建的 TextView 实例。
     */
    fun createLeftView(context: Context): TextView = TextView(context).apply {
        gravity = Gravity.CENTER_VERTICAL
        isFocusable = true
        isSingleLine = true
    }

    /**
     * 创建标题栏右侧视图。
     *
     * @param context 上下文对象。
     * @return 创建的 TextView 实例。
     */
    fun createRightView(context: Context): TextView = TextView(context).apply {
        gravity = Gravity.CENTER_VERTICAL
        isFocusable = true
        isSingleLine = true
    }

    /**
     * 创建标题栏分隔线视图。
     *
     * @param context 上下文对象。
     * @return 创建的 View 实例。
     */
    fun createLineView(context: Context): View = View(context)

    /**
     * 获取标题栏背景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏背景的 Drawable 对象，可能为 null。
     */
    fun getTitleBarBackground(context: Context): Drawable?

    /**
     * 获取标题栏左侧背景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏左侧背景的 Drawable 对象，可能为 null。
     */
    fun getLeftTitleBackground(context: Context): Drawable?

    /**
     * 获取标题栏右侧背景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏右侧背景的 Drawable 对象，可能为 null。
     */
    fun getRightTitleBackground(context: Context): Drawable?

    /**
     * 获取标题栏左侧前景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏左侧前景的 Drawable 对象，可能为 null。
     */
    fun getLeftTitleForeground(context: Context): Drawable? = null

    /**
     * 获取标题栏右侧前景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏右侧前景的 Drawable 对象，可能为 null。
     */
    fun getRightTitleForeground(context: Context): Drawable? = null

    /**
     * 获取返回按钮的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 返回按钮的 Drawable 对象，可能为 null。
     */
    fun getBackButtonDrawable(context: Context): Drawable?

    /**
     * 获取左侧标题文本的水平内边距。
     *
     * @param context 上下文对象。
     * @return 左侧标题文本的水平内边距大小。
     */
    fun getLeftHorizontalPadding(context: Context): Int = 16.dpInt

    /**
     * 获取中间标题文本的水平内边距。
     *
     * @param context 上下文对象。
     * @return 中间标题文本的水平内边距。
     */
    fun getTitleHorizontalPadding(context: Context): Int = 0

    /**
     * 获取右侧标题文本的水平内边距。
     *
     * @param context 上下文对象。
     * @return 右侧标题文本的水平内边距。
     */
    fun getRightHorizontalPadding(context: Context): Int = 16.dpInt

    /**
     * 获取标题栏子控件的垂直内边距。
     *
     * @param context 上下文对象。
     * @return 标题栏子控件的垂直内边距大小。
     */
    fun getChildVerticalPadding(context: Context): Int = 16.dpInt


    /**
     * 获取左侧标题文本。
     *
     * @param context 上下文对象。
     * @return 左侧标题文本。
     */
    fun getLeftTitle(context: Context): CharSequence? = ""


    /**
     * 获取中间标题文本。
     *
     * @param context 上下文对象。
     * @return 中间标题文本。
     */
    fun getTitle(context: Context): CharSequence? = ""

    /**
     * 获取右侧标题文本。
     *
     * @param context 上下文对象。
     * @return 右侧标题文本。
     */
    fun getRightTitle(context: Context): CharSequence? = ""

    /**
     * 获取左侧标题文本颜色的状态列表。
     *
     * @param context 上下文对象。
     * @return 左侧标题文本颜色的状态列表，可能为 null。
     */
    fun getLeftTitleColor(context: Context): ColorStateList?

    /**
     * 获取中间标题文本颜色的状态列表。
     *
     * @param context 上下文对象。
     * @return 中间标题文本颜色的状态列表，可能为 null。
     */
    fun getTitleColor(context: Context): ColorStateList?

    /**
     * 获取右侧标题文本颜色的状态列表。
     *
     * @param context 上下文对象。
     * @return 右侧标题文本颜色的状态列表，可能为 null。
     */
    fun getRightTitleColor(context: Context): ColorStateList?

    /**
     * 获取左侧标题文本的字号大小。
     *
     * @param context 上下文对象。
     * @return 左侧标题文本的字号大小。
     */
    fun getLeftTitleSize(context: Context): Float = 16.spFloat

    /**
     * 获取中间标题文本的字号大小。
     *
     * @param context 上下文对象。
     * @return 中间标题文本的字号大小。
     */
    fun getTitleSize(context: Context): Float = 16.spFloat

    /**
     * 获取右侧标题文本的字号大小。
     *
     * @param context 上下文对象。
     * @return 右侧标题文本的字号大小。
     */
    fun getRightTitleSize(context: Context): Float = 16.spFloat

    /**
     * 获取左侧标题文本的字体样式。
     *
     * @param context 上下文对象。
     * @param style 字体样式，如 Typeface.BOLD。
     * @return 左侧标题文本的字体样式。
     */
    fun getLeftTitleTypeface(context: Context, style: Int): Typeface? = style.asTypeface

    /**
     * 获取中间标题文本的字体样式。
     *
     * @param context 上下文对象。
     * @param style 字体样式，如 Typeface.BOLD。
     * @return 中间标题文本的字体样式。
     */
    fun getTitleTypeface(context: Context, style: Int): Typeface? = style.asTypeface

    /**
     * 获取右侧标题文本的字体样式。
     *
     * @param context 上下文对象。
     * @param style 字体样式，如 Typeface.BOLD。
     * @return 右侧标题文本的字体样式。
     */
    fun getRightTitleTypeface(context: Context, style: Int): Typeface? = style.asTypeface

    /**
     * 获取左侧标题文本的样式。
     *
     * @param context 上下文对象。
     * @return 左侧标题文本的样式。
     */
    fun getLeftTitleStyle(context: Context): Int = Typeface.NORMAL

    /**
     * 获取中间标题文本的样式。
     *
     * @param context 上下文对象。
     * @return 中间标题文本的样式。
     */
    fun getTitleStyle(context: Context): Int = Typeface.NORMAL

    /**
     * 获取右侧标题文本的样式。
     *
     * @param context 上下文对象。
     * @return 右侧标题文本的样式。
     */
    fun getRightTitleStyle(context: Context): Int = Typeface.NORMAL

    /**
     * 获取左侧图标的位置。
     *
     * @param context 上下文对象。
     * @return 左侧图标的位置。
     */
    fun getLeftIconGravity(context: Context): Int = Gravity.START

    /**
     * 获取中间图标的位置。
     *
     * @param context 上下文对象。
     * @return 中间图标的位置。
     */
    fun getTitleIconGravity(context: Context): Int = Gravity.END

    /**
     * 获取右侧图标的位置。
     *
     * @param context 上下文对象。
     * @return 右侧图标的位置。
     */
    fun getRightIconGravity(context: Context): Int = Gravity.END

    /**
     * 获取左侧图标的边距。
     *
     * @param context 上下文对象。
     * @return 左侧图标的边距。
     */
    fun getLeftIconPadding(context: Context): Int = 2.dpInt

    /**
     * 获取中间图标的边距。
     *
     * @param context 上下文对象。
     * @return 中间图标的边距。
     */
    fun getTitleIconPadding(context: Context): Int = 2.dpInt

    /**
     * 获取右侧图标的边距。
     *
     * @param context 上下文对象。
     * @return 右侧图标的边距。
     */
    fun getRightIconPadding(context: Context): Int = 2.dpInt

    /**
     * 获取左侧图标的宽度。
     *
     * @param context 上下文对象。
     * @return 左侧图标的宽度。
     */
    fun getLeftIconWidth(context: Context): Int = 0

    /**
     * 获取中间图标的宽度。
     *
     * @param context 上下文对象。
     * @return 中间图标的宽度。
     */
    fun getTitleIconWidth(context: Context): Int = 0

    /**
     * 获取右侧图标的宽度。
     *
     * @param context 上下文对象。
     * @return 右侧图标的宽度。
     */
    fun getRightIconWidth(context: Context): Int = 0

    /**
     * 获取左侧图标的高度。
     *
     * @param context 上下文对象。
     * @return 左侧图标的高度。
     */
    fun getLeftIconHeight(context: Context): Int = 0

    /**
     * 获取中间图标的高度。
     *
     * @param context 上下文对象。
     * @return 中间图标的高度。
     */
    fun getTitleIconHeight(context: Context): Int = 0

    /**
     * 获取右侧图标的高度。
     *
     * @param context 上下文对象。
     * @return 右侧图标的高度。
     */
    fun getRightIconHeight(context: Context): Int = 0

    /**
     * 获取左侧标题文本的溢出显示模式。
     *
     * @param context 上下文对象。
     * @return 左侧标题文本的溢出显示模式，可能为 null。
     */
    fun getLeftTitleOverflowMode(context: Context): TruncateAt? = null


    /**
     * 获取中间标题文本的溢出显示模式。
     *
     * @param context 上下文对象。
     * @return 中间标题文本的溢出显示模式，可能为 null。
     */
    fun getTitleOverflowMode(context: Context): TruncateAt? = TruncateAt.MARQUEE


    /**
     * 获取右侧标题文本的溢出显示模式。
     *
     * @param context 上下文对象。
     * @return 右侧标题文本的溢出显示模式，可能为 null。
     */
    fun getRightTitleOverflowMode(context: Context): TruncateAt? = null

    /**
     * 获取标题栏底部的线条是否可见。
     *
     * @param context 上下文对象。
     * @return 如果线条可见返回 true，否则返回 false。
     */
    fun isLineVisible(context: Context): Boolean = true

    /**
     * 获取标题栏底部的线条高度。
     *
     * @param context 上下文对象。
     * @return 标题栏底部的线条高度。
     */
    fun getLineSize(context: Context): Int = 0.5F.dpInt

    /**
     * 获取标题栏底部的线条背景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏底部的线条背景的 Drawable 对象。
     */
    fun getLineDrawable(context: Context): Drawable?

}