/*
 * Copyright 2012-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.mikerasch.logback.adapter.spring;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.spi.TurboFilterList;
import ch.qos.logback.classic.turbo.TurboFilter;
import ch.qos.logback.core.spi.FilterReply;
import io.github.mikerasch.logback.adapter.spring.ansi.AnsiOutput;
import io.github.mikerasch.logback.adapter.spring.config.DefaultLogbackConfiguration;
import io.github.mikerasch.logback.adapter.spring.config.LogbackConfigurator;
import org.slf4j.ILoggerFactory;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.helpers.SubstituteLoggerFactory;

public class LogbackLoggingSystem {
  private final DefaultLogbackConfiguration defaultLogbackConfiguration;

  public LogbackLoggingSystem() {
    defaultLogbackConfiguration = new DefaultLogbackConfiguration();
    AnsiOutput.configure();
  }

  public void loadDefaults() {
    LoggerContext loggerContext = getLoggerContext();
    stopAndReset(loggerContext);
    withLoggingSuppressed(() -> {
      LogbackConfigurator configurator = new LogbackConfigurator(loggerContext);
      defaultLogbackConfiguration.apply(configurator);
      loggerContext.setPackagingDataEnabled(true);
      loggerContext.start();
    });
  }

  private void withLoggingSuppressed(Runnable action) {
    TurboFilterList turboFilters = getLoggerContext().getTurboFilterList();
    turboFilters.add(SUPPRESS_ALL_FILTER);
    try {
      action.run();
    }
    finally {
      turboFilters.remove(SUPPRESS_ALL_FILTER);
    }
  }

  private void stopAndReset(LoggerContext loggerContext) {
    loggerContext.stop();
    loggerContext.reset();
  }

  private LoggerContext getLoggerContext() {
    ILoggerFactory factory = getLoggerFactory();
    return (LoggerContext) factory;
  }

  private ILoggerFactory getLoggerFactory() {
    ILoggerFactory factory = LoggerFactory.getILoggerFactory();
    while (factory instanceof SubstituteLoggerFactory) {
      try {
        Thread.sleep(50);
      }
      catch (InterruptedException ex) {
        Thread.currentThread().interrupt();
        throw new IllegalStateException("Interrupted while waiting for non-substitute logger factory", ex);
      }
      factory = LoggerFactory.getILoggerFactory();
    }
    return factory;
  }

  private static final TurboFilter SUPPRESS_ALL_FILTER = new TurboFilter() {

    @Override
    public FilterReply decide(Marker marker, ch.qos.logback.classic.Logger logger, Level level, String format,
        Object[] params, Throwable t) {
      return FilterReply.DENY;
    }
  };
}
