/*
 * Copyright 2012-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.github.mikerasch.logback.adapter.spring.rules;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.pattern.CompositeConverter;
import io.github.mikerasch.logback.adapter.spring.ansi.AnsiColor;
import io.github.mikerasch.logback.adapter.spring.ansi.AnsiElement;
import io.github.mikerasch.logback.adapter.spring.ansi.AnsiOutput;
import io.github.mikerasch.logback.adapter.spring.ansi.AnsiStyle;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

public class ColorConverter extends CompositeConverter<ILoggingEvent> {

  private static final Map<String, AnsiElement> ELEMENTS;

  static {
    Map<String, AnsiElement> ansiElements = new HashMap<>();
    Arrays.stream(AnsiColor.values())
        .filter((color) -> color != AnsiColor.DEFAULT)
        .forEach((color) -> ansiElements.put(color.name().toLowerCase(Locale.ROOT), color));
    ansiElements.put("faint", AnsiStyle.FAINT);
    ELEMENTS = Collections.unmodifiableMap(ansiElements);
  }

  private static final Map<Integer, AnsiElement> LEVELS;

  static {
    Map<Integer, AnsiElement> ansiLevels = new HashMap<>();
    ansiLevels.put(Level.ERROR_INTEGER, AnsiColor.RED);
    ansiLevels.put(Level.WARN_INTEGER, AnsiColor.YELLOW);
    LEVELS = Collections.unmodifiableMap(ansiLevels);
  }

  @Override
  protected String transform(ILoggingEvent event, String in) {
    AnsiElement color = ELEMENTS.get(getFirstOption());
    if (color == null) {
      // Assume highlighting
      color = LEVELS.get(event.getLevel().toInteger());
      color = (color != null) ? color : AnsiColor.GREEN;
    }
    return toAnsiString(in, color);
  }

  protected String toAnsiString(String in, AnsiElement element) {
    return AnsiOutput.toString(element, in);
  }

  public static String getName(AnsiElement element) {
    return ELEMENTS.entrySet()
        .stream()
        .filter((entry) -> entry.getValue().equals(element))
        .map(Map.Entry::getKey)
        .findFirst()
        .orElseThrow();
  }
}
