package com.monmonkeygroup.openapi.quote;

import com.monmonkeygroup.openapi.Config;
import com.monmonkeygroup.openapi.OpenApiException;
import com.monmonkeygroup.openapi.client.Client;
import com.monmonkeygroup.openapi.client.Profile;
import com.monmonkeygroup.openapi.protocol.ActionName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public class QuoteContext implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(QuoteContext.class);
    private final Config config;
    private final Client client;
    private final Map<String, Set<String>> subscriptions = new ConcurrentHashMap<>();

    public QuoteContext(Config config) {
        if (null == config.getQuoteAccessToken() || config.getQuoteAccessToken().isEmpty()) {
            throw new RuntimeException("don't has quoteAccessToken. please set quote access token");
        }

        Profile profile = new Profile(config.getQuoteUrl(), config.getUser(), config.getQuoteAccessToken());
        this.config = config;
        this.client = new Client(profile);
        this.client.setAfterConnected(this::afterReconnect);
        try {
            this.client.init();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void afterReconnect(Client client) {
        try {
            resubscribe();
        } catch (Exception e) {
            log.error("quote resubscribe error: {}", e.toString());
        }
    }

    private void resubscribe() throws OpenApiException {
        for (Map.Entry<String, Set<String>> entry : this.subscriptions.entrySet()) {
            doSubscribe(Collections.singletonList(entry.getKey()), new ArrayList<>(entry.getValue()), true);
        }
    }

    private void doSubscribe(List<String> symbolList, List<String> actionNameList, boolean isFirstPush) throws OpenApiException {
        SubscriptionRequest request = new SubscriptionRequest(symbolList, actionNameList, true, false, isFirstPush);
        this.client.doRequest(ActionName.QOT_SUBSCRIBE_REQUEST, request, SubscriptionReply.class);
    }

    /**
     * Subscribe
     *
     * @param symbolList     Security symbols
     * @param actionNameList List of {@link SubActionName}
     * @param isFirstPush    After subscription,Immediately push the latest data once
     * @throws OpenApiException If an error occurs
     */
    public void subscribe(List<String> symbolList, List<String> actionNameList, boolean isFirstPush) throws OpenApiException {
        doSubscribe(symbolList, actionNameList, isFirstPush);

        symbolList.forEach(symbol -> this.subscriptions.computeIfAbsent(
                symbol, k -> ConcurrentHashMap.newKeySet()).addAll(actionNameList));
    }

    /**
     * Unubscribe
     *
     * @param unsubAll       One-click cancel all subscriptions currently connected, and ignore other parameters when it is set to true
     * @param symbolList     Security symbols
     * @param actionNameList List of {@link SubActionName}
     * @throws OpenApiException If an error occurs
     */
    public void unsubscribe(boolean unsubAll, List<String> symbolList, List<String> actionNameList) throws OpenApiException {
        SubscriptionRequest request = new SubscriptionRequest(symbolList, actionNameList,
                false, unsubAll, false);
        this.client.doRequest(ActionName.QOT_SUBSCRIBE_REQUEST, request, SubscriptionReply.class);

        if (unsubAll) {
            this.subscriptions.clear();
        } else {
            symbolList.forEach(symbol -> {
                Set<String> set = this.subscriptions.get(symbol);
                if (null == set) return;
                actionNameList.forEach(set::remove);
            });
        }
    }

    /**
     * Get subscription information
     *
     * @param reqAllConn Whether to return the subscription information of all connections.
     * @return {@link SubscriptionInfoReply}
     * @throws OpenApiException If an error occurs
     */
    public SubscriptionInfoReply getSubscriptionInfo(boolean reqAllConn) throws OpenApiException {
        SubscriptionInfoRequest request = new SubscriptionInfoRequest(reqAllConn);
        SubscriptionInfoReply reply = this.client.doRequest(ActionName.QOT_SUBSCRIPTION_INFO_REQUEST, request, SubscriptionInfoReply.class);
        return reply;
    }

    /**
     * Set quote callback
     *
     * @param handler A quote handler
     */
    public void setOnQuote(QuoteHandler handler) {
        if (null == handler) return;
        this.client.addPushHandler(ActionName.QOT_QUOTE_PUSH, packet -> {
            try {
                PushQuote update = packet.deserialize(PushQuote.class);
                handler.onQuote(update);
            } catch (Exception e) {
                log.error("quote update,unmarshal error: {}", e.toString());
            }
        });
    }

    /**
     * Set depth callback
     *
     * @param handler A depth handler
     */
    public void setOnDepth(DepthHandler handler) {
        if (null == handler) return;
        this.client.addPushHandler(ActionName.QOT_DEPTH_PUSH, packet -> {
            try {
                PushDepth update = packet.deserialize(PushDepth.class);
                handler.onDepth(update);
            } catch (Exception e) {
                log.error("depth update,unmarshal error: {}", e.toString());
            }
        });
    }

    /**
     * Set broker queue callback
     *
     * @param handler A broker queue handler
     */
    public void setOnBrokerQueue(BrokerQueueHandler handler) {
        if (null == handler) return;
        this.client.addPushHandler(ActionName.QOT_BROKER_QUEUE_PUSH, packet -> {
            try {
                PushBrokerQueue update = packet.deserialize(PushBrokerQueue.class);
                handler.onBrokerQueue(update);
            } catch (Exception e) {
                log.error("broker queue update,unmarshal error: {}", e.toString());
            }
        });
    }

    /**
     * Set trade callback
     *
     * @param handler A trade handler
     */
    public void setOnTrade(TradeHandler handler) {
        if (null == handler) return;
        this.client.addPushHandler(ActionName.QOT_TRADE_PUSH, packet -> {
            try {
                PushTrade update = packet.deserialize(PushTrade.class);
                handler.onTrade(update);
            } catch (Exception e) {
                log.error("trade update,unmarshal error: {}", e.toString());
            }
        });
    }

    /**
     * Get static information of securities
     *
     * @param symbolList Security symbols
     * @return list of {@link SecurityStaticInfo}
     * @throws OpenApiException If an error occurs
     */
    public List<SecurityStaticInfo> getStaticInfo(List<String> symbolList) throws OpenApiException {
        SecurityStaticInfoRequest request = new SecurityStaticInfoRequest(symbolList);
        SecurityStaticInfoReply reply = this.client.doRequest(ActionName.QOT_SECURITY_STATIC_INFO_REQUEST, request, SecurityStaticInfoReply.class);
        return reply.getStaticInfoList();
    }

    /**
     * Get quote of securities
     *
     * @param symbolList Security symbols
     * @return list of {@link SecurityQuote}
     * @throws OpenApiException If an error occurs
     */
    public List<SecurityQuote> getQuote(List<String> symbolList) throws OpenApiException {
        SecurityQuoteRequest request = new SecurityQuoteRequest(symbolList);
        SecurityQuoteReply reply = client.doRequest(ActionName.QOT_QUOTE_REQUEST, request, SecurityQuoteReply.class);
        return reply.getSecurityQuoteList();
    }

    /**
     * Get security trades
     *
     * @param symbol Security symbol
     * @return list of {@link Trade}
     * @throws OpenApiException If an error occurs
     */
    public List<Trade> getTrade(String symbol) throws OpenApiException {
        TradeRequest request = new TradeRequest(symbol);
        TradeReply reply = client.doRequest(ActionName.QOT_TRADE_REQUEST, request, TradeReply.class);
        return reply.getTradeList();
    }

    /**
     * Get market depth
     *
     * @param symbol Security symbol
     * @return {@link MarketDepth}
     * @throws OpenApiException If an error occurs
     */
    public MarketDepth getDepth(String symbol) throws OpenApiException {
        DepthRequest request = new DepthRequest(symbol);
        return client.doRequest(ActionName.QOT_DEPTH_REQUEST, request, DepthReply.class);
    }

    /**
     * Get security broker queue
     *
     * @param symbol Security symbol
     * @return {@link BrokerQueue}
     * @throws OpenApiException If an error occurs
     */
    public BrokerQueue getBroker(String symbol) throws OpenApiException {
        BrokerQueueRequest request = new BrokerQueueRequest(symbol);
        return client.doRequest(ActionName.QOT_BROKER_QUEUE_REQUEST, request, BrokerQueueReply.class);
    }

    /**
     * Get participant IDs
     *
     * @return list of {@link ParticipantInfo}
     * @throws OpenApiException If an error occurs
     */
    public List<ParticipantInfo> getParticipants() throws OpenApiException {
        ParticipantInfoRequest request = new ParticipantInfoRequest();
        ParticipantInfoReply reply = client.doRequest(ActionName.QOT_PARTICIPANT_INFO_REQUEST, request, ParticipantInfoReply.class);
        return reply.getList();
    }

    /**
     * Get security timeshares
     *
     * @param symbol  Security symbol
     * @param section {@link Constants.SectionType} section type
     * @return list of {@link TimeShare}
     * @throws OpenApiException If an error occurs
     */
    public List<TimeShare> getTimeShare(String symbol, int section) throws OpenApiException {
        TimeShareRequest request = new TimeShareRequest(symbol, section);
        TimeShareReply reply = client.doRequest(ActionName.QOT_TIMESHARE_REQUEST, request, TimeShareReply.class);
        return reply.getTimeShareList();
    }

    /**
     * Get security kbars
     *
     * @param symbol    Security symbol
     * @param period    {@link Constants.KbarPeriod} period of kbar
     * @param rehabType {@link Constants.RehabType} types of adjustment
     * @param maxNum    The number of kbar requested
     * @return list of {@link Kbar}
     * @throws OpenApiException If an error occurs
     */
    public List<Kbar> getKbar(String symbol, int period, int rehabType, int maxNum) throws OpenApiException {
        KbarRequest request = new KbarRequest(symbol, period, rehabType, maxNum);
        KbarReply reply = client.doRequest(ActionName.QOT_KBAR_REQUEST, request, KbarReply.class);
        return reply.getKbarList();
    }

    /**
     * Get security kbars
     *
     * @param symbol    Security symbol
     * @param period    {@link Constants.KbarPeriod} period of kbar
     * @param rehabType {@link Constants.RehabType} types of adjustment
     * @param beginTime begin time (Format: yyyy-MM-dd)
     * @param endTime   end time (Format: yyyy-MM-dd)
     * @return list of {@link Kbar}
     * @throws OpenApiException If an error occurs
     */
    public List<Kbar> getHistoryKbar(String symbol, int period, int rehabType, String beginTime, String endTime) throws OpenApiException {
        HistoryKbarRequest request = new HistoryKbarRequest(symbol, period, rehabType, beginTime, endTime);
        HistoryKbarReply reply = client.doRequest(ActionName.QOT_HISTORY_KBAR_REQUEST, request, HistoryKbarReply.class);
        return reply.getKbarList();
    }

    /**
     * Get all market states
     *
     * @return list of {@link MarketInfo}
     * @throws OpenApiException If an error occurs
     */
    public List<MarketInfo> getMarketInfo() throws OpenApiException {
        MarketInfoRequest request = new MarketInfoRequest();
        MarketInfoReply reply = client.doRequest(ActionName.QOT_MARKET_INFO_REQUEST, request, MarketInfoReply.class);
        return reply.getMarketInfoList();
    }

    /**
     * Get calculate index of securities
     *
     * @param symbolList Security symbols
     * @return list of {@link SecurityCalcIndex}
     * @throws OpenApiException If an error occurs
     */
    public List<SecurityCalcIndex> getSecurityCalcIndex(List<String> symbolList) throws OpenApiException {
        SecurityCalcIndexRequest request = new SecurityCalcIndexRequest(symbolList);
        SecurityCalcIndexReply reply = client.doRequest(ActionName.QOT_SECURITY_CALC_INDEX_REQUEST, request, SecurityCalcIndexReply.class);
        return reply.getCalcIndexList();
    }

    /**
     * Get real-time price of securities
     *
     * @param symbolList Security symbols
     * @return list of {@link RealTimePrice}
     * @throws OpenApiException If an error occurs
     */
    public List<RealTimePrice> getRealTimePrice(List<String> symbolList) throws OpenApiException {
        RealTimeFullPriceRequest request = new RealTimeFullPriceRequest(symbolList);
        RealTimeFullPriceReply reply = client.doRequest(ActionName.QOT_REAL_TIME_FULL_PRICE_REQUEST, request, RealTimeFullPriceReply.class);
        return reply.getList();
    }

    public List<SecurityName> getSecurityList(String market, boolean availableTrading) throws OpenApiException {
        SecurityListRequest request = new SecurityListRequest(market, availableTrading);
        SecurityListReply reply = client.doRequest(ActionName.QOT_SECURITY_LIST_REQUEST, request, SecurityListReply.class);
        return reply.getSecurityList();
    }

    public Config getConfig() {
        return config;
    }


    @Override
    public void close() {
        this.client.close();
    }
}
