/*
 * Decompiled with CFR 0.152.
 */
package robaho.net.httpserver.http2;

import com.sun.net.httpserver.Headers;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.util.EnumSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;
import robaho.net.httpserver.NoSyncBufferedOutputStream;
import robaho.net.httpserver.OptimizedHeaders;
import robaho.net.httpserver.http2.HTTP2Connection;
import robaho.net.httpserver.http2.HTTP2ErrorCode;
import robaho.net.httpserver.http2.HTTP2Exception;
import robaho.net.httpserver.http2.frame.BaseFrame;
import robaho.net.httpserver.http2.frame.DataFrame;
import robaho.net.httpserver.http2.frame.FrameFlag;
import robaho.net.httpserver.http2.frame.FrameHeader;
import robaho.net.httpserver.http2.frame.FrameType;
import robaho.net.httpserver.http2.frame.ResetStreamFrame;
import robaho.net.httpserver.http2.frame.SettingIdentifier;
import robaho.net.httpserver.http2.frame.SettingParameter;
import robaho.net.httpserver.http2.frame.WindowUpdateFrame;
import robaho.net.httpserver.http2.hpack.HPackContext;

public class HTTP2Stream {
    private final int streamId;
    final AtomicLong sendWindow = new AtomicLong(65535L);
    private final HTTP2Connection connection;
    private final System.Logger logger;
    private final OutputStream outputStream;
    private final Pipe pipe;
    private final HTTP2Connection.StreamHandler handler;
    private final Headers requestHeaders;
    private final Headers responseHeaders = new OptimizedHeaders(16);
    private final AtomicBoolean headersSent = new AtomicBoolean(false);
    private volatile Thread thread;
    private volatile boolean streamOpen = true;
    private volatile boolean halfClosed = false;
    private long dataInSize = 0L;

    public HTTP2Stream(int streamId, HTTP2Connection connection, Headers requestHeaders, HTTP2Connection.StreamHandler handler) throws IOException {
        this.streamId = streamId;
        this.connection = connection;
        this.logger = connection.logger;
        this.requestHeaders = requestHeaders;
        this.handler = handler;
        this.pipe = new Pipe();
        this.outputStream = new NoSyncBufferedOutputStream(new Http2OutputStream(streamId));
        SettingParameter setting = connection.getRemoteSettings().get(SettingIdentifier.SETTINGS_INITIAL_WINDOW_SIZE);
        if (setting != null) {
            this.sendWindow.addAndGet((int)(setting.value - 65535L));
        }
        this.logger.log(System.Logger.Level.TRACE, () -> "new stream, window size " + this.sendWindow.get() + " on stream " + streamId);
    }

    public OutputStream getOutputStream() {
        return this.outputStream;
    }

    public Headers getRequestHeaders() {
        return this.requestHeaders;
    }

    public Headers getResponseHeaders() {
        return this.responseHeaders;
    }

    public String toString() {
        return this.connection.toString() + ", stream " + this.streamId;
    }

    public boolean isOpen() {
        return this.streamOpen;
    }

    public boolean isHalfClosed() {
        return this.halfClosed;
    }

    public void close() {
        this.streamOpen = false;
        this.halfClosed = true;
        if (this.connection.http2Streams.put(this.streamId, null) == null) {
            return;
        }
        try {
            this.pipe.close();
            this.outputStream.close();
            if (this.thread != null) {
                this.thread.interrupt();
            }
        }
        catch (IOException e) {
            if (!this.connection.isClosed()) {
                this.connection.close();
                this.logger.log(this.connection.httpConnection.requestCount.get() > 0L ? System.Logger.Level.WARNING : System.Logger.Level.DEBUG, "IOException closing http2 stream", (Throwable)e);
            }
        }
    }

    public void processFrame(BaseFrame frame) throws HTTP2Exception, IOException {
        switch (frame.getHeader().getType()) {
            case HEADERS: 
            case CONTINUATION: {
                if (this.halfClosed) {
                    throw new HTTP2Exception(HTTP2ErrorCode.STREAM_CLOSED);
                }
                this.performRequest(frame.getHeader().getFlags().contains((Object)FrameFlag.END_STREAM));
                break;
            }
            case DATA: {
                DataFrame dataFrame = (DataFrame)frame;
                this.logger.log(System.Logger.Level.TRACE, "received data frame, length " + dataFrame.body.length + " on stream " + this.streamId);
                if (this.halfClosed) {
                    throw new HTTP2Exception(HTTP2ErrorCode.STREAM_CLOSED);
                }
                if (!this.streamOpen) {
                    throw new HTTP2Exception(HTTP2ErrorCode.PROTOCOL_ERROR);
                }
                this.pipe.getOutputStream().write(dataFrame.body);
                this.logger.log(System.Logger.Level.TRACE, "wrote data frame to pipe, length " + dataFrame.body.length + " on stream " + this.streamId);
                this.dataInSize += (long)dataFrame.body.length;
                if (!dataFrame.getHeader().getFlags().contains((Object)FrameFlag.END_STREAM)) break;
                if (this.requestHeaders.containsKey("Content-length") && this.dataInSize != Long.parseLong(this.requestHeaders.getFirst("Content-length"))) {
                    this.connection.sendResetStream(HTTP2ErrorCode.PROTOCOL_ERROR, this.streamId);
                    this.close();
                    break;
                }
                this.pipe.closeOutput();
                this.halfClosed = true;
                break;
            }
            case PRIORITY: {
                if (!this.streamOpen) break;
                throw new HTTP2Exception(HTTP2ErrorCode.PROTOCOL_ERROR);
            }
            case RST_STREAM: {
                ResetStreamFrame resetFrame = (ResetStreamFrame)frame;
                this.logger.log(System.Logger.Level.DEBUG, "received reset stream " + String.valueOf((Object)resetFrame.errorCode) + ", on stream " + this.streamId);
                this.close();
                break;
            }
            case WINDOW_UPDATE: {
                int windowSizeIncrement = ((WindowUpdateFrame)frame).getWindowSizeIncrement();
                if (this.sendWindow.addAndGet(windowSizeIncrement) > Integer.MAX_VALUE) {
                    this.connection.sendResetStream(HTTP2ErrorCode.FLOW_CONTROL_ERROR, this.streamId);
                    this.close();
                }
                this.logger.log(System.Logger.Level.DEBUG, "received window update " + windowSizeIncrement + ", new size " + this.sendWindow.get() + ", on stream " + this.streamId);
                break;
            }
        }
    }

    private void performRequest(boolean halfClosed) throws IOException {
        InputStream in;
        this.connection.httpConnection.requestCount.incrementAndGet();
        this.connection.requestsInProgress.incrementAndGet();
        InputStream inputStream = in = halfClosed ? InputStream.nullInputStream() : this.pipe.getInputStream();
        if (halfClosed) {
            this.halfClosed = true;
            this.pipe.closeOutput();
        }
        this.handler.getExecutor().execute(() -> {
            this.thread = Thread.currentThread();
            try {
                this.handler.handleStream(this, in, this.outputStream);
            }
            catch (IOException ex) {
                this.close();
            }
        });
    }

    public void writeResponseHeaders() throws IOException {
        if (!this.headersSent.compareAndSet(false, true)) {
            return;
        }
        this.connection.lock();
        try {
            HPackContext.writeHeaderFrame(this.responseHeaders, this.connection.outputStream, this.streamId);
        }
        finally {
            this.connection.unlock();
        }
    }

    public InetSocketAddress getLocalAddress() {
        return this.connection.getLocalAddress();
    }

    public InetSocketAddress getRemoteAddress() {
        return this.connection.getRemoteAddress();
    }

    private static class Pipe {
        private final CustomPipedInputStream inputStream = new CustomPipedInputStream();
        private final CustomPipedOutputStream outputStream = new CustomPipedOutputStream(this.inputStream);

        public InputStream getInputStream() {
            return this.inputStream;
        }

        public OutputStream getOutputStream() {
            return this.outputStream;
        }

        public void close() throws IOException {
            this.inputStream.close();
            this.outputStream.close();
        }

        public void closeOutput() throws IOException {
            this.outputStream.close();
        }
    }

    class Http2OutputStream
    extends OutputStream {
        private static final EnumSet<FrameFlag> END_STREAM = EnumSet.of(FrameFlag.END_STREAM);
        private final int streamId;
        private final int max_frame_size;
        private boolean closed;
        private long pauses = 0L;

        public Http2OutputStream(int streamId) {
            this.streamId = streamId;
            SettingParameter setting = HTTP2Stream.this.connection.getRemoteSettings().get(SettingIdentifier.SETTINGS_MAX_FRAME_SIZE);
            this.max_frame_size = setting != null ? (int)setting.value : 16384;
        }

        @Override
        public void write(int b) throws IOException {
            this.write(new byte[]{(byte)b});
        }

        @Override
        public void write(byte[] b) throws IOException {
            this.write(b, 0, b.length);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            while (HTTP2Stream.this.sendWindow.get() <= 0L && !HTTP2Stream.this.connection.isClosed()) {
                ++this.pauses;
                LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(1L));
            }
            HTTP2Stream.this.writeResponseHeaders();
            while (len > 0) {
                int _len = Math.min(Math.min(len, this.max_frame_size), (int)Math.min(HTTP2Stream.this.connection.sendWindow.get(), HTTP2Stream.this.sendWindow.get()));
                if (_len <= 0) {
                    ++this.pauses;
                    HTTP2Stream.this.connection.lock();
                    try {
                        HTTP2Stream.this.connection.outputStream.flush();
                    }
                    finally {
                        HTTP2Stream.this.connection.unlock();
                    }
                    LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(1L));
                    if (HTTP2Stream.this.connection.isClosed()) {
                        throw new IOException("connection closed");
                    }
                    int remaining = len;
                    continue;
                }
                if (HTTP2Stream.this.connection.sendWindow.addAndGet(-_len) < 0L) {
                    HTTP2Stream.this.connection.sendWindow.addAndGet(_len);
                    continue;
                }
                HTTP2Stream.this.connection.lock();
                try {
                    FrameHeader.writeTo(HTTP2Stream.this.connection.outputStream, _len, FrameType.DATA, FrameFlag.NONE, this.streamId);
                    HTTP2Stream.this.connection.outputStream.write(b, off, _len);
                }
                finally {
                    HTTP2Stream.this.connection.unlock();
                }
                off += _len;
                len -= _len;
                HTTP2Stream.this.sendWindow.addAndGet(-_len);
                HTTP2Stream.this.logger.log(System.Logger.Level.TRACE, () -> "sent data frame, length " + _len + ", new send window " + HTTP2Stream.this.sendWindow.get() + " on stream " + this.streamId);
            }
        }

        @Override
        public void flush() throws IOException {
        }

        @Override
        public void close() throws IOException {
            if (this.closed) {
                return;
            }
            if (this.pauses > 0L) {
                HTTP2Stream.this.logger.log(System.Logger.Level.INFO, () -> "sending stream window exhausted " + this.pauses + " on stream " + this.streamId);
            }
            try {
                if (HTTP2Stream.this.connection.isClosed()) {
                    if (!HTTP2Stream.this.headersSent.get()) {
                        HTTP2Stream.this.logger.log(System.Logger.Level.WARNING, "stream connection is closed and headers not sent on stream " + this.streamId);
                    }
                    return;
                }
                HTTP2Stream.this.connection.requestsInProgress.decrementAndGet();
                HTTP2Stream.this.writeResponseHeaders();
                HTTP2Stream.this.connection.lock();
                try {
                    FrameHeader.writeTo(HTTP2Stream.this.connection.outputStream, 0, FrameType.DATA, END_STREAM, this.streamId);
                    if (HTTP2Stream.this.connection.requestsInProgress.get() <= 0) {
                        HTTP2Stream.this.connection.outputStream.flush();
                    }
                }
                finally {
                    HTTP2Stream.this.connection.unlock();
                }
            }
            finally {
                this.closed = true;
                HTTP2Stream.this.close();
            }
        }
    }

    private static class CustomPipedOutputStream
    extends OutputStream {
        private final CustomPipedInputStream inputStream;
        private boolean closed = false;

        public CustomPipedOutputStream(CustomPipedInputStream inputStream) {
            this.inputStream = inputStream;
        }

        @Override
        public void write(int b) throws IOException {
            this.write(new byte[]{(byte)b});
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.inputStream.lock.lock();
            try {
                while (len > 0) {
                    while ((this.inputStream.writePos == this.inputStream.readPos - 1 || this.inputStream.writePos == this.inputStream.buffer.length - 1 && this.inputStream.readPos == 0) && !this.closed) {
                        try {
                            this.inputStream.notFull.await();
                        }
                        catch (InterruptedException e) {
                            throw new IOException("Interrupted while waiting for buffer space", e);
                        }
                    }
                    if (this.closed) {
                        throw new IOException("Stream closed");
                    }
                    int space = this.inputStream.readPos <= this.inputStream.writePos ? this.inputStream.buffer.length - this.inputStream.writePos : this.inputStream.readPos - this.inputStream.writePos - 1;
                    int bytesToWrite = Math.min(len, space);
                    System.arraycopy(b, off, this.inputStream.buffer, this.inputStream.writePos, bytesToWrite);
                    this.inputStream.writePos += bytesToWrite;
                    if (this.inputStream.writePos == this.inputStream.buffer.length) {
                        this.inputStream.writePos = 0;
                    }
                    off += bytesToWrite;
                    len -= bytesToWrite;
                    this.inputStream.notEmpty.signal();
                }
            }
            finally {
                this.inputStream.lock.unlock();
            }
        }

        @Override
        public void close() throws IOException {
            this.inputStream.lock.lock();
            try {
                this.closed = true;
                this.inputStream.close();
            }
            finally {
                this.inputStream.lock.unlock();
            }
        }
    }

    private static class CustomPipedInputStream
    extends InputStream {
        private final byte[] buffer = new byte[1024];
        private int readPos = 0;
        private int writePos = 0;
        private boolean closed = false;
        private final Lock lock = new ReentrantLock();
        private final Condition notEmpty = this.lock.newCondition();
        private final Condition notFull = this.lock.newCondition();

        private CustomPipedInputStream() {
        }

        @Override
        public int read() throws IOException {
            this.lock.lock();
            try {
                while (this.readPos == this.writePos && !this.closed) {
                    try {
                        this.notEmpty.await();
                    }
                    catch (InterruptedException e) {
                        throw new IOException("Interrupted while waiting for data", e);
                    }
                }
                if (this.closed && this.readPos == this.writePos) {
                    int e = -1;
                    return e;
                }
                int result = this.buffer[this.readPos++] & 0xFF;
                if (this.readPos == this.buffer.length) {
                    this.readPos = 0;
                }
                this.notFull.signal();
                int n = result;
                return n;
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            int bytesRead = 0;
            while (bytesRead < len) {
                int byteRead = this.read();
                if (byteRead == -1) {
                    return bytesRead == 0 ? -1 : bytesRead;
                }
                b[off + bytesRead++] = (byte)byteRead;
            }
            return bytesRead;
        }

        @Override
        public void close() throws IOException {
            this.lock.lock();
            try {
                this.closed = true;
                this.notEmpty.signalAll();
                this.notFull.signalAll();
            }
            finally {
                this.lock.unlock();
            }
        }
    }
}

