;; Copyright 2013 the original author or authors.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;      http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns vertx.http.sockjs
  "This is an implementation of the server side part of https://github.com/sockjs.

  SockJS enables browsers to communicate with the server using a
  simple WebSocket-like api for sending and receiving messages. Under
  the bonnet SockJS chooses to use one of several protocols depending
  on browser capabilities and what appears to be working across the
  network.

  Available protocols include:

  * WebSockets
  * xhr-polling
  * xhr-streaming
  * json-polling
  * event-source
  * html-file

  This means it should just work irrespective of what browser
  is being used, and whether there are nasty things like proxies and
  load balancers between the client and the server.

  For more detailed information on SockJS, see their website.

  On the server side, you interact using instances of SockJSSocket -
  this allows you to send data to the client or receive data via
  vertx.stream/on-data.

  You can register multiple applications with the same SockJSServer,
  each using different path prefixes, each application will have its
  own handler, and configuration."
  (:require [clojure.string :as string]
            [vertx.utils :as u]
            [vertx.core :as core])
  (:import [org.vertx.java.core.sockjs EventBusBridgeHook]))

(defn sockjs-server
  "Create a SockJS server that wraps an HTTP server."
  [http-server]
  (-> (core/get-vertx) (.createSockJSServer http-server)))

(defn install-app
  "Installs a SockJS application.
   When the server receives a SockJS request that matches the
   configured :prefix, it creates an instance of SockJSSocket and
   passes it to the connect handler. handler can either be a
   single-arity fn or a Handler instance that will be passed the
   socket. Returns the server instance.

   config can contain the following values:

   * :prefix - A url prefix for the application. All http requests whose
     paths begins with selected prefix will be handled by the
     application. This property is mandatory.
   * :insert_JSESSIONID - Some hosting providers enable sticky sessions
     only to requests that have JSESSIONID cookie set. This setting
     controls if the server should set this cookie to a dummy value. By
     default setting JSESSIONID cookie is enabled. More sophisticated
     behaviour can be achieved by supplying a function.
   * :session_timeout - The server sends a close event when a client
     receiving connection have not been seen for a while. This delay is
     configured by this setting. By default the close event will be
     emitted when a receiving connection wasn't seen for 5 seconds.
   * :heartbeat_period - In order to keep proxies and load balancers
     from closing long running http requests we need to pretend that the
     connecion is active and send a heartbeat packet once in a
     while. This setting controlls how often this is done. By default a
     heartbeat packet is sent every 5 seconds.
   * :max_bytes_streaming - Most streaming transports save responses on
     the client side and don't free memory used by delivered
     messages. Such transports need to be garbage-collected once in a
     while. :max_bytes_streaming sets a minimum number of bytes that can
     be send over a single http streaming request before it will be
     closed. After that client needs to open new request. Setting this
     value to one effectively disables streaming and will make streaming
     transports to behave like polling transports. The default value is
     128K.
   * :library_url - Transports which don't support cross-domain
     communication natively ('eventsource' to name one) use an iframe
     trick. A simple page is served from the SockJS server (using its
     foreign domain) and is placed in an invisible iframe. Code run from
     this iframe doesn't need to worry about cross-domain issues, as it's
     being run from domain local to the SockJS server. This iframe also
     does need to load SockJS javascript client library, and this option
     lets you specify its url (if you're unsure, point it to the latest
     minified SockJS client release, this is the default). The default
     value is http://cdn.sockjs.org/sockjs-0.3.4.min.js"
  [server config handler]
  (.installApp server (u/encode config) (core/as-handler handler)))

(def ^:const default-auth-timeout (* 5 60 1000))
(def ^:const default-auth-address (str "vertx.basicauthmanager.authorise"))

(defn bridge
  "Install an app which bridges the SockJS server to the event bus.
   config is a map of configuration options (see install-app).
   inbound-permitted and outbound-permitted are lists of JSON objects
   which define permitted matches for inbound (client->server) and
   outbound (server->client) traffic, respectively. See the \"Securing
   the Bridge\" section of the manual for the proper usage of these
   options. auth-timeout specifies the amount of time (in ms) an
   authorisation will be cached in the bridge (defaults to 5 minutes).
   auth-address specifies the address of the auth manager (defaults to
   'vertx.basicauthmanager.authorise')"
  ([server config inbound-permitted outbound-permitted]
     (bridge server config inbound-permitted outbound-permitted
             default-auth-timeout default-auth-address))

  ([server config inbound-permitted outbound-permitted auth-timeout]
     (bridge server config inbound-permitted outbound-permitted auth-timeout
             default-auth-address))

  ([server config inbound-permitted outbound-permitted auth-timeout auth-address]
     (.bridge server (u/encode config) (u/encode inbound-permitted)
              (u/encode outbound-permitted) auth-timeout auth-address)))


(defn- eb-bridge-hook
  "Make a implememtion of EventBusBridgeHook, take a kv pair as handlers."
  [hooks]
  (letfn [(call-if [f & args]
            (if f
              (boolean (apply f args))
              true))] 
    (reify EventBusBridgeHook
      (handleSocketClosed [_ sock]
        (call-if (:closed hooks) sock))

      (handleSendOrPub [_ sock is-send msg address]
        (call-if
         (if is-send (:send hooks) (:publish hooks))
         sock msg address))
      
      (handlePreRegister [_ sock address]
        (call-if (:pre-register hooks) sock address))

      (handlePostRegister [_ sock address]
        (call-if (:post-register hooks) sock address))

      (handleUnregister [_ sock address]
        (call-if (:unregister hooks) sock address)))))

(defn set-hooks
  "Registers functions to be called when certain events occur on an event bus bridge.
   Takes the following kwargs:

   :closed        Called when the socket has been closed. The fn will be
                  passed the SockJSSocket
   :send          Called when the clent is sends data. The fn will be
                  passed SockJSSocket, the message, and the eventbus
                  address. The fn must return truthy for the send to be
                  allowed.
   :publish       Called when the clent is publishes data. The fn will be
                  passed SockJSSocket, the message, and the eventbus
                  address. The fn must return truthy for the publish to be
                  allowed.
   :pre-register  Called before a client handler registration is processed.
                  The fn will be passed the SockJSSocket and the address.
                  The fn must return truthy for the registration to be
                  allowed.
   :post-register Called after a client handler registration is processed.
                  The fn will be passed the SockJSSocket and the address.
   :unregister    Called before a client handler unregistration is processed.
                  The fn will be passed the SockJSSocket and the address.
                  The fn must return truthy for the unregistration to be
                  allowed.

  Returns the server. Calling set-hooks more than once will overwrite
  the hooks set previously."
  [server & {:as hooks}]
  (.setHook server (eb-bridge-hook hooks)))
