// Copyright (c) Corporation for National Research Initiatives
package org.python.core;

import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import org.python.core.StringFormatter.DecimalFormatTemplate;
import org.python.core.buffer.BaseBuffer;
import org.python.core.buffer.SimpleStringBuffer;
import org.python.core.stringlib.FieldNameIterator;
import org.python.core.stringlib.InternalFormatSpec;
import org.python.core.stringlib.InternalFormatSpecParser;
import org.python.core.stringlib.MarkupIterator;
import org.python.core.util.ExtraMath;
import org.python.core.util.StringUtil;
import org.python.expose.ExposedMethod;
import org.python.expose.ExposedNew;
import org.python.expose.ExposedType;
import org.python.expose.MethodType;

/**
 * A builtin python string.
 */
@ExposedType(name = "str", doc = BuiltinDocs.str_doc)
public class PyString extends PyBaseString implements BufferProtocol {

    public static final PyType TYPE = PyType.fromClass(PyString.class);
    protected String string; // cannot make final because of Python intern support
    protected transient boolean interned = false;
    /** Supports the buffer API, see {@link #getBuffer(int)}. */
    private Reference<BaseBuffer> export;

    public String getString() {
        return string;
    }

    // for PyJavaClass.init()
    public PyString() {
        this(TYPE, "");
    }

    public PyString(PyType subType, String string) {
        super(subType);
        if (string == null) {
            throw new IllegalArgumentException("Cannot create PyString from null!");
        }
        this.string = string;
    }

    public PyString(String string) {
        this(TYPE, string);
    }

    public PyString(char c) {
        this(TYPE, String.valueOf(c));
    }

    PyString(StringBuilder buffer) {
        this(TYPE, new String(buffer));
    }

    /**
     * Creates a PyString from an already interned String. Just means it won't be reinterned if used
     * in a place that requires interned Strings.
     */
    public static PyString fromInterned(String interned) {
        PyString str = new PyString(TYPE, interned);
        str.interned = true;
        return str;
    }

    @ExposedNew
    static PyObject str_new(PyNewWrapper new_, boolean init, PyType subtype, PyObject[] args,
            String[] keywords) {
        ArgParser ap = new ArgParser("str", args, keywords, new String[] {"object"}, 0);
        PyObject S = ap.getPyObject(0, null);
        if (new_.for_type == subtype) {
            if (S == null) {
                return new PyString("");
            }
            return new PyString(S.__str__().toString());
        } else {
            if (S == null) {
                return new PyStringDerived(subtype, "");
            }
            return new PyStringDerived(subtype, S.__str__().toString());
        }
    }

    public int[] toCodePoints() {
        int n = getString().length();
        int[] codePoints = new int[n];
        for (int i = 0; i < n; i++) {
            codePoints[i] = getString().charAt(i);
        }
        return codePoints;
    }

    /**
     * Return a read-only buffer view of the contents of the string, treating it as a sequence of
     * unsigned bytes. The caller specifies its requirements and navigational capabilities in the
     * <code>flags</code> argument (see the constants in interface {@link PyBUF} for an
     * explanation). The method may return the same PyBuffer object to more than one consumer.
     *
     * @param flags consumer requirements
     * @return the requested buffer
     */
    @Override
    public synchronized PyBuffer getBuffer(int flags) {
        // If we have already exported a buffer it may still be available for re-use
        BaseBuffer pybuf = getExistingBuffer(flags);
        if (pybuf == null) {
            /*
             * No existing export we can re-use. Return a buffer, but specialised to defer
             * construction of the buf object, and cache a soft reference to it.
             */
            pybuf = new SimpleStringBuffer(flags, getString());
            export = new SoftReference<BaseBuffer>(pybuf);
        }
        return pybuf;
    }

    /**
     * Helper for {@link #getBuffer(int)} that tries to re-use an existing exported buffer, or
     * returns null if can't.
     */
    private BaseBuffer getExistingBuffer(int flags) {
        BaseBuffer pybuf = null;
        if (export != null) {
            // A buffer was exported at some time.
            pybuf = export.get();
            if (pybuf != null) {
                /*
                 * And this buffer still exists. Even in the case where the buffer has been released
                 * by all its consumers, it remains safe to re-acquire it because the target String
                 * has not changed.
                 */
                pybuf = pybuf.getBufferAgain(flags);
            }
        }
        return pybuf;
    }

    public String substring(int start, int end) {
        return getString().substring(start, end);
    }

    @Override
    public PyString __str__() {
        return str___str__();
    }

    public @ExposedMethod(doc = BuiltinDocs.str___str___doc)
    final PyString str___str__() {
        if (getClass() == PyString.class) {
            return this;
        }
        return new PyString(getString());
    }

    @Override
    public PyUnicode __unicode__() {
        return new PyUnicode(this);
    }

    @Override
    public int __len__() {
        return str___len__();
    }

    @ExposedMethod(doc = BuiltinDocs.str___len___doc)
    final int str___len__() {
        return getString().length();
    }

    @Override
    public String toString() {
        return getString();
    }

    public String internedString() {
        if (interned) {
            return getString();
        } else {
            string = getString().intern();
            interned = true;
            return getString();
        }
    }

    @Override
    public PyString __repr__() {
        return str___repr__();
    }

    @ExposedMethod(doc = BuiltinDocs.str___repr___doc)
    final PyString str___repr__() {
        return new PyString(encode_UnicodeEscape(getString(), true));
    }

    private static char[] hexdigit = "0123456789abcdef".toCharArray();

    public static String encode_UnicodeEscape(String str, boolean use_quotes) {
        int size = str.length();
        StringBuilder v = new StringBuilder(str.length());

        char quote = 0;

        if (use_quotes) {
            quote = str.indexOf('\'') >= 0 && str.indexOf('"') == -1 ? '"' : '\'';
            v.append(quote);
        }

        for (int i = 0; size-- > 0;) {
            int ch = str.charAt(i++);
            /* Escape quotes */
            if ((use_quotes && ch == quote) || ch == '\\') {
                v.append('\\');
                v.append((char)ch);
                continue;
            }
            /* Map UTF-16 surrogate pairs to Unicode \UXXXXXXXX escapes */
            else if (ch >= 0xD800 && ch < 0xDC00) {
                char ch2 = str.charAt(i++);
                size--;
                if (ch2 >= 0xDC00 && ch2 <= 0xDFFF) {
                    int ucs = (((ch & 0x03FF) << 10) | (ch2 & 0x03FF)) + 0x00010000;
                    v.append('\\');
                    v.append('U');
                    v.append(hexdigit[(ucs >> 28) & 0xf]);
                    v.append(hexdigit[(ucs >> 24) & 0xf]);
                    v.append(hexdigit[(ucs >> 20) & 0xf]);
                    v.append(hexdigit[(ucs >> 16) & 0xf]);
                    v.append(hexdigit[(ucs >> 12) & 0xf]);
                    v.append(hexdigit[(ucs >> 8) & 0xf]);
                    v.append(hexdigit[(ucs >> 4) & 0xf]);
                    v.append(hexdigit[ucs & 0xf]);
                    continue;
                }
                /* Fall through: isolated surrogates are copied as-is */
                i--;
                size++;
            }
            /* Map 16-bit characters to '\\uxxxx' */
            if (ch >= 256) {
                v.append('\\');
                v.append('u');
                v.append(hexdigit[(ch >> 12) & 0xf]);
                v.append(hexdigit[(ch >> 8) & 0xf]);
                v.append(hexdigit[(ch >> 4) & 0xf]);
                v.append(hexdigit[ch & 15]);
            }
            /* Map special whitespace to '\t', \n', '\r' */
            else if (ch == '\t') {
                v.append("\\t");
            } else if (ch == '\n') {
                v.append("\\n");
            } else if (ch == '\r') {
                v.append("\\r");
            } else if (ch < ' ' || ch >= 127) {
                /* Map non-printable US ASCII to '\xNN' */
                v.append('\\');
                v.append('x');
                v.append(hexdigit[(ch >> 4) & 0xf]);
                v.append(hexdigit[ch & 0xf]);
            } else {/* Copy everything else as-is */
                v.append((char)ch);
            }
        }
        if (use_quotes) {
            v.append(quote);
        }
        return v.toString();
    }

    private static ucnhashAPI pucnHash = null;

    public static String decode_UnicodeEscape(String str, int start, int end, String errors,
            boolean unicode) {
        StringBuilder v = new StringBuilder(end - start);
        for (int s = start; s < end;) {
            char ch = str.charAt(s);
            /* Non-escape characters are interpreted as Unicode ordinals */
            if (ch != '\\') {
                v.append(ch);
                s++;
                continue;
            }
            int loopStart = s;
            /* \ - Escapes */
            s++;
            if (s == end) {
                s = codecs.insertReplacementAndGetResume(v, errors, "unicodeescape", //
                        str, loopStart, s + 1, "\\ at end of string");
                continue;
            }
            ch = str.charAt(s++);
            switch (ch) {
            /* \x escapes */
                case '\n':
                    break;
                case '\\':
                    v.append('\\');
                    break;
                case '\'':
                    v.append('\'');
                    break;
                case '\"':
                    v.append('\"');
                    break;
                case 'b':
                    v.append('\b');
                    break;
                case 'f':
                    v.append('\014');
                    break; /* FF */
                case 't':
                    v.append('\t');
                    break;
                case 'n':
                    v.append('\n');
                    break;
                case 'r':
                    v.append('\r');
                    break;
                case 'v':
                    v.append('\013');
                    break; /* VT */
                case 'a':
                    v.append('\007');
                    break; /* BEL, not classic C */
                /* \OOO (octal) escapes */
                case '0':
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                    int x = Character.digit(ch, 8);
                    for (int j = 0; j < 2 && s < end; j++, s++) {
                        ch = str.charAt(s);
                        if (ch < '0' || ch > '7') {
                            break;
                        }
                        x = (x << 3) + Character.digit(ch, 8);
                    }
                    v.append((char)x);
                    break;
                case 'x':
                    s = hexescape(v, errors, 2, s, str, end, "truncated \\xXX");
                    break;
                case 'u':
                    if (!unicode) {
                        v.append('\\');
                        v.append('u');
                        break;
                    }
                    s = hexescape(v, errors, 4, s, str, end, "truncated \\uXXXX");
                    break;
                case 'U':
                    if (!unicode) {
                        v.append('\\');
                        v.append('U');
                        break;
                    }
                    s = hexescape(v, errors, 8, s, str, end, "truncated \\UXXXXXXXX");
                    break;
                case 'N':
                    if (!unicode) {
                        v.append('\\');
                        v.append('N');
                        break;
                    }
                    /*
                     * Ok, we need to deal with Unicode Character Names now, make sure we've
                     * imported the hash table data...
                     */
                    if (pucnHash == null) {
                        PyObject mod = imp.importName("ucnhash", true);
                        mod = mod.__call__();
                        pucnHash = (ucnhashAPI)mod.__tojava__(Object.class);
                        if (pucnHash.getCchMax() < 0) {
                            throw Py.UnicodeError("Unicode names not loaded");
                        }
                    }
                    if (str.charAt(s) == '{') {
                        int startName = s + 1;
                        int endBrace = startName;
                        /*
                         * look for either the closing brace, or we exceed the maximum length of the
                         * unicode character names
                         */
                        int maxLen = pucnHash.getCchMax();
                        while (endBrace < end && str.charAt(endBrace) != '}'
                                && (endBrace - startName) <= maxLen) {
                            endBrace++;
                        }
                        if (endBrace != end && str.charAt(endBrace) == '}') {
                            int value = pucnHash.getValue(str, startName, endBrace);
                            if (storeUnicodeCharacter(value, v)) {
                                s = endBrace + 1;
                            } else {
                                s = codecs.insertReplacementAndGetResume( //
                                        v, errors, "unicodeescape", //
                                        str, loopStart, endBrace + 1, "illegal Unicode character");
                            }
                        } else {
                            s = codecs.insertReplacementAndGetResume(v, errors, "unicodeescape", //
                                    str, loopStart, endBrace, "malformed \\N character escape");
                        }
                        break;
                    } else {
                        s = codecs.insertReplacementAndGetResume(v, errors, "unicodeescape", //
                                str, loopStart, s + 1, "malformed \\N character escape");
                    }
                    break;
                default:
                    v.append('\\');
                    v.append(str.charAt(s - 1));
                    break;
            }
        }
        return v.toString();
    }

    private static int hexescape(StringBuilder partialDecode, String errors, int digits,
            int hexDigitStart, String str, int size, String errorMessage) {
        if (hexDigitStart + digits > size) {
            return codecs.insertReplacementAndGetResume(partialDecode, errors, "unicodeescape",
                    str, hexDigitStart - 2, size, errorMessage);
        }
        int i = 0;
        int x = 0;
        for (; i < digits; ++i) {
            char c = str.charAt(hexDigitStart + i);
            int d = Character.digit(c, 16);
            if (d == -1) {
                return codecs.insertReplacementAndGetResume(partialDecode, errors, "unicodeescape",
                        str, hexDigitStart - 2, hexDigitStart + i + 1, errorMessage);
            }
            x = (x << 4) & ~0xF;
            if (c >= '0' && c <= '9') {
                x += c - '0';
            } else if (c >= 'a' && c <= 'f') {
                x += 10 + c - 'a';
            } else {
                x += 10 + c - 'A';
            }
        }
        if (storeUnicodeCharacter(x, partialDecode)) {
            return hexDigitStart + i;
        } else {
            return codecs.insertReplacementAndGetResume(partialDecode, errors, "unicodeescape",
                    str, hexDigitStart - 2, hexDigitStart + i + 1, "illegal Unicode character");
        }
    }

    /* pass in an int since this can be a UCS-4 character */
    private static boolean storeUnicodeCharacter(int value, StringBuilder partialDecode) {
        if (value < 0 || (value >= 0xD800 && value <= 0xDFFF)) {
            return false;
        } else if (value <= PySystemState.maxunicode) {
            partialDecode.appendCodePoint(value);
            return true;
        }
        return false;
    }

    @ExposedMethod(doc = BuiltinDocs.str___getitem___doc)
    final PyObject str___getitem__(PyObject index) {
        PyObject ret = seq___finditem__(index);
        if (ret == null) {
            throw Py.IndexError("string index out of range");
        }
        return ret;
    }

    // XXX: need doc
    @ExposedMethod(defaults = "null")
    final PyObject str___getslice__(PyObject start, PyObject stop, PyObject step) {
        return seq___getslice__(start, stop, step);
    }

    @Override
    public int __cmp__(PyObject other) {
        return str___cmp__(other);
    }

    @ExposedMethod(type = MethodType.CMP)
    final int str___cmp__(PyObject other) {
        if (!(other instanceof PyString)) {
            return -2;
        }

        int c = getString().compareTo(((PyString)other).getString());
        return c < 0 ? -1 : c > 0 ? 1 : 0;
    }

    @Override
    public PyObject __eq__(PyObject other) {
        return str___eq__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___eq___doc)
    final PyObject str___eq__(PyObject other) {
        String s = coerce(other);
        if (s == null) {
            return null;
        }
        return getString().equals(s) ? Py.True : Py.False;
    }

    @Override
    public PyObject __ne__(PyObject other) {
        return str___ne__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___ne___doc)
    final PyObject str___ne__(PyObject other) {
        String s = coerce(other);
        if (s == null) {
            return null;
        }
        return getString().equals(s) ? Py.False : Py.True;
    }

    @Override
    public PyObject __lt__(PyObject other) {
        return str___lt__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___lt___doc)
    final PyObject str___lt__(PyObject other) {
        String s = coerce(other);
        if (s == null) {
            return null;
        }
        return getString().compareTo(s) < 0 ? Py.True : Py.False;
    }

    @Override
    public PyObject __le__(PyObject other) {
        return str___le__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___le___doc)
    final PyObject str___le__(PyObject other) {
        String s = coerce(other);
        if (s == null) {
            return null;
        }
        return getString().compareTo(s) <= 0 ? Py.True : Py.False;
    }

    @Override
    public PyObject __gt__(PyObject other) {
        return str___gt__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___gt___doc)
    final PyObject str___gt__(PyObject other) {
        String s = coerce(other);
        if (s == null) {
            return null;
        }
        return getString().compareTo(s) > 0 ? Py.True : Py.False;
    }

    @Override
    public PyObject __ge__(PyObject other) {
        return str___ge__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___ge___doc)
    final PyObject str___ge__(PyObject other) {
        String s = coerce(other);
        if (s == null) {
            return null;
        }
        return getString().compareTo(s) >= 0 ? Py.True : Py.False;
    }

    private static String coerce(PyObject o) {
        if (o instanceof PyString) {
            return o.toString();
        }
        return null;
    }

    @Override
    public int hashCode() {
        return str___hash__();
    }

    @ExposedMethod(doc = BuiltinDocs.str___hash___doc)
    final int str___hash__() {
        return getString().hashCode();
    }

    /**
     * @return a byte array with one byte for each char in this object's underlying String. Each
     *         byte contains the low-order bits of its corresponding char.
     */
    public byte[] toBytes() {
        return StringUtil.toBytes(getString());
    }

    @Override
    public Object __tojava__(Class<?> c) {
        if (c.isAssignableFrom(String.class)) {
            return getString();
        }

        if (c == Character.TYPE || c == Character.class) {
            if (getString().length() == 1) {
                return new Character(getString().charAt(0));
            }
        }

        if (c.isArray()) {
            if (c.getComponentType() == Byte.TYPE) {
                return toBytes();
            }
            if (c.getComponentType() == Character.TYPE) {
                return getString().toCharArray();
            }
        }

        if (c.isInstance(this)) {
            return this;
        }

        return Py.NoConversion;
    }

    @Override
    protected PyObject pyget(int i) {
        return Py.newString(getString().charAt(i));
    }

    @Override
    protected PyObject getslice(int start, int stop, int step) {
        if (step > 0 && stop < start) {
            stop = start;
        }
        if (step == 1) {
            return fromSubstring(start, stop);
        } else {
            int n = sliceLength(start, stop, step);
            char new_chars[] = new char[n];
            int j = 0;
            for (int i = start; j < n; i += step) {
                new_chars[j++] = getString().charAt(i);
            }

            return createInstance(new String(new_chars), true);
        }
    }

    public PyString createInstance(String str) {
        return new PyString(str);
    }

    protected PyString createInstance(String str, boolean isBasic) {
        // ignore isBasic, doesn't apply to PyString, just PyUnicode
        return new PyString(str);
    }

    /**
     * Return a String equivalent to the argument. This is a helper function to those methods that
     * accept any byte array type (any object that supports a one-dimensional byte buffer).
     *
     * @param obj to coerce to a String
     * @return coerced value or <code>null</code> if it can't be
     */
    private static String asStringOrNull(PyObject obj) {
        if (obj instanceof PyString) {
            // str or unicode object: go directly to the String
            return ((PyString)obj).getString();
        } else if (obj instanceof BufferProtocol) {
            // Other object with buffer API: briefly access the buffer
            PyBuffer buf = ((BufferProtocol)obj).getBuffer(PyBUF.SIMPLE);
            try {
                return buf.toString();
            } finally {
                buf.release();
            }
        } else {
            return null;
        }
    }

    /**
     * Return a String equivalent to the argument. This is a helper function to those methods that
     * accept any byte array type (any object that supports a one-dimensional byte buffer).
     *
     * @param obj to coerce to a String
     * @return coerced value
     * @throws PyException if the coercion fails
     */
    private static String asStringOrError(PyObject obj) throws PyException {
        String ret = asStringOrNull(obj);
        if (ret != null) {
            return ret;
        } else {
            throw Py.TypeError("expected str, bytearray or other buffer compatible object");
        }
    }

    /**
     * Return a String equivalent to the argument according to the calling conventions of methods
     * that accept anything bearing the buffer interface as a byte string, but also
     * <code>PyNone</code>. (Or the argument may be omitted, showing up here as null.) These include
     * the <code>strip</code> and <code>split</code> methods of <code>str</code>, where a null
     * indicates that the criterion is whitespace, and <code>str.translate</code>.
     *
     * @param obj to coerce to a String or null
     * @param name of method
     * @return coerced value or null
     * @throws PyException if the coercion fails
     */
    private static String asStringNullOrError(PyObject obj, String name) throws PyException {

        if (obj == null || obj == Py.None) {
            return null;
        } else {
            String ret = asStringOrNull(obj);
            if (ret != null) {
                return ret;
            } else if (name == null) {
                // A nameless method is the client
                throw Py.TypeError("expected None, str or buffer compatible object");
            } else {
                // Tuned for .strip and its relations, which supply their name
                throw Py.TypeError(name + " arg must be None, str or buffer compatible object");
            }
        }
    }

    /**
     * Return a String equivalent to the argument according to the calling conventions of the
     * certain methods of <code>str</code>. Those methods accept anything bearing the buffer
     * interface as a byte string, or accept a unicode argument for which they accept responsibility
     * to interpret from its UTF16 encoded form (the internal representation returned by
     * {@link PyUnicode#getString()}).
     *
     * @param obj to coerce to a String
     * @return coerced value
     * @throws PyException if the coercion fails
     */
    private static String asBMPStringOrError(PyObject obj) {
        // PyUnicode accepted here. Care required in the client if obj is not basic plane.
        String ret = asStringOrNull(obj);
        if (ret != null) {
            return ret;
        } else {
            throw Py.TypeError("expected str, bytearray, unicode or buffer compatible object");
        }
    }

    @Override
    public boolean __contains__(PyObject o) {
        return str___contains__(o);
    }

    @ExposedMethod(doc = BuiltinDocs.str___contains___doc)
    final boolean str___contains__(PyObject o) {
        String other = asStringOrError(o);
        return getString().indexOf(other) >= 0;
    }

    @Override
    protected PyObject repeat(int count) {
        if (count < 0) {
            count = 0;
        }
        int s = getString().length();
        if ((long)s * count > Integer.MAX_VALUE) {
            // Since Strings store their data in an array, we can't make one
            // longer than Integer.MAX_VALUE. Without this check we get
            // NegativeArraySize exceptions when we create the array on the
            // line with a wrapped int.
            throw Py.OverflowError("max str len is " + Integer.MAX_VALUE);
        }
        char new_chars[] = new char[s * count];
        for (int i = 0; i < count; i++) {
            getString().getChars(0, s, new_chars, i * s);
        }
        return createInstance(new String(new_chars));
    }

    @Override
    public PyObject __mul__(PyObject o) {
        return str___mul__(o);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___mul___doc)
    final PyObject str___mul__(PyObject o) {
        if (!o.isIndex()) {
            return null;
        }
        return repeat(o.asIndex(Py.OverflowError));
    }

    @Override
    public PyObject __rmul__(PyObject o) {
        return str___rmul__(o);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___rmul___doc)
    final PyObject str___rmul__(PyObject o) {
        if (!o.isIndex()) {
            return null;
        }
        return repeat(o.asIndex(Py.OverflowError));
    }

    /**
     * {@inheritDoc} For a <code>str</code> addition means concatenation and returns a
     * <code>str</code> ({@link PyString}) result, except when a {@link PyUnicode} argument is
     * given, when a <code>PyUnicode</code> results.
     */
    @Override
    public PyObject __add__(PyObject other) {
        return str___add__(other);
    }

    @ExposedMethod(type = MethodType.BINARY, doc = BuiltinDocs.str___add___doc)
    final PyObject str___add__(PyObject other) {

        if (other instanceof PyUnicode) {
            // Convert self to PyUnicode and escalate the problem
            return decode().__add__(other);

        } else {
            // Some kind of object with the buffer API
            String otherStr = asStringOrNull(other);
            if (otherStr == null) {
                // Allow PyObject._basic_add to pick up the pieces or raise informative error
                return null;
            } else {
                // Concatenate as strings
                return new PyString(getString().concat(otherStr));
            }
        }
    }

    @ExposedMethod(doc = BuiltinDocs.str___getnewargs___doc)
    final PyTuple str___getnewargs__() {
        return new PyTuple(new PyString(this.getString()));
    }

    @Override
    public PyTuple __getnewargs__() {
        return str___getnewargs__();
    }

    @Override
    public PyObject __mod__(PyObject other) {
        return str___mod__(other);
    }

    @ExposedMethod(doc = BuiltinDocs.str___mod___doc)
    public PyObject str___mod__(PyObject other) {
        StringFormatter fmt = new StringFormatter(getString(), false);
        return fmt.format(other);
    }

    @Override
    public PyObject __int__() {
        try {
            return Py.newInteger(atoi(10));
        } catch (PyException e) {
            if (e.match(Py.OverflowError)) {
                return atol(10);
            }
            throw e;
        }
    }

    @Override
    public PyObject __long__() {
        return atol(10);
    }

    @Override
    public PyFloat __float__() {
        return new PyFloat(atof());
    }

    @Override
    public PyObject __pos__() {
        throw Py.TypeError("bad operand type for unary +");
    }

    @Override
    public PyObject __neg__() {
        throw Py.TypeError("bad operand type for unary -");
    }

    @Override
    public PyObject __invert__() {
        throw Py.TypeError("bad operand type for unary ~");
    }

    @SuppressWarnings("fallthrough")
    @Override
    public PyComplex __complex__() {
        boolean got_re = false;
        boolean got_im = false;
        boolean done = false;
        boolean sw_error = false;

        int s = 0;
        int n = getString().length();
        while (s < n && Character.isSpaceChar(getString().charAt(s))) {
            s++;
        }

        if (s == n) {
            throw Py.ValueError("empty string for complex()");
        }

        double z = -1.0;
        double x = 0.0;
        double y = 0.0;

        int sign = 1;
        do {
            char c = getString().charAt(s);
            switch (c) {
                case '-':
                    sign = -1;
                    /* Fallthrough */
                case '+':
                    if (done || s + 1 == n) {
                        sw_error = true;
                        break;
                    }
                    // a character is guaranteed, but it better be a digit
                    // or J or j
                    c = getString().charAt(++s);  // eat the sign character
                    // and check the next
                    if (!Character.isDigit(c) && c != 'J' && c != 'j') {
                        sw_error = true;
                    }
                    break;

                case 'J':
                case 'j':
                    if (got_im || done) {
                        sw_error = true;
                        break;
                    }
                    if (z < 0.0) {
                        y = sign;
                    } else {
                        y = sign * z;
                    }
                    got_im = true;
                    done = got_re;
                    sign = 1;
                    s++; // eat the J or j
                    break;

                case ' ':
                    while (s < n && Character.isSpaceChar(getString().charAt(s))) {
                        s++;
                    }
                    if (s != n) {
                        sw_error = true;
                    }
                    break;

                default:
                    boolean digit_or_dot = (c == '.' || Character.isDigit(c));
                    if (!digit_or_dot) {
                        sw_error = true;
                        break;
                    }
                    int end = endDouble(getString(), s);
                    z = Double.valueOf(getString().substring(s, end)).doubleValue();
                    if (z == Double.POSITIVE_INFINITY) {
                        throw Py.ValueError(String.format("float() out of range: %.150s",
                                getString()));
                    }

                    s = end;
                    if (s < n) {
                        c = getString().charAt(s);
                        if (c == 'J' || c == 'j') {
                            break;
                        }
                    }
                    if (got_re) {
                        sw_error = true;
                        break;
                    }

                    /* accept a real part */
                    x = sign * z;
                    got_re = true;
                    done = got_im;
                    z = -1.0;
                    sign = 1;
                    break;

            } /* end of switch */

        } while (s < n && !sw_error);

        if (sw_error) {
            throw Py.ValueError("malformed string for complex() " + getString().substring(s));
        }

        return new PyComplex(x, y);
    }

    private int endDouble(String string, int s) {
        int n = string.length();
        while (s < n) {
            char c = string.charAt(s++);
            if (Character.isDigit(c)) {
                continue;
            }
            if (c == '.') {
                continue;
            }
            if (c == 'e' || c == 'E') {
                if (s < n) {
                    c = string.charAt(s);
                    if (c == '+' || c == '-') {
                        s++;
                    }
                    continue;
                }
            }
            return s - 1;
        }
        return s;
    }

    // Add in methods from string module
    public String lower() {
        return str_lower();
    }

    @ExposedMethod(doc = BuiltinDocs.str_lower_doc)
    final String str_lower() {
        return getString().toLowerCase();
    }

    public String upper() {
        return str_upper();
    }

    @ExposedMethod(doc = BuiltinDocs.str_upper_doc)
    final String str_upper() {
        return getString().toUpperCase();
    }

    public String title() {
        return str_title();
    }

    @ExposedMethod(doc = BuiltinDocs.str_title_doc)
    final String str_title() {
        char[] chars = getString().toCharArray();
        int n = chars.length;

        boolean previous_is_cased = false;
        for (int i = 0; i < n; i++) {
            char ch = chars[i];
            if (previous_is_cased) {
                chars[i] = Character.toLowerCase(ch);
            } else {
                chars[i] = Character.toTitleCase(ch);
            }

            if (Character.isLowerCase(ch) || Character.isUpperCase(ch) || Character.isTitleCase(ch)) {
                previous_is_cased = true;
            } else {
                previous_is_cased = false;
            }
        }
        return new String(chars);
    }

    public String swapcase() {
        return str_swapcase();
    }

    @ExposedMethod(doc = BuiltinDocs.str_swapcase_doc)
    final String str_swapcase() {
        char[] chars = getString().toCharArray();
        int n = chars.length;
        for (int i = 0; i < n; i++) {
            char c = chars[i];
            if (Character.isUpperCase(c)) {
                chars[i] = Character.toLowerCase(c);
            } else if (Character.isLowerCase(c)) {
                chars[i] = Character.toUpperCase(c);
            }
        }
        return new String(chars);
    }

    /**
     * Equivalent of Python <code>str.strip()</code> with no argument, meaning strip whitespace. Any
     * whitespace byte/character will be discarded from either end of this <code>str</code>.
     *
     * @return a new String, stripped of the whitespace characters/bytes
     */
    public String strip() {
        return _strip();
    }

    /**
     * Equivalent of Python <code>str.strip()</code>.
     *
     * @param stripChars characters to strip from either end of this str/bytes, or null
     * @return a new String, stripped of the specified characters/bytes
     */
    public String strip(String stripChars) {
        return _strip(stripChars);
    }

    /**
     * Equivalent of Python <code>str.strip()</code>. Any byte/character matching one of those in
     * <code>stripChars</code> will be discarded from either end of this <code>str</code>. If
     * <code>stripChars == null</code>, whitespace will be stripped. If <code>stripChars</code> is a
     * <code>PyUnicode</code>, the result will also be a <code>PyUnicode</code>.
     *
     * @param stripChars characters to strip from either end of this str/bytes, or null
     * @return a new <code>PyString</code> (or {@link PyUnicode}), stripped of the specified
     *         characters/bytes
     */
    public PyObject strip(PyObject stripChars) {
        return str_strip(stripChars);
    }

    @ExposedMethod(defaults = "null", doc = BuiltinDocs.str_strip_doc)
    final PyObject str_strip(PyObject chars) {
        if (chars instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_strip(chars);
        } else {
            // It ought to be None, null, some kind of bytes with the buffer API.
            String stripChars = asStringNullOrError(chars, "strip");
            // Strip specified characters or whitespace if stripChars == null
            return new PyString(_strip(stripChars));
        }
    }

    /**
     * Implementation of Python <code>str.strip()</code> common to exposed and Java API, when
     * stripping whitespace. Any whitespace byte/character will be discarded from either end of this
     * <code>str</code>.
     * <p>
     * Implementation note: although a <code>str</code> contains only bytes, this method is also
     * called by {@link PyUnicode#unicode_strip(PyObject)} when this is a basic-plane string.
     *
     * @return a new String, stripped of the whitespace characters/bytes
     */
    protected final String _strip() {
        String s = getString();
        // Rightmost non-whitespace
        int right = _stripRight(s);
        if (right < 0) {
            // They're all whitespace
            return "";
        } else {
            // Leftmost non-whitespace character: right known not to be a whitespace
            int left = _stripLeft(s, right);
            return s.substring(left, right + 1);
        }
    }

    /**
     * Implementation of Python <code>str.strip()</code> common to exposed and Java API. Any
     * byte/character matching one of those in <code>stripChars</code> will be discarded from either
     * end of this <code>str</code>. If <code>stripChars == null</code>, whitespace will be
     * stripped.
     * <p>
     * Implementation note: although a <code>str</code> contains only bytes, this method is also
     * called by {@link PyUnicode#unicode_strip(PyObject)} when both arguments are basic-plane
     * strings.
     *
     * @param stripChars characters to strip or null
     * @return a new String, stripped of the specified characters/bytes
     */
    protected final String _strip(String stripChars) {
        if (stripChars == null) {
            // Divert to the whitespace version
            return _strip();
        } else {
            String s = getString();
            // Rightmost non-matching character
            int right = _stripRight(s, stripChars);
            if (right < 0) {
                // They all match
                return "";
            } else {
                // Leftmost non-matching character: right is known not to match
                int left = _stripLeft(s, stripChars, right);
                return s.substring(left, right + 1);
            }
        }
    }

    /**
     * Helper for <code>strip</code>, <code>lstrip</code> implementation, when stripping whitespace.
     *
     * @param s string to search (only <code>s[0:right]</code> is searched).
     * @param right rightmost extent of string search
     * @return index of lefttmost non-whitespace character or <code>right</code> if they all are.
     */
    private static final int _stripLeft(String s, int right) {
        for (int left = 0; left < right; left++) {
            if (!Character.isWhitespace(s.charAt(left))) {
                return left;
            }
        }
        return right;
    }

    /**
     * Helper for <code>strip</code>, <code>lstrip</code> implementation, when stripping specified
     * characters.
     *
     * @param s string to search (only <code>s[0:right]</code> is searched).
     * @param stripChars specifies set of characters to strip
     * @param right rightmost extent of string search
     * @return index of leftmost character not in <code>stripChars</code> or <code>right</code> if
     *         they all are.
     */
    private static final int _stripLeft(String s, String stripChars, int right) {
        for (int left = 0; left < right; left++) {
            if (stripChars.indexOf(s.charAt(left)) < 0) {
                return left;
            }
        }
        return right;
    }

    /**
     * Helper for <code>strip</code>, <code>rstrip</code> implementation, when stripping whitespace.
     *
     * @param s string to search.
     * @return index of rightmost non-whitespace character or -1 if they all are.
     */
    private static final int _stripRight(String s) {
        for (int right = s.length(); --right >= 0;) {
            if (!Character.isWhitespace(s.charAt(right))) {
                return right;
            }
        }
        return -1;
    }

    /**
     * Helper for <code>strip</code>, <code>rstrip</code> implementation, when stripping specified
     * characters.
     *
     * @param s string to search.
     * @param stripChars specifies set of characters to strip
     * @return index of rightmost character not in <code>stripChars</code> or -1 if they all are.
     */
    private static final int _stripRight(String s, String stripChars) {
        for (int right = s.length(); --right >= 0;) {
            if (stripChars.indexOf(s.charAt(right)) < 0) {
                return right;
            }
        }
        return -1;
    }

    /**
     * Equivalent of Python <code>str.lstrip()</code> with no argument, meaning strip whitespace.
     * Any whitespace byte/character will be discarded from the left of this <code>str</code>.
     *
     * @return a new String, stripped of the whitespace characters/bytes
     */
    public String lstrip() {
        return _lstrip();
    }

    /**
     * Equivalent of Python <code>str.lstrip()</code>.
     *
     * @param stripChars characters to strip from the left end of this str/bytes, or null
     * @return a new String, stripped of the specified characters/bytes
     */
    public String lstrip(String sep) {
        return _lstrip(sep);
    }

    /**
     * Equivalent of Python <code>str.lstrip()</code>. Any byte/character matching one of those in
     * <code>stripChars</code> will be discarded from the left end of this <code>str</code>. If
     * <code>stripChars == null</code>, whitespace will be stripped. If <code>stripChars</code> is a
     * <code>PyUnicode</code>, the result will also be a <code>PyUnicode</code>.
     *
     * @param stripChars characters to strip from the left end of this str/bytes, or null
     * @return a new <code>PyString</code> (or {@link PyUnicode}), stripped of the specified
     *         characters/bytes
     */
    public PyObject lstrip(PyObject sep) {
        return str_lstrip(sep);
    }

    @ExposedMethod(defaults = "null", doc = BuiltinDocs.str_lstrip_doc)
    final PyObject str_lstrip(PyObject chars) {
        if (chars instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_lstrip(chars);
        } else {
            // It ought to be None, null, some kind of bytes with the buffer API.
            String stripChars = asStringNullOrError(chars, "lstrip");
            // Strip specified characters or whitespace if stripChars == null
            return new PyString(_lstrip(stripChars));
        }
    }

    /**
     * Implementation of Python <code>str.lstrip()</code> common to exposed and Java API, when
     * stripping whitespace. Any whitespace byte/character will be discarded from the left end of
     * this <code>str</code>.
     * <p>
     * Implementation note: although a str contains only bytes, this method is also called by
     * {@link PyUnicode#unicode_lstrip(PyObject)} when this is a basic-plane string.
     *
     * @return a new String, stripped of the whitespace characters/bytes
     */
    protected final String _lstrip() {
        String s = getString();
        // Leftmost non-whitespace character: cannot exceed length
        int left = _stripLeft(s, s.length());
        return s.substring(left);
    }

    /**
     * Implementation of Python <code>str.lstrip()</code> common to exposed and Java API. Any
     * byte/character matching one of those in <code>stripChars</code> will be discarded from the
     * left end of this <code>str</code>. If <code>stripChars == null</code>, whitespace will be
     * stripped.
     * <p>
     * Implementation note: although a <code>str</code> contains only bytes, this method is also
     * called by {@link PyUnicode#unicode_lstrip(PyObject)} when both arguments are basic-plane
     * strings.
     *
     * @param stripChars characters to strip or null
     * @return a new String, stripped of the specified characters/bytes
     */
    protected final String _lstrip(String stripChars) {
        if (stripChars == null) {
            // Divert to the whitespace version
            return _lstrip();
        } else {
            String s = getString();
            // Leftmost matching character: cannot exceed length
            int left = _stripLeft(s, stripChars, s.length());
            return s.substring(left);
        }
    }

    /**
     * Equivalent of Python <code>str.rstrip()</code> with no argument, meaning strip whitespace.
     * Any whitespace byte/character will be discarded from the right end of this <code>str</code>.
     *
     * @return a new String, stripped of the whitespace characters/bytes
     */
    public String rstrip() {
        return _rstrip();
    }

    /**
     * Equivalent of Python <code>str.rstrip()</code>.
     *
     * @param stripChars characters to strip from either end of this str/bytes, or null
     * @return a new String, stripped of the specified characters/bytes
     */
    public String rstrip(String sep) {
        return _rstrip(sep);
    }

    /**
     * Equivalent of Python <code>str.rstrip()</code>. Any byte/character matching one of those in
     * <code>stripChars</code> will be discarded from the right end of this <code>str</code>. If
     * <code>stripChars == null</code>, whitespace will be stripped. If <code>stripChars</code> is a
     * <code>PyUnicode</code>, the result will also be a <code>PyUnicode</code>.
     *
     * @param stripChars characters to strip from the right end of this str/bytes, or null
     * @return a new <code>PyString</code> (or {@link PyUnicode}), stripped of the specified
     *         characters/bytes
     */
    public PyObject rstrip(PyObject sep) {
        return str_rstrip(sep);
    }

    @ExposedMethod(defaults = "null", doc = BuiltinDocs.str_rstrip_doc)
    final PyObject str_rstrip(PyObject chars) {
        if (chars instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_rstrip(chars);
        } else {
            // It ought to be None, null, some kind of bytes with the buffer API.
            String stripChars = asStringNullOrError(chars, "rstrip");
            // Strip specified characters or whitespace if stripChars == null
            return new PyString(_rstrip(stripChars));
        }
    }

    /**
     * Implementation of Python <code>str.rstrip()</code> common to exposed and Java API, when
     * stripping whitespace. Any whitespace byte/character will be discarded from the right end of
     * this <code>str</code>.
     * <p>
     * Implementation note: although a <code>str</code> contains only bytes, this method is also
     * called by {@link PyUnicode#unicode_rstrip(PyObject)} when this is a basic-plane string.
     *
     * @return a new String, stripped of the whitespace characters/bytes
     */
    protected final String _rstrip() {
        String s = getString();
        // Rightmost non-whitespace
        int right = _stripRight(s);
        if (right < 0) {
            // They're all whitespace
            return "";
        } else {
            // Substring up to and including this rightmost non-whitespace
            return s.substring(0, right + 1);
        }
    }

    /**
     * Implementation of Python <code>str.rstrip()</code> common to exposed and Java API. Any
     * byte/character matching one of those in <code>stripChars</code> will be discarded from the
     * right end of this <code>str</code>. If <code>stripChars == null</code>, whitespace will be
     * stripped.
     * <p>
     * Implementation note: although a <code>str</code> contains only bytes, this method is also
     * called by {@link PyUnicode#unicode_strip(PyObject)} when both arguments are basic-plane
     * strings.
     *
     * @param stripChars characters to strip or null
     * @return a new String, stripped of the specified characters/bytes
     */
    protected final String _rstrip(String stripChars) {
        if (stripChars == null) {
            // Divert to the whitespace version
            return _rstrip();
        } else {
            String s = getString();
            // Rightmost non-matching character
            int right = _stripRight(s, stripChars);
            // Substring up to and including this rightmost non-matching character (or "")
            return s.substring(0, right + 1);
        }
    }

    /**
     * Equivalent to Python <code>str.split()</code>, splitting on runs of whitespace.
     *
     * @return list(str) result
     */
    public PyList split() {
        return _split(null, -1);
    }

    /**
     * Equivalent to Python <code>str.split()</code>, splitting on a specified string.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @return list(str) result
     */
    public PyList split(String sep) {
        return _split(sep, -1);
    }

    /**
     * Equivalent to Python <code>str.split()</code>, splitting on a specified string.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @param maxsplit maximum number of splits to make (there may be <code>maxsplit+1</code>
     *            parts).
     * @return list(str) result
     */
    public PyList split(String sep, int maxsplit) {
        return _split(sep, maxsplit);
    }

    /**
     * Equivalent to Python <code>str.split()</code> returning a {@link PyList} of
     * <code>PyString</code>s (or <code>PyUnicode</code>s). The <code>str</code> will be split at
     * each occurrence of <code>sep</code>. If <code>sep == null</code>, whitespace will be used as
     * the criterion. If <code>sep</code> has zero length, a Python <code>ValueError</code> is
     * raised.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @return list(str) result
     */
    public PyList split(PyObject sep) {
        return str_split(sep, -1);
    }

    /**
     * As {@link #split(PyObject)} but if <code>maxsplit</code> &gt;=0 and there are more feasible
     * splits than <code>maxsplit</code>, the last element of the list contains the rest of the
     * string.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @param maxsplit maximum number of splits to make (there may be <code>maxsplit+1</code>
     *            parts).
     * @return list(str) result
     */
    public PyList split(PyObject sep, int maxsplit) {
        return str_split(sep, maxsplit);
    }

    @ExposedMethod(defaults = {"null", "-1"}, doc = BuiltinDocs.str_split_doc)
    final PyList str_split(PyObject sepObj, int maxsplit) {
        if (sepObj instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_split(sepObj, maxsplit);
        } else {
            // It ought to be None, null, some kind of bytes with the buffer API.
            String sep = asStringNullOrError(sepObj, "split");
            // Split on specified string or whitespace if sep == null
            return _split(sep, maxsplit);
        }
    }

    /**
     * Implementation of Python str.split() common to exposed and Java API returning a
     * {@link PyList} of <code>PyString</code>s. The <code>str</code> will be split at each
     * occurrence of <code>sep</code>. If <code>sep == null</code>, whitespace will be used as the
     * criterion. If <code>sep</code> has zero length, a Python <code>ValueError</code> is raised.
     * If <code>maxsplit</code> &gt;=0 and there are more feasible splits than <code>maxsplit</code>
     * the last element of the list contains the what is left over after the last split.
     * <p>
     * Implementation note: although a str contains only bytes, this method is also called by
     * {@link PyUnicode#unicode_split(PyObject)}.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @param maxsplit maximum number of splits to make (there may be <code>maxsplit+1</code>
     *            parts).
     * @return list(str) result
     */
    protected final PyList _split(String sep, int maxsplit) {
        if (sep == null) {
            // Split on runs of whitespace
            return splitfields(maxsplit);
        } else if (sep.length() == 0) {
            throw Py.ValueError("empty separator");
        } else {
            // Split on specified (non-empty) string
            return splitfields(sep, maxsplit);
        }
    }

    /**
     * Helper function for <code>.split</code>, in <code>str</code> and <code>unicode</code>,
     * splitting on white space and returning a list of the separated parts. If there are more than
     * <code>maxsplit</code> feasible the last element of the list is the remainder of the original
     * (this) string. The split sections will be {@link PyUnicode} if this object is a
     * <code>PyUnicode</code>.
     *
     * @param maxsplit limit on the number of splits (if &gt;=0)
     * @return <code>PyList</code> of split sections
     */
    private PyList splitfields(int maxsplit) {
        /*
         * Result built here is a list of split parts, exactly as required for s.split(None,
         * maxsplit). If there are to be n splits, there will be n+1 elements in L.
         */
        PyList list = new PyList();

        String s = getString();
        int length = s.length(), start = 0, splits = 0, index;

        if (maxsplit < 0) {
            // Make all possible splits: there can't be more than:
            maxsplit = length;
        }

        // start is always the first character not consumed into a piece on the list
        while (start < length) {

            // Find the next occurrence of non-whitespace
            while (start < length) {
                if (!Character.isWhitespace(s.charAt(start))) {
                    // Break leaving start pointing at non-whitespace
                    break;
                }
                start++;
            }

            if (start >= length) {
                // Only found whitespace so there is no next segment
                break;

            } else if (splits >= maxsplit) {
                // The next segment is the last and contains all characters up to the end
                index = length;

            } else {
                // The next segment runs up to the next next whitespace or end
                for (index = start; index < length; index++) {
                    if (Character.isWhitespace(s.charAt(index))) {
                        // Break leaving index pointing at whitespace
                        break;
                    }
                }
            }

            // Make a piece from start up to index
            list.append(fromSubstring(start, index));
            splits++;

            // Start next segment search at that point
            start = index;
        }

        return list;
    }

    /**
     * Helper function for <code>.split</code> and <code>.replace</code>, in <code>str</code> and
     * <code>unicode</code>, returning a list of the separated parts. If there are more than
     * <code>maxsplit</code> occurrences of <code>sep</code> the last element of the list is the
     * remainder of the original (this) string. If <code>sep</code> is the zero-length string, the
     * split is between each character (as needed by <code>.replace</code>). The split sections will
     * be {@link PyUnicode} if this object is a <code>PyUnicode</code>.
     *
     * @param sep at occurrences of which this string should be split
     * @param maxsplit limit on the number of splits (if &gt;=0)
     * @return <code>PyList</code> of split sections
     */
    private PyList splitfields(String sep, int maxsplit) {
        /*
         * Result built here is a list of split parts, exactly as required for s.split(sep), or to
         * produce the result of s.replace(sep, r) by a subsequent call r.join(L). If there are to
         * be n splits, there will be n+1 elements in L.
         */
        PyList list = new PyList();

        String s = getString();
        int length = s.length();
        int sepLength = sep.length();

        if (maxsplit < 0) {
            // Make all possible splits: there can't be more than:
            maxsplit = length + 1;
        }

        if (maxsplit == 0) {
            // Degenerate case
            list.append(this);

        } else if (sepLength == 0) {
            /*
             * The separator is "". This cannot happen with s.split(""), as that's an error, but it
             * is used by s.replace("", A) and means that the result should be A interleaved between
             * the characters of s, before the first, and after the last, the number always limited
             * by maxsplit.
             */

            // There will be m+1 parts, where m = maxsplit or length+1 whichever is smaller.
            int m = (maxsplit > length) ? length + 1 : maxsplit;

            // Put an empty string first to make one split before the first character
            list.append(createInstance("")); // PyString or PyUnicode as this class
            int index;

            // Add m-1 pieces one character long
            for (index = 0; index < m - 1; index++) {
                list.append(fromSubstring(index, index + 1));
            }

            // And add the last piece, so there are m+1 splits (m+1 pieces)
            list.append(fromSubstring(index, length));

        } else {
            // Index of first character not yet in a piece on the list
            int start = 0;

            // Add at most maxsplit pieces
            for (int splits = 0; splits < maxsplit; splits++) {

                // Find the next occurrence of sep
                int index = s.indexOf(sep, start);

                if (index < 0) {
                    // No more occurrences of sep: we're done
                    break;

                } else {
                    // Make a piece from start up to where we found sep
                    list.append(fromSubstring(start, index));
                    // New start (of next piece) is just after sep
                    start = index + sepLength;
                }
            }

            // Last piece is the rest of the string (even if start==length)
            list.append(fromSubstring(start, length));
        }

        return list;
    }

    /**
     * Equivalent to Python <code>str.rsplit()</code>, splitting on runs of whitespace.
     *
     * @return list(str) result
     */
    public PyList rsplit() {
        return _rsplit(null, -1);
    }

    /**
     * Equivalent to Python <code>str.rsplit()</code>, splitting on a specified string.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @return list(str) result
     */
    public PyList rsplit(String sep) {
        return _rsplit(sep, -1);
    }

    /**
     * Equivalent to Python <code>str.rsplit()</code>, splitting on a specified string.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @param maxsplit maximum number of splits to make (there may be <code>maxsplit+1</code>
     *            parts).
     * @return list(str) result
     */
    public PyList rsplit(String sep, int maxsplit) {
        return _rsplit(sep, maxsplit);
    }

    /**
     * Equivalent to Python <code>str.rsplit()</code> returning a {@link PyList} of
     * <code>PyString</code>s (or <code>PyUnicode</code>s). The <code>str</code> will be split at
     * each occurrence of <code>sep</code>, working from the right. If <code>sep == null</code>,
     * whitespace will be used as the criterion. If <code>sep</code> has zero length, a Python
     * <code>ValueError</code> is raised.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @return list(str) result
     */
    public PyList rsplit(PyObject sep) {
        return str_rsplit(sep, -1);
    }

    /**
     * As {@link #rsplit(PyObject)} but if <code>maxsplit</code> &gt;=0 and there are more feasible
     * splits than <code>maxsplit</code> the last element of the list contains the rest of the
     * string.
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @param maxsplit maximum number of splits to make (there may be <code>maxsplit+1</code>
     *            parts).
     * @return list(str) result
     */
    public PyList rsplit(PyObject sep, int maxsplit) {
        return str_rsplit(sep, maxsplit);
    }

    @ExposedMethod(defaults = {"null", "-1"}, doc = BuiltinDocs.str_split_doc)
    final PyList str_rsplit(PyObject sepObj, int maxsplit) {
        if (sepObj instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_rsplit(sepObj, maxsplit);
        } else {
            // It ought to be None, null, some kind of bytes with the buffer API.
            String sep = asStringNullOrError(sepObj, "rsplit");
            // Split on specified string or whitespace if sep == null
            return _rsplit(sep, maxsplit);
        }
    }

    /**
     * Implementation of Python <code>str.rsplit()</code> common to exposed and Java API returning a
     * {@link PyList} of <code>PyString</code>s. The <code>str</code> will be split at each
     * occurrence of <code>sep</code>, working from the right. If <code>sep == null</code>,
     * whitespace will be used as the criterion. If <code>sep</code> has zero length, a Python
     * <code>ValueError</code> is raised. If <code>maxsplit</code> &gt;=0 and there are more
     * feasible splits than <code>maxsplit</code> the first element of the list contains the what is
     * left over after the last split.
     * <p>
     * Implementation note: although a str contains only bytes, this method is also called by
     * {@link PyUnicode#unicode_rsplit(PyObject)} .
     *
     * @param sep string to use as separator (or <code>null</code> if to split on whitespace)
     * @param maxsplit maximum number of splits to make (there may be <code>maxsplit+1</code>
     *            parts).
     * @return list(str) result
     */
    protected final PyList _rsplit(String sep, int maxsplit) {
        if (sep == null) {
            // Split on runs of whitespace
            return rsplitfields(maxsplit);
        } else if (sep.length() == 0) {
            throw Py.ValueError("empty separator");
        } else {
            // Split on specified (non-empty) string
            return rsplitfields(sep, maxsplit);
        }
    }

    /**
     * Helper function for <code>.rsplit</code>, in <code>str</code> and <code>unicode</code>,
     * splitting on white space and returning a list of the separated parts. If there are more than
     * <code>maxsplit</code> feasible the first element of the list is the remainder of the original
     * (this) string. The split sections will be {@link PyUnicode} if this object is a
     * <code>PyUnicode</code>.
     *
     * @param maxsplit limit on the number of splits (if &gt;=0)
     * @return <code>PyList</code> of split sections
     */
    private PyList rsplitfields(int maxsplit) {
        /*
         * Result built here (in reverse) is a list of split parts, exactly as required for
         * s.rsplit(None, maxsplit). If there are to be n splits, there will be n+1 elements.
         */
        PyList list = new PyList();

        String s = getString();
        int length = s.length(), end = length - 1, splits = 0, index;

        if (maxsplit < 0) {
            // Make all possible splits: there can't be more than:
            maxsplit = length;
        }

        // end is always the rightmost character not consumed into a piece on the list
        while (end >= 0) {

            // Find the next occurrence of non-whitespace (working leftwards)
            while (end >= 0) {
                if (!Character.isWhitespace(s.charAt(end))) {
                    // Break leaving end pointing at non-whitespace
                    break;
                }
                --end;
            }

            if (end < 0) {
                // Only found whitespace so there is no next segment
                break;

            } else if (splits >= maxsplit) {
                // The next segment is the last and contains all characters back to the beginning
                index = -1;

            } else {
                // The next segment runs back to the next next whitespace or beginning
                for (index = end; index >= 0; --index) {
                    if (Character.isWhitespace(s.charAt(index))) {
                        // Break leaving index pointing at whitespace
                        break;
                    }
                }
            }

            // Make a piece from index+1 start up to end+1
            list.append(fromSubstring(index + 1, end + 1));
            splits++;

            // Start next segment search at that point
            end = index;
        }

        list.reverse();
        return list;
    }

    /**
     * Helper function for <code>.rsplit</code>, in <code>str</code> and <code>unicode</code>,
     * returning a list of the separated parts, <em>in the reverse order</em> of their occurrence in
     * this string. If there are more than <code>maxsplit</code> occurrences of <code>sep</code> the
     * first element of the list is the left end of the original (this) string. The split sections
     * will be {@link PyUnicode} if this object is a <code>PyUnicode</code>.
     *
     * @param sep at occurrences of which this string should be split
     * @param maxsplit limit on the number of splits (if &gt;=0)
     * @return <code>PyList</code> of split sections
     */
    private PyList rsplitfields(String sep, int maxsplit) {
        /*
         * Result built here (in reverse) is a list of split parts, exactly as required for
         * s.rsplit(sep, maxsplit). If there are to be n splits, there will be n+1 elements.
         */
        PyList list = new PyList();

        String s = getString();
        int length = s.length();
        int sepLength = sep.length();

        if (maxsplit < 0) {
            // Make all possible splits: there can't be more than:
            maxsplit = length + 1;
        }

        if (maxsplit == 0) {
            // Degenerate case
            list.append(this);

        } else if (sepLength == 0) {
            // Empty separator is not allowed
            throw Py.ValueError("empty separator");

        } else {
            // Index of first character of the last piece already on the list
            int end = length;

            // Add at most maxsplit pieces
            for (int splits = 0; splits < maxsplit; splits++) {

                // Find the next occurrence of sep (working leftwards)
                int index = s.lastIndexOf(sep, end - sepLength);

                if (index < 0) {
                    // No more occurrences of sep: we're done
                    break;

                } else {
                    // Make a piece from where we found sep up to end
                    list.append(fromSubstring(index + sepLength, end));
                    // New end (of next piece) is where we found sep
                    end = index;
                }
            }

            // Last piece is the rest of the string (even if end==0)
            list.append(fromSubstring(0, end));
        }

        list.reverse();
        return list;
    }

    /**
     * Equivalent to Python <code>str.partition()</code>, splits the <code>PyString</code> at the
     * first occurrence of <code>sepObj</code> returning a {@link PyTuple} containing the part
     * before the separator, the separator itself, and the part after the separator.
     *
     * @param sepObj str, unicode or object implementing {@link BufferProtocol}
     * @return tuple of parts
     */
    public PyTuple partition(PyObject sepObj) {
        return str_partition(sepObj);
    }

    @ExposedMethod(doc = BuiltinDocs.str_partition_doc)
    final PyTuple str_partition(PyObject sepObj) {

        if (sepObj instanceof PyUnicode) {
            // Deal with Unicode separately
            return unicodePartition(sepObj);

        } else {
            // It ought to be some kind of bytes with the buffer API.
            String sep = asStringOrError(sepObj);

            if (sep.length() == 0) {
                throw Py.ValueError("empty separator");
            }

            int index = getString().indexOf(sep);
            if (index != -1) {
                return new PyTuple(fromSubstring(0, index), sepObj, fromSubstring(
                        index + sep.length(), getString().length()));
            } else {
                return new PyTuple(this, Py.EmptyString, Py.EmptyString);
            }
        }
    }

    final PyTuple unicodePartition(PyObject sepObj) {
        PyUnicode strObj = __unicode__();
        String str = strObj.getString();

        // Will throw a TypeError if not a basestring
        String sep = sepObj.asString();
        sepObj = sepObj.__unicode__();

        if (sep.length() == 0) {
            throw Py.ValueError("empty separator");
        }

        int index = str.indexOf(sep);
        if (index != -1) {
            return new PyTuple(strObj.fromSubstring(0, index), sepObj, strObj.fromSubstring(index
                    + sep.length(), str.length()));
        } else {
            PyUnicode emptyUnicode = Py.newUnicode("");
            return new PyTuple(this, emptyUnicode, emptyUnicode);
        }
    }

    /**
     * Equivalent to Python <code>str.rpartition()</code>, splits the <code>PyString</code> at the
     * last occurrence of <code>sepObj</code> returning a {@link PyTuple} containing the part before
     * the separator, the separator itself, and the part after the separator.
     *
     * @param sepObj str, unicode or object implementing {@link BufferProtocol}
     * @return tuple of parts
     */
    public PyTuple rpartition(PyObject sepObj) {
        return str_rpartition(sepObj);
    }

    @ExposedMethod(doc = BuiltinDocs.str_rpartition_doc)
    final PyTuple str_rpartition(PyObject sepObj) {

        if (sepObj instanceof PyUnicode) {
            // Deal with Unicode separately
            return unicodeRpartition(sepObj);

        } else {
            // It ought to be some kind of bytes with the buffer API.
            String sep = asStringOrError(sepObj);

            if (sep.length() == 0) {
                throw Py.ValueError("empty separator");
            }

            int index = getString().lastIndexOf(sep);
            if (index != -1) {
                return new PyTuple(fromSubstring(0, index), sepObj, fromSubstring(
                        index + sep.length(), getString().length()));
            } else {
                return new PyTuple(Py.EmptyString, Py.EmptyString, this);
            }
        }
    }

    final PyTuple unicodeRpartition(PyObject sepObj) {
        PyUnicode strObj = __unicode__();
        String str = strObj.getString();

        // Will throw a TypeError if not a basestring
        String sep = sepObj.asString();
        sepObj = sepObj.__unicode__();

        if (sep.length() == 0) {
            throw Py.ValueError("empty separator");
        }

        int index = str.lastIndexOf(sep);
        if (index != -1) {
            return new PyTuple(strObj.fromSubstring(0, index), sepObj, strObj.fromSubstring(index
                    + sep.length(), str.length()));
        } else {
            PyUnicode emptyUnicode = Py.newUnicode("");
            return new PyTuple(emptyUnicode, emptyUnicode, this);
        }
    }

    public PyList splitlines() {
        return str_splitlines(false);
    }

    public PyList splitlines(boolean keepends) {
        return str_splitlines(keepends);
    }

    @ExposedMethod(defaults = "false", doc = BuiltinDocs.str_splitlines_doc)
    final PyList str_splitlines(boolean keepends) {
        PyList list = new PyList();

        char[] chars = getString().toCharArray();
        int n = chars.length;

        int j = 0;
        for (int i = 0; i < n;) {
            /* Find a line and append it */
            while (i < n && chars[i] != '\n' && chars[i] != '\r'
                    && Character.getType(chars[i]) != Character.LINE_SEPARATOR) {
                i++;
            }

            /* Skip the line break reading CRLF as one line break */
            int eol = i;
            if (i < n) {
                if (chars[i] == '\r' && i + 1 < n && chars[i + 1] == '\n') {
                    i += 2;
                } else {
                    i++;
                }
                if (keepends) {
                    eol = i;
                }
            }
            list.append(fromSubstring(j, eol));
            j = i;
        }
        if (j < n) {
            list.append(fromSubstring(j, n));
        }
        return list;
    }

    /**
     * Return a new object <em>of the same type as this one</em> equal to the slice
     * <code>[begin:end]</code>. (Python end-relative indexes etc. are not supported.) Subclasses (
     * {@link PyUnicode#fromSubstring(int, int)}) override this to return their own type.)
     *
     * @param begin first included character.
     * @param end first excluded character.
     * @return new object.
     */
    protected PyString fromSubstring(int begin, int end) {
        return createInstance(getString().substring(begin, end), true);
    }

    /**
     * Return the lowest index in the string where substring <code>sub</code> is found. Raises
     * <code>ValueError</code> if the substring is not found.
     *
     * @param sub substring to find.
     * @return index of <code>sub</code> in this object.
     * @throws PyException(ValueError) if not found.
     */
    public int index(PyObject sub) {
        return str_index(sub, null, null);
    }

    /**
     * Return the lowest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:]</code>. Raises
     * <code>ValueError</code> if the substring is not found.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @return index of <code>sub</code> in this object.
     * @throws PyException(ValueError) if not found.
     */
    public int index(PyObject sub, PyObject start) throws PyException {
        return str_index(sub, start, null);
    }

    /**
     * Return the lowest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:end]</code>. Arguments
     * <code>start</code> and <code>end</code> are interpreted as in slice notation, with null or
     * {@link Py#None} representing "missing". Raises <code>ValueError</code> if the substring is
     * not found.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return index of <code>sub</code> in this object.
     * @throws PyException(ValueError) if not found.
     */
    public int index(PyObject sub, PyObject start, PyObject end) throws PyException {
        return checkIndex(str_index(sub, start, end));
    }

    /** Equivalent to {@link #index(PyObject)} specialized to <code>String</code>. */
    public int index(String sub) {
        return index(sub, null, null);
    }

    /** Equivalent to {@link #index(PyObject, PyObject)} specialized to <code>String</code>. */
    public int index(String sub, PyObject start) {
        return index(sub, start, null);
    }

    /**
     * Equivalent to {@link #index(PyObject, PyObject, PyObject)} specialized to <code>String</code>
     * .
     */
    public int index(String sub, PyObject start, PyObject end) {
        return checkIndex(_find(sub, start, end));
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_index_doc)
    final int str_index(PyObject subObj, PyObject start, PyObject end) {
        return checkIndex(str_find(subObj, start, end));
    }

    /**
     * Return the highest index in the string where substring <code>sub</code> is found. Raises
     * <code>ValueError</code> if the substring is not found.
     *
     * @param sub substring to find.
     * @return index of <code>sub</code> in this object.
     * @throws PyException(ValueError) if not found.
     */
    public int rindex(PyObject sub) {
        return str_rindex(sub, null, null);
    }

    /**
     * Return the highest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:]</code>. Raises
     * <code>ValueError</code> if the substring is not found.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @return index of <code>sub</code> in this object.
     * @throws PyException(ValueError) if not found.
     */
    public int rindex(PyObject sub, PyObject start) throws PyException {
        return str_rindex(sub, start, null);
    }

    /**
     * Return the highest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:end]</code>. Arguments
     * <code>start</code> and <code>end</code> are interpreted as in slice notation, with null or
     * {@link Py#None} representing "missing". Raises <code>ValueError</code> if the substring is
     * not found.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return index of <code>sub</code> in this object.
     * @throws PyException(ValueError) if not found.
     */
    public int rindex(PyObject sub, PyObject start, PyObject end) throws PyException {
        return checkIndex(str_rindex(sub, start, end));
    }

    /** Equivalent to {@link #rindex(PyObject)} specialized to <code>String</code>. */
    public int rindex(String sub) {
        return rindex(sub, null, null);
    }

    /** Equivalent to {@link #rindex(PyObject, PyObject)} specialized to <code>String</code>. */
    public int rindex(String sub, PyObject start) {
        return rindex(sub, start, null);
    }

    /**
     * Equivalent to {@link #rindex(PyObject, PyObject, PyObject)} specialized to
     * <code>String</code>.
     */
    public int rindex(String sub, PyObject start, PyObject end) {
        return checkIndex(_rfind(sub, start, end));
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_rindex_doc)
    final int str_rindex(PyObject subObj, PyObject start, PyObject end) {
        return checkIndex(str_rfind(subObj, start, end));
    }

    /**
     * A little helper for converting str.find to str.index that will raise
     * <code>ValueError("substring not found")</code> if the argument is negative, otherwise passes
     * the argument through.
     *
     * @param index to check
     * @return <code>index</code> if non-negative
     * @throws PyException(ValueError) if not found
     */
    protected final int checkIndex(int index) throws PyException {
        if (index >= 0) {
            return index;
        } else {
            throw Py.ValueError("substring not found");
        }
    }

    /**
     * Return the number of non-overlapping occurrences of substring <code>sub</code>.
     *
     * @param sub substring to find.
     * @return count of occurrences.
     */
    public int count(PyObject sub) {
        return count(sub, null, null);
    }

    /**
     * Return the number of non-overlapping occurrences of substring <code>sub</code> in the range
     * <code>[start:]</code>.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @return count of occurrences.
     */
    public int count(PyObject sub, PyObject start) {
        return count(sub, start, null);
    }

    /**
     * Return the number of non-overlapping occurrences of substring <code>sub</code> in the range
     * <code>[start:end]</code>. Optional arguments <code>start</code> and <code>end</code> are
     * interpreted as in slice notation.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return count of occurrences.
     */
    public int count(PyObject sub, PyObject start, PyObject end) {
        return str_count(sub, start, end);
    }

    /** Equivalent to {@link #count(PyObject)} specialized to <code>String</code>. */
    public int count(String sub) {
        return count(sub, null, null);
    }

    /** Equivalent to {@link #count(PyObject, PyObject)} specialized to <code>String</code>. */
    public int count(String sub, PyObject start) {
        return count(sub, start, null);
    }

    /**
     * Equivalent to {@link #count(PyObject, PyObject, PyObject)} specialized to <code>String</code>
     * .
     */
    public int count(String sub, PyObject start, PyObject end) {
        return _count(sub, start, end);
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_count_doc)
    final int str_count(PyObject subObj, PyObject start, PyObject end) {
        if (subObj instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_count(subObj, start, end);
        } else {
            // It ought to be some kind of bytes with the buffer API.
            String sub = asStringOrError(subObj);
            return _count(sub, start, end);
        }
    }

    /**
     * Helper common to the Python and Java API returning the number of occurrences of a substring.
     * It accepts slice-like arguments, which may be <code>None</code> or end-relative (negative).
     * This method also supports {@link PyUnicode#unicode_count(PyObject, PyObject, PyObject)}.
     *
     * @param sub substring to find.
     * @param startObj start of slice.
     * @param endObj end of slice.
     * @return count of occurrences
     */
    protected final int _count_old(String sub, PyObject startObj, PyObject endObj) {
// xxx
        // Interpret the slice indices as concrete values
        int[] indices = translateIndices(startObj, endObj);
        int subLen = sub.length();

        if (subLen == 0) {
            // Special case counting the occurrences of an empty string
            if (indices[2] > getString().length()) {
                return 0;
            } else {
                return indices[1] - indices[0] + 1;
            }

        } else {
            // Skip down this string finding occurrences of sub
            int start = indices[0], end = indices[1], count = 0;
            while (true) {
                int index = getString().indexOf(sub, start);
                if (index < 0) {
                    break; // not found
                } else {
                    // Found at index. Next search begins at end of this instance, at:
                    start = index + subLen;
                    if (start <= end) {
                        count += 1; // ... and the instance found fits within this string.
                    } else {
                        break; // ... but the instance found overlaps the end, so is not valid.
                    }
                }
            }
            return count;
        }
    }

    protected final int _count(String sub, PyObject startObj, PyObject endObj) {

        // Interpret the slice indices as concrete values
        int[] indices = translateIndices(startObj, endObj);
        int subLen = sub.length();

        if (subLen == 0) {
            // Special case counting the occurrences of an empty string
            if (indices[2] > getString().length()) {
                return 0;
            } else {
                return indices[1] - indices[0] + 1;
            }

        } else {

            // Skip down this string finding occurrences of sub
            int start = indices[0], limit = indices[1] - subLen, count = 0;

            while (start <= limit) {
                int index = getString().indexOf(sub, start);
                if (index >= 0 && index <= limit) {
                    // Found at index.
                    count += 1;
                    // Next search begins after this instance, at:
                    start = index + subLen;
                } else {
                    // not found, or found too far right (index>limit)
                    break;
                }
            }
            return count;
        }
    }

    /**
     * Return the lowest index in the string where substring <code>sub</code> is found.
     *
     * @param sub substring to find.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    public int find(PyObject sub) {
        return find(sub, null, null);
    }

    /**
     * Return the lowest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:]</code>.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    public int find(PyObject sub, PyObject start) {
        return find(sub, start, null);
    }

    /**
     * Return the lowest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:end]</code>. Arguments
     * <code>start</code> and <code>end</code> are interpreted as in slice notation, with null or
     * {@link Py#None} representing "missing".
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    public int find(PyObject sub, PyObject start, PyObject end) {
        return str_find(sub, start, end);
    }

    /** Equivalent to {@link #find(PyObject)} specialized to <code>String</code>. */
    public int find(String sub) {
        return find(sub, null, null);
    }

    /** Equivalent to {@link #find(PyObject, PyObject)} specialized to <code>String</code>. */
    public int find(String sub, PyObject start) {
        return find(sub, start, null);
    }

    /**
     * Equivalent to {@link #find(PyObject, PyObject, PyObject)} specialized to <code>String</code>.
     */
    public int find(String sub, PyObject start, PyObject end) {
        return _find(sub, start, end);
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_find_doc)
    final int str_find(PyObject subObj, PyObject start, PyObject end) {
        if (subObj instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_find(subObj, start, end);
        } else {
            // It ought to be some kind of bytes with the buffer API.
            String sub = asStringOrError(subObj);
            return _find(sub, start, end);
        }
    }

    /**
     * Helper common to the Python and Java API returning the index of the substring or -1 for not
     * found. It accepts slice-like arguments, which may be <code>None</code> or end-relative
     * (negative). This method also supports
     * {@link PyUnicode#unicode_find(PyObject, PyObject, PyObject)}.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    protected final int _find(String sub, PyObject start, PyObject end) {
        int[] indices = translateIndices(start, end);
        int index = getString().indexOf(sub, indices[0]);
        if (index < indices[2] || index > indices[1]) {
            return -1;
        }
        return index;
    }

    /**
     * Return the highest index in the string where substring <code>sub</code> is found.
     *
     * @param sub substring to find.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    public int rfind(PyObject sub) {
        return rfind(sub, null, null);
    }

    /**
     * Return the highest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:]</code>.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    public int rfind(PyObject sub, PyObject start) {
        return rfind(sub, start, null);
    }

    /**
     * Return the highest index in the string where substring <code>sub</code> is found, such that
     * <code>sub</code> is contained in the slice <code>s[start:end]</code>. Arguments
     * <code>start</code> and <code>end</code> are interpreted as in slice notation, with null or
     * {@link Py#None} representing "missing".
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    public int rfind(PyObject sub, PyObject start, PyObject end) {
        return str_rfind(sub, start, end);
    }

    /** Equivalent to {@link #find(PyObject)} specialized to <code>String</code>. */
    public int rfind(String sub) {
        return rfind(sub, null, null);
    }

    /** Equivalent to {@link #find(PyObject, PyObject)} specialized to <code>String</code>. */
    public int rfind(String sub, PyObject start) {
        return rfind(sub, start, null);
    }

    /**
     * Equivalent to {@link #find(PyObject, PyObject, PyObject)} specialized to <code>String</code>.
     */
    public int rfind(String sub, PyObject start, PyObject end) {
        return _rfind(sub, start, end);
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_rfind_doc)
    final int str_rfind(PyObject subObj, PyObject start, PyObject end) {
        if (subObj instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_rfind(subObj, start, end);
        } else {
            // It ought to be some kind of bytes with the buffer API.
            String sub = asStringOrError(subObj);
            return _rfind(sub, start, end);
        }
    }

    /**
     * Helper common to the Python and Java API returning the last index of the substring or -1 for
     * not found. It accepts slice-like arguments, which may be <code>None</code> or end-relative
     * (negative). This method also supports
     * {@link PyUnicode#unicode_rfind(PyObject, PyObject, PyObject)}.
     *
     * @param sub substring to find.
     * @param start start of slice.
     * @param end end of slice.
     * @return index of <code>sub</code> in this object or -1 if not found.
     */
    protected final int _rfind(String sub, PyObject start, PyObject end) {
        int[] indices = translateIndices(start, end);
        int index = getString().lastIndexOf(sub, indices[1] - sub.length());
        if (index < indices[2]) {
            return -1;
        }
        return index;
    }

    public double atof() {
        StringBuilder s = null;
        int n = getString().length();
        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);
            if (ch == '\u0000') {
                throw Py.ValueError("null byte in argument for float()");
            }
            if (Character.isDigit(ch)) {
                if (s == null) {
                    s = new StringBuilder(getString());
                }
                int val = Character.digit(ch, 10);
                s.setCharAt(i, Character.forDigit(val, 10));
            }
        }
        String sval = getString();
        if (s != null) {
            sval = s.toString();
        }
        try {
            // Double.valueOf allows format specifier ("d" or "f") at the end
            String lowSval = sval.toLowerCase();
            if (lowSval.equals("nan")) {
                return Double.NaN;
            } else if (lowSval.equals("+nan")) {
                return Double.NaN;
            } else if (lowSval.equals("-nan")) {
                return Double.NaN;
            } else if (lowSval.equals("inf")) {
                return Double.POSITIVE_INFINITY;
            } else if (lowSval.equals("+inf")) {
                return Double.POSITIVE_INFINITY;
            } else if (lowSval.equals("-inf")) {
                return Double.NEGATIVE_INFINITY;
            } else if (lowSval.equals("infinity")) {
                return Double.POSITIVE_INFINITY;
            } else if (lowSval.equals("+infinity")) {
                return Double.POSITIVE_INFINITY;
            } else if (lowSval.equals("-infinity")) {
                return Double.NEGATIVE_INFINITY;
            }

            if (lowSval.endsWith("d") || lowSval.endsWith("f")) {
                throw new NumberFormatException("format specifiers not allowed");
            }
            return Double.valueOf(sval).doubleValue();
        } catch (NumberFormatException exc) {
            throw Py.ValueError("invalid literal for __float__: " + getString());
        }
    }

    private BigInteger asciiToBigInteger(int base, boolean isLong) {
        String str = getString();

        int b = 0;
        int e = str.length();

        while (b < e && Character.isWhitespace(str.charAt(b))) {
            b++;
        }

        while (e > b && Character.isWhitespace(str.charAt(e - 1))) {
            e--;
        }

        char sign = 0;
        if (b < e) {
            sign = str.charAt(b);
            if (sign == '-' || sign == '+') {
                b++;
                while (b < e && Character.isWhitespace(str.charAt(b))) {
                    b++;
                }
            }

            if (base == 16) {
                if (str.charAt(b) == '0') {
                    if (b < e - 1 && Character.toUpperCase(str.charAt(b + 1)) == 'X') {
                        b += 2;
                    }
                }
            } else if (base == 0) {
                if (str.charAt(b) == '0') {
                    if (b < e - 1 && Character.toUpperCase(str.charAt(b + 1)) == 'X') {
                        base = 16;
                        b += 2;
                    } else if (b < e - 1 && Character.toUpperCase(str.charAt(b + 1)) == 'O') {
                        base = 8;
                        b += 2;
                    } else if (b < e - 1 && Character.toUpperCase(str.charAt(b + 1)) == 'B') {
                        base = 2;
                        b += 2;
                    } else {
                        base = 8;
                    }
                }
            } else if (base == 8) {
                if (b < e - 1 && Character.toUpperCase(str.charAt(b + 1)) == 'O') {
                    b += 2;
                }
            } else if (base == 2) {
                if (b < e - 1 && Character.toUpperCase(str.charAt(b + 1)) == 'B') {
                    b += 2;
                }
            }
        }

        if (base == 0) {
            base = 10;
        }

        // if the base >= 22, then an 'l' or 'L' is a digit!
        if (isLong && base < 22 && e > b && (str.charAt(e - 1) == 'L' || str.charAt(e - 1) == 'l')) {
            e--;
        }

        String s = str;
        if (b > 0 || e < str.length()) {
            s = str.substring(b, e);
        }

        BigInteger bi;
        if (sign == '-') {
            bi = new BigInteger("-" + s, base);
        } else {
            bi = new BigInteger(s, base);
        }
        return bi;
    }

    public int atoi() {
        return atoi(10);
    }

    public int atoi(int base) {
        if ((base != 0 && base < 2) || (base > 36)) {
            throw Py.ValueError("invalid base for atoi()");
        }

        try {
            BigInteger bi = asciiToBigInteger(base, false);
            if (bi.compareTo(PyInteger.MAX_INT) > 0 || bi.compareTo(PyInteger.MIN_INT) < 0) {
                throw Py.OverflowError("long int too large to convert to int");
            }
            return bi.intValue();
        } catch (NumberFormatException exc) {
            throw Py.ValueError("invalid literal for int() with base " + base + ": '" + getString()
                    + "'");
        } catch (StringIndexOutOfBoundsException exc) {
            throw Py.ValueError("invalid literal for int() with base " + base + ": '" + getString()
                    + "'");
        }
    }

    public PyLong atol() {
        return atol(10);
    }

    public PyLong atol(int base) {
        if ((base != 0 && base < 2) || (base > 36)) {
            throw Py.ValueError("invalid base for long literal:" + base);
        }

        try {
            BigInteger bi = asciiToBigInteger(base, true);
            return new PyLong(bi);
        } catch (NumberFormatException exc) {
            if (this instanceof PyUnicode) {
                // TODO: here's a basic issue: do we use the BigInteger constructor
                // above, or add an equivalent to CPython's PyUnicode_EncodeDecimal;
                // we should note that the current error string does not quite match
                // CPython regardless of the codec, that's going to require some more work
                throw Py.UnicodeEncodeError("decimal", "codec can't encode character", 0, 0,
                        "invalid decimal Unicode string");
            } else {
                throw Py.ValueError("invalid literal for long() with base " + base + ": '"
                        + getString() + "'");
            }
        } catch (StringIndexOutOfBoundsException exc) {
            throw Py.ValueError("invalid literal for long() with base " + base + ": '"
                    + getString() + "'");
        }
    }

    private static String padding(int n, char pad) {
        char[] chars = new char[n];
        for (int i = 0; i < n; i++) {
            chars[i] = pad;
        }
        return new String(chars);
    }

    private static char parse_fillchar(String function, String fillchar) {
        if (fillchar == null) {
            return ' ';
        }
        if (fillchar.length() != 1) {
            throw Py.TypeError(function + "() argument 2 must be char, not str");
        }
        return fillchar.charAt(0);
    }

    public String ljust(int width) {
        return str_ljust(width, null);
    }

    public String ljust(int width, String padding) {
        return str_ljust(width, padding);
    }

    @ExposedMethod(defaults = "null", doc = BuiltinDocs.str_ljust_doc)
    final String str_ljust(int width, String fillchar) {
        char pad = parse_fillchar("ljust", fillchar);
        int n = width - getString().length();
        if (n <= 0) {
            return getString();
        }
        return getString() + padding(n, pad);
    }

    public String rjust(int width) {
        return str_rjust(width, null);
    }

    @ExposedMethod(defaults = "null", doc = BuiltinDocs.str_rjust_doc)
    final String str_rjust(int width, String fillchar) {
        char pad = parse_fillchar("rjust", fillchar);
        int n = width - getString().length();
        if (n <= 0) {
            return getString();
        }
        return padding(n, pad) + getString();
    }

    public String center(int width) {
        return str_center(width, null);
    }

    @ExposedMethod(defaults = "null", doc = BuiltinDocs.str_center_doc)
    final String str_center(int width, String fillchar) {
        char pad = parse_fillchar("center", fillchar);
        int n = width - getString().length();
        if (n <= 0) {
            return getString();
        }
        int half = n / 2;
        if (n % 2 > 0 && width % 2 > 0) {
            half += 1;
        }

        return padding(half, pad) + getString() + padding(n - half, pad);
    }

    public String zfill(int width) {
        return str_zfill(width);
    }

    @ExposedMethod(doc = BuiltinDocs.str_zfill_doc)
    final String str_zfill(int width) {
        String s = getString();
        int n = s.length();
        if (n >= width) {
            return s;
        }
        char[] chars = new char[width];
        int nzeros = width - n;
        int i = 0;
        int sStart = 0;
        if (n > 0) {
            char start = s.charAt(0);
            if (start == '+' || start == '-') {
                chars[0] = start;
                i += 1;
                nzeros++;
                sStart = 1;
            }
        }
        for (; i < nzeros; i++) {
            chars[i] = '0';
        }
        s.getChars(sStart, s.length(), chars, i);
        return new String(chars);
    }

    public String expandtabs() {
        return str_expandtabs(8);
    }

    public String expandtabs(int tabsize) {
        return str_expandtabs(tabsize);
    }

    @ExposedMethod(defaults = "8", doc = BuiltinDocs.str_expandtabs_doc)
    final String str_expandtabs(int tabsize) {
        String s = getString();
        StringBuilder buf = new StringBuilder((int)(s.length() * 1.5));
        char[] chars = s.toCharArray();
        int n = chars.length;
        int position = 0;

        for (int i = 0; i < n; i++) {
            char c = chars[i];
            if (c == '\t') {
                int spaces = tabsize - position % tabsize;
                position += spaces;
                while (spaces-- > 0) {
                    buf.append(' ');
                }
                continue;
            }
            if (c == '\n' || c == '\r') {
                position = -1;
            }
            buf.append(c);
            position++;
        }
        return buf.toString();
    }

    public String capitalize() {
        return str_capitalize();
    }

    @ExposedMethod(doc = BuiltinDocs.str_capitalize_doc)
    final String str_capitalize() {
        if (getString().length() == 0) {
            return getString();
        }
        String first = getString().substring(0, 1).toUpperCase();
        return first.concat(getString().substring(1).toLowerCase());
    }

    /**
     * Equivalent to Python str.replace(old, new), returning a copy of the string with all
     * occurrences of substring old replaced by new. If either argument is a {@link PyUnicode} (or
     * this object is), the result will be a <code>PyUnicode</code>.
     *
     * @param oldPiece to replace where found.
     * @param newPiece replacement text.
     * @param count maximum number of replacements to make, or -1 meaning all of them.
     * @return PyString (or PyUnicode if any string is one), this string after replacements.
     */
    public PyString replace(PyObject oldPieceObj, PyObject newPieceObj) {
        return str_replace(oldPieceObj, newPieceObj, -1);
    }

    /**
     * Equivalent to Python str.replace(old, new[, count]), returning a copy of the string with all
     * occurrences of substring old replaced by new. If argument <code>count</code> is nonnegative,
     * only the first <code>count</code> occurrences are replaced. If either argument is a
     * {@link PyUnicode} (or this object is), the result will be a <code>PyUnicode</code>.
     *
     * @param oldPiece to replace where found.
     * @param newPiece replacement text.
     * @param count maximum number of replacements to make, or -1 meaning all of them.
     * @return PyString (or PyUnicode if any string is one), this string after replacements.
     */
    public PyString replace(PyObject oldPieceObj, PyObject newPieceObj, int count) {
        return str_replace(oldPieceObj, newPieceObj, count);
    }

    @ExposedMethod(defaults = "-1", doc = BuiltinDocs.str_replace_doc)
    final PyString str_replace(PyObject oldPieceObj, PyObject newPieceObj, int count) {
        if (oldPieceObj instanceof PyUnicode || newPieceObj instanceof PyUnicode) {
            // Promote the problem to a Unicode one
            return ((PyUnicode)decode()).unicode_replace(oldPieceObj, newPieceObj, count);
        } else {
            // Neither is a PyUnicode: both ought to be some kind of bytes with the buffer API.
            String oldPiece = asStringOrError(oldPieceObj);
            String newPiece = asStringOrError(newPieceObj);
            return _replace(oldPiece, newPiece, count);
        }
    }

    /**
     * Helper common to the Python and Java API for <code>str.replace</code>, returning a new string
     * equal to this string with ocurrences of <code>oldPiece</code> replaced by
     * <code>newPiece</code>, up to a maximum of <code>count</code> occurrences, or all of them.
     * This method also supports {@link PyUnicode#unicode_replace(PyObject, PyObject, int)}, in
     * which context it returns a <code>PyUnicode</code>
     *
     * @param oldPiece to replace where found.
     * @param newPiece replacement text.
     * @param count maximum number of replacements to make, or -1 meaning all of them.
     * @return PyString (or PyUnicode if this string is one), this string after replacements.
     */
    protected final PyString _replace(String oldPiece, String newPiece, int count) {

        String s = getString();
        int len = s.length();
        int oldLen = oldPiece.length();
        int newLen = newPiece.length();

        if (len == 0) {
            if (count < 0 && oldLen == 0) {
                return createInstance(newPiece, true);
            }
            return createInstance(s, true);

        } else if (oldLen == 0 && newLen != 0 && count != 0) {
            /*
             * old="" and new != "", interleave new piece with each char in original, taking into
             * account count
             */
            StringBuilder buffer = new StringBuilder();
            int i = 0;
            buffer.append(newPiece);
            for (; i < len && (count < 0 || i < count - 1); i++) {
                buffer.append(s.charAt(i)).append(newPiece);
            }
            buffer.append(s.substring(i));
            return createInstance(buffer.toString(), true);

        } else {
            if (count < 0) {
                count = (oldLen == 0) ? len + 1 : len;
            }
            return createInstance(newPiece).join(splitfields(oldPiece, count));
        }
    }

    public PyString join(PyObject seq) {
        return str_join(seq);
    }

    @ExposedMethod(doc = BuiltinDocs.str_join_doc)
    final PyString str_join(PyObject obj) {
        PySequence seq = fastSequence(obj, "");
        int seqLen = seq.__len__();
        if (seqLen == 0) {
            return Py.EmptyString;
        }

        PyObject item;
        if (seqLen == 1) {
            item = seq.pyget(0);
            if (item.getType() == PyString.TYPE || item.getType() == PyUnicode.TYPE) {
                return (PyString)item;
            }
        }

        // There are at least two things to join, or else we have a subclass of the
        // builtin types in the sequence. Do a pre-pass to figure out the total amount of
        // space we'll need, see whether any argument is absurd, and defer to the Unicode
        // join if appropriate
        int i = 0;
        long size = 0;
        int sepLen = getString().length();
        for (; i < seqLen; i++) {
            item = seq.pyget(i);
            if (!(item instanceof PyString)) {
                throw Py.TypeError(String.format("sequence item %d: expected string, %.80s found",
                        i, item.getType().fastGetName()));
            }
            if (item instanceof PyUnicode) {
                // Defer to Unicode join. CAUTION: There's no gurantee that the original
                // sequence can be iterated over again, so we must pass seq here
                return unicodeJoin(seq);
            }

            if (i != 0) {
                size += sepLen;
            }
            size += ((PyString)item).getString().length();
            if (size > Integer.MAX_VALUE) {
                throw Py.OverflowError("join() result is too long for a Python string");
            }
        }

        // Catenate everything
        StringBuilder buf = new StringBuilder((int)size);
        for (i = 0; i < seqLen; i++) {
            item = seq.pyget(i);
            if (i != 0) {
                buf.append(getString());
            }
            buf.append(((PyString)item).getString());
        }
        return new PyString(buf.toString());
    }

    final PyUnicode unicodeJoin(PyObject obj) {
        PySequence seq = fastSequence(obj, "");
        // A codec may be invoked to convert str objects to Unicode, and so it's possible
        // to call back into Python code during PyUnicode_FromObject(), and so it's
        // possible for a sick codec to change the size of fseq (if seq is a list).
        // Therefore we have to keep refetching the size -- can't assume seqlen is
        // invariant.
        int seqLen = seq.__len__();
        // If empty sequence, return u""
        if (seqLen == 0) {
            return new PyUnicode();
        }

        // If singleton sequence with an exact Unicode, return that
        PyObject item;
        if (seqLen == 1) {
            item = seq.pyget(0);
            if (item.getType() == PyUnicode.TYPE) {
                return (PyUnicode)item;
            }
        }

        String sep = null;
        if (seqLen > 1) {
            if (this instanceof PyUnicode) {
                sep = getString();
            } else {
                sep = ((PyUnicode)decode()).getString();
                // In case decode()'s codec mutated seq
                seqLen = seq.__len__();
            }
        }

        // At least two items to join, or one that isn't exact Unicode
        long size = 0;
        int sepLen = getString().length();
        StringBuilder buf = new StringBuilder();
        String itemString;
        for (int i = 0; i < seqLen; i++) {
            item = seq.pyget(i);
            // Convert item to Unicode
            if (!(item instanceof PyString)) {
                throw Py.TypeError(String.format("sequence item %d: expected string or Unicode,"
                        + " %.80s found", i, item.getType().fastGetName()));
            }
            if (!(item instanceof PyUnicode)) {
                item = ((PyString)item).decode();
                // In case decode()'s codec mutated seq
                seqLen = seq.__len__();
            }
            itemString = ((PyUnicode)item).getString();

            if (i != 0) {
                size += sepLen;
                buf.append(sep);
            }
            size += itemString.length();
            if (size > Integer.MAX_VALUE) {
                throw Py.OverflowError("join() result is too long for a Python string");
            }
            buf.append(itemString);
        }
        return new PyUnicode(buf.toString());
    }

    /**
     * Equivalent to the Python <code>str.startswith</code> method testing whether a string starts
     * with a specified prefix. <code>prefix</code> can also be a tuple of prefixes to look for.
     *
     * @param prefix string to check for (or a <code>PyTuple</code> of them).
     * @return <code>true</code> if this string slice starts with a specified prefix, otherwise
     *         <code>false</code>.
     */
    public boolean startswith(PyObject prefix) {
        return str_startswith(prefix, null, null);
    }

    /**
     * Equivalent to the Python <code>str.startswith</code> method, testing whether a string starts
     * with a specified prefix, where a sub-range is specified by <code>[start:]</code>.
     * <code>start</code> is interpreted as in slice notation, with null or {@link Py#None}
     * representing "missing". <code>prefix</code> can also be a tuple of prefixes to look for.
     *
     * @param prefix string to check for (or a <code>PyTuple</code> of them).
     * @param start start of slice.
     * @return <code>true</code> if this string slice starts with a specified prefix, otherwise
     *         <code>false</code>.
     */
    public boolean startswith(PyObject prefix, PyObject offset) {
        return str_startswith(prefix, offset, null);
    }

    /**
     * Equivalent to the Python <code>str.startswith</code> method, testing whether a string starts
     * with a specified prefix, where a sub-range is specified by <code>[start:end]</code>.
     * Arguments <code>start</code> and <code>end</code> are interpreted as in slice notation, with
     * null or {@link Py#None} representing "missing". <code>prefix</code> can also be a tuple of
     * prefixes to look for.
     *
     * @param prefix string to check for (or a <code>PyTuple</code> of them).
     * @param start start of slice.
     * @param end end of slice.
     * @return <code>true</code> if this string slice starts with a specified prefix, otherwise
     *         <code>false</code>.
     */
    public boolean startswith(PyObject prefix, PyObject start, PyObject end) {
        return str_startswith(prefix, start, end);
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_startswith_doc)
    final boolean str_startswith(PyObject prefix, PyObject startObj, PyObject endObj) {
        int[] indices = translateIndices(startObj, endObj);
        int start = indices[0];
        int sliceLen = indices[1] - start;

        if (!(prefix instanceof PyTuple)) {
            // It ought to be PyUnicode or some kind of bytes with the buffer API.
            String s = asBMPStringOrError(prefix);
            // If s is non-BMP, and this is a PyString (bytes), result will correctly be false.
            return sliceLen >= s.length() && getString().startsWith(s, start);

        } else {
            // Loop will return true if this slice starts with any prefix in the tuple
            for (PyObject prefixObj : ((PyTuple)prefix).getArray()) {
                // It ought to be PyUnicode or some kind of bytes with the buffer API.
                String s = asBMPStringOrError(prefixObj);
                // If s is non-BMP, and this is a PyString (bytes), result will correctly be false.
                if (sliceLen >= s.length() && getString().startsWith(s, start)) {
                    return true;
                }
            }
            // None matched
            return false;
        }
    }

    /**
     * Equivalent to the Python <code>str.endswith</code> method, testing whether a string ends with
     * a specified suffix. <code>suffix</code> can also be a tuple of suffixes to look for.
     *
     * @param suffix string to check for (or a <code>PyTuple</code> of them).
     * @return <code>true</code> if this string slice ends with a specified suffix, otherwise
     *         <code>false</code>.
     */
    public boolean endswith(PyObject suffix) {
        return str_endswith(suffix, null, null);
    }

    /**
     * Equivalent to the Python <code>str.endswith</code> method, testing whether a string ends with
     * a specified suffix, where a sub-range is specified by <code>[start:]</code>.
     * <code>start</code> is interpreted as in slice notation, with null or {@link Py#None}
     * representing "missing". <code>suffix</code> can also be a tuple of suffixes to look for.
     *
     * @param suffix string to check for (or a <code>PyTuple</code> of them).
     * @param start start of slice.
     * @return <code>true</code> if this string slice ends with a specified suffix, otherwise
     *         <code>false</code>.
     */
    public boolean endswith(PyObject suffix, PyObject start) {
        return str_endswith(suffix, start, null);
    }

    /**
     * Equivalent to the Python <code>str.endswith</code> method, testing whether a string ends with
     * a specified suffix, where a sub-range is specified by <code>[start:end]</code>. Arguments
     * <code>start</code> and <code>end</code> are interpreted as in slice notation, with null or
     * {@link Py#None} representing "missing". <code>suffix</code> can also be a tuple of suffixes
     * to look for.
     *
     * @param suffix string to check for (or a <code>PyTuple</code> of them).
     * @param start start of slice.
     * @param end end of slice.
     * @return <code>true</code> if this string slice ends with a specified suffix, otherwise
     *         <code>false</code>.
     */
    public boolean endswith(PyObject suffix, PyObject start, PyObject end) {
        return str_endswith(suffix, start, end);
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_endswith_doc)
    final boolean str_endswith(PyObject suffix, PyObject startObj, PyObject endObj) {

        int[] indices = translateIndices(startObj, endObj);
        String substr = getString().substring(indices[0], indices[1]);

        if (!(suffix instanceof PyTuple)) {
            // It ought to be PyUnicode or some kind of bytes with the buffer API.
            String s = asBMPStringOrError(suffix);
            // If s is non-BMP, and this is a PyString (bytes), result will correctly be false.
            return substr.endsWith(s);

        } else {
            // Loop will return true if this slice ends with any suffix in the tuple
            for (PyObject suffixObj : ((PyTuple)suffix).getArray()) {
                // It ought to be PyUnicode or some kind of bytes with the buffer API.
                String s = asBMPStringOrError(suffixObj);
                // If s is non-BMP, and this is a PyString (bytes), result will correctly be false.
                if (substr.endsWith(s)) {
                    return true;
                }
            }
            // None matched
            return false;
        }
    }

    /**
     * Turns the possibly negative Python slice start and end into valid indices into this string.
     *
     * @return a 3 element array of indices into this string describing a substring from [0] to [1].
     *         [0] <= [1], [0] >= 0 and [1] <= string.length(). The third element contains the
     *         unadjusted start value (or nearest int).
     */
    protected int[] translateIndices(PyObject start, PyObject end) {
        int iStart, iStartUnadjusted, iEnd;
        int n = getString().length();

        // Make sure the slice end decodes to something in range
        if (end == null || end == Py.None) {
            iEnd = n;
        } else {
            // Convert to int but limit to Integer.MIN_VALUE <= iEnd <= Integer.MAX_VALUE
            iEnd = end.asIndex(null);
            if (iEnd > n) {
                iEnd = n;
            } else if (iEnd < 0) {
                iEnd = n + iEnd;
                if (iEnd < 0) {
                    iEnd = 0;
                }
            }
        }

        // Make sure the slice start decodes to something in range
        if (start == null || start == Py.None) {
            iStartUnadjusted = iStart = 0;
        } else {
            // Convert to int but limit to Integer.MIN_VALUE <= iStart <= Integer.MAX_VALUE
            iStartUnadjusted = iStart = start.asIndex(null);
            if (iStart > iEnd) {
                iStart = iEnd;
            } else if (iStart < 0) {
                iStart = n + iStart;
                if (iStart > iEnd) {
                    iStart = iEnd;
                } else if (iStart < 0) {
                    iStart = 0;
                }
            }
        }

        return new int[] {iStart, iEnd, iStartUnadjusted};
    }

    /**
     * Equivalent to Python <code>str.translate</code> returning a copy of this string where the
     * characters have been mapped through the translation <code>table</code>. <code>table</code>
     * must be equivalent to a string of length 256 (if it is not <code>null</code>).
     *
     * @param table of character (byte) translations (or <code>null</code>)
     * @return transformed byte string
     */
    public String translate(PyObject table) {
        return translate(table, null);
    }

    /**
     * Equivalent to Python <code>str.translate</code> returning a copy of this string where all
     * characters (bytes) occurring in the argument <code>deletechars</code> are removed (if it is
     * not <code>null</code>), and the remaining characters have been mapped through the translation
     * <code>table</code>. <code>table</code> must be equivalent to a string of length 256 (if it is
     * not <code>null</code>).
     *
     * @param table of character (byte) translations (or <code>null</code>)
     * @param deletechars set of characters to remove (or <code>null</code>)
     * @return transformed byte string
     */
    public String translate(PyObject table, PyObject deletechars) {
        return str_translate(table, deletechars);
    }

    /**
     * Equivalent to {@link #translate(PyObject)} specialized to <code>String</code>.
     */
    public String translate(String table) {
        return _translate(table, null);
    }

    /**
     * Equivalent to {@link #translate(PyObject, PyObject)} specialized to <code>String</code>.
     */
    public String translate(String table, String deletechars) {
        return _translate(table, deletechars);
    }

    @ExposedMethod(defaults = {"null", "null"}, doc = BuiltinDocs.str_translate_doc)
    final String str_translate(PyObject tableObj, PyObject deletecharsObj) {
        // Accept anythiong withthe buffer API or null
        String table = asStringNullOrError(tableObj, null);
        String deletechars = asStringNullOrError(deletecharsObj, null);
        return _translate(table, deletechars);
    }

    /**
     * Helper common to the Python and Java API implementing <code>str.translate</code> returning a
     * copy of this string where all characters (bytes) occurring in the argument
     * <code>deletechars</code> are removed (if it is not <code>null</code>), and the remaining
     * characters have been mapped through the translation <code>table</code>, which must be
     * equivalent to a string of length 256 (if it is not <code>null</code>).
     *
     * @param table of character (byte) translations (or <code>null</code>)
     * @param deletechars set of characters to remove (or <code>null</code>)
     * @return transformed byte string
     */
    private final String _translate(String table, String deletechars) {

        if (table != null && table.length() != 256) {
            throw Py.ValueError("translation table must be 256 characters long");
        }

        StringBuilder buf = new StringBuilder(getString().length());

        for (int i = 0; i < getString().length(); i++) {
            char c = getString().charAt(i);
            if (deletechars != null && deletechars.indexOf(c) >= 0) {
                continue;
            }
            if (table == null) {
                buf.append(c);
            } else {
                try {
                    buf.append(table.charAt(c));
                } catch (IndexOutOfBoundsException e) {
                    throw Py.TypeError("translate() only works for 8-bit character strings");
                }
            }
        }
        return buf.toString();
    }

    public boolean islower() {
        return str_islower();
    }

    @ExposedMethod(doc = BuiltinDocs.str_islower_doc)
    final boolean str_islower() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return Character.isLowerCase(getString().charAt(0));
        }

        boolean cased = false;
        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (Character.isUpperCase(ch) || Character.isTitleCase(ch)) {
                return false;
            } else if (!cased && Character.isLowerCase(ch)) {
                cased = true;
            }
        }
        return cased;
    }

    public boolean isupper() {
        return str_isupper();
    }

    @ExposedMethod(doc = BuiltinDocs.str_isupper_doc)
    final boolean str_isupper() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return Character.isUpperCase(getString().charAt(0));
        }

        boolean cased = false;
        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (Character.isLowerCase(ch) || Character.isTitleCase(ch)) {
                return false;
            } else if (!cased && Character.isUpperCase(ch)) {
                cased = true;
            }
        }
        return cased;
    }

    public boolean isalpha() {
        return str_isalpha();
    }

    @ExposedMethod(doc = BuiltinDocs.str_isalpha_doc)
    final boolean str_isalpha() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return Character.isLetter(getString().charAt(0));
        }

        if (n == 0) {
            return false;
        }

        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (!Character.isLetter(ch)) {
                return false;
            }
        }
        return true;
    }

    public boolean isalnum() {
        return str_isalnum();
    }

    @ExposedMethod(doc = BuiltinDocs.str_isalnum_doc)
    final boolean str_isalnum() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return _isalnum(getString().charAt(0));
        }

        if (n == 0) {
            return false;
        }

        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (!_isalnum(ch)) {
                return false;
            }
        }
        return true;
    }

    private boolean _isalnum(char ch) {
        // This can ever be entirely compatible with CPython. In CPython
        // The type is not used, the numeric property is determined from
        // the presense of digit, decimal or numeric fields. These fields
        // are not available in exactly the same way in java.
        return Character.isLetterOrDigit(ch) || Character.getType(ch) == Character.LETTER_NUMBER;
    }

    public boolean isdecimal() {
        return str_isdecimal();
    }

    @ExposedMethod(doc = BuiltinDocs.unicode_isdecimal_doc)
    final boolean str_isdecimal() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            char ch = getString().charAt(0);
            return _isdecimal(ch);
        }

        if (n == 0) {
            return false;
        }

        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (!_isdecimal(ch)) {
                return false;
            }
        }
        return true;
    }

    private boolean _isdecimal(char ch) {
        // See the comment in _isalnum. Here it is even worse.
        return Character.getType(ch) == Character.DECIMAL_DIGIT_NUMBER;
    }

    public boolean isdigit() {
        return str_isdigit();
    }

    @ExposedMethod(doc = BuiltinDocs.str_isdigit_doc)
    final boolean str_isdigit() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return Character.isDigit(getString().charAt(0));
        }

        if (n == 0) {
            return false;
        }

        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (!Character.isDigit(ch)) {
                return false;
            }
        }
        return true;
    }

    public boolean isnumeric() {
        return str_isnumeric();
    }

    @ExposedMethod(doc = BuiltinDocs.unicode_isnumeric_doc)
    final boolean str_isnumeric() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return _isnumeric(getString().charAt(0));
        }

        if (n == 0) {
            return false;
        }

        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);
            if (!_isnumeric(ch)) {
                return false;
            }
        }
        return true;
    }

    private boolean _isnumeric(char ch) {
        int type = Character.getType(ch);
        return type == Character.DECIMAL_DIGIT_NUMBER || type == Character.LETTER_NUMBER
                || type == Character.OTHER_NUMBER;
    }

    public boolean istitle() {
        return str_istitle();
    }

    @ExposedMethod(doc = BuiltinDocs.str_istitle_doc)
    final boolean str_istitle() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return Character.isTitleCase(getString().charAt(0))
                    || Character.isUpperCase(getString().charAt(0));
        }

        boolean cased = false;
        boolean previous_is_cased = false;
        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (Character.isUpperCase(ch) || Character.isTitleCase(ch)) {
                if (previous_is_cased) {
                    return false;
                }
                previous_is_cased = true;
                cased = true;
            } else if (Character.isLowerCase(ch)) {
                if (!previous_is_cased) {
                    return false;
                }
                previous_is_cased = true;
                cased = true;
            } else {
                previous_is_cased = false;
            }
        }
        return cased;
    }

    public boolean isspace() {
        return str_isspace();
    }

    @ExposedMethod(doc = BuiltinDocs.str_isspace_doc)
    final boolean str_isspace() {
        int n = getString().length();

        /* Shortcut for single character strings */
        if (n == 1) {
            return Character.isWhitespace(getString().charAt(0));
        }

        if (n == 0) {
            return false;
        }

        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);

            if (!Character.isWhitespace(ch)) {
                return false;
            }
        }
        return true;
    }

    public boolean isunicode() {
        return str_isunicode();
    }

    @ExposedMethod(doc = "isunicode is deprecated.")
    final boolean str_isunicode() {
        Py.warning(Py.DeprecationWarning, "isunicode is deprecated.");
        int n = getString().length();
        for (int i = 0; i < n; i++) {
            char ch = getString().charAt(i);
            if (ch > 255) {
                return true;
            }
        }
        return false;
    }

    public String encode() {
        return encode(null, null);
    }

    public String encode(String encoding) {
        return encode(encoding, null);
    }

    public String encode(String encoding, String errors) {
        return codecs.encode(this, encoding, errors);
    }

    @ExposedMethod(doc = BuiltinDocs.str_encode_doc)
    final String str_encode(PyObject[] args, String[] keywords) {
        ArgParser ap = new ArgParser("encode", args, keywords, "encoding", "errors");
        String encoding = ap.getString(0, null);
        String errors = ap.getString(1, null);
        return encode(encoding, errors);
    }

    public PyObject decode() {
        return decode(null, null);
    }

    public PyObject decode(String encoding) {
        return decode(encoding, null);
    }

    public PyObject decode(String encoding, String errors) {
        return codecs.decode(this, encoding, errors);
    }

    @ExposedMethod(doc = BuiltinDocs.str_decode_doc)
    final PyObject str_decode(PyObject[] args, String[] keywords) {
        ArgParser ap = new ArgParser("decode", args, keywords, "encoding", "errors");
        String encoding = ap.getString(0, null);
        String errors = ap.getString(1, null);
        return decode(encoding, errors);
    }

    @ExposedMethod(doc = BuiltinDocs.str__formatter_parser_doc)
    final PyObject str__formatter_parser() {
        return new MarkupIterator(getString());
    }

    @ExposedMethod(doc = BuiltinDocs.str__formatter_field_name_split_doc)
    final PyObject str__formatter_field_name_split() {
        FieldNameIterator iterator = new FieldNameIterator(getString());
        Object headObj = iterator.head();
        PyObject head =
                headObj instanceof Integer ? new PyInteger((Integer)headObj) : new PyString(
                        (String)headObj);
        return new PyTuple(head, iterator);
    }

    @ExposedMethod(doc = BuiltinDocs.str_format_doc)
    final PyObject str_format(PyObject[] args, String[] keywords) {
        try {
            return new PyString(buildFormattedString(getString(), args, keywords, null));
        } catch (IllegalArgumentException e) {
            throw Py.ValueError(e.getMessage());
        }
    }

    protected String buildFormattedString(String value, PyObject[] args, String[] keywords,
            MarkupIterator enclosingIterator) {
        StringBuilder result = new StringBuilder();
        MarkupIterator it = new MarkupIterator(value, enclosingIterator);
        while (true) {
            MarkupIterator.Chunk chunk = it.nextChunk();
            if (chunk == null) {
                break;
            }
            result.append(chunk.literalText);
            if (chunk.fieldName.length() > 0) {
                PyObject fieldObj = getFieldObject(chunk.fieldName, args, keywords);
                if (fieldObj == null) {
                    continue;
                }
                if ("r".equals(chunk.conversion)) {
                    fieldObj = fieldObj.__repr__();
                } else if ("s".equals(chunk.conversion)) {
                    fieldObj = fieldObj.__str__();
                } else if (chunk.conversion != null) {
                    throw Py.ValueError("Unknown conversion specifier " + chunk.conversion);
                }
                String formatSpec = chunk.formatSpec;
                if (chunk.formatSpecNeedsExpanding) {
                    if (enclosingIterator != null) {
                        // PEP 3101 says only 2 levels
                        throw Py.ValueError("Max string recursion exceeded");
                    }
                    formatSpec = buildFormattedString(formatSpec, args, keywords, it);
                }
                renderField(fieldObj, formatSpec, result);
            }
        }
        return result.toString();
    }

    private PyObject getFieldObject(String fieldName, PyObject[] args, String[] keywords) {
        FieldNameIterator iterator = new FieldNameIterator(fieldName);
        Object head = iterator.head();
        PyObject obj = null;
        int positionalCount = args.length - keywords.length;

        if (head instanceof Integer) {
            int index = (Integer)head;
            if (index >= positionalCount) {
                throw Py.IndexError("tuple index out of range");
            }
            obj = args[index];
        } else {
            for (int i = 0; i < keywords.length; i++) {
                if (keywords[i].equals(head)) {
                    obj = args[positionalCount + i];
                    break;
                }
            }
            if (obj == null) {
                throw Py.KeyError((String)head);
            }
        }
        if (obj != null) {
            while (true) {
                FieldNameIterator.Chunk chunk = iterator.nextChunk();
                if (chunk == null) {
                    break;
                }
                if (chunk.is_attr) {
                    obj = obj.__getattr__((String)chunk.value);
                } else {
                    PyObject key =
                            chunk.value instanceof String ? new PyString((String)chunk.value)
                                    : new PyInteger((Integer)chunk.value);
                    obj = obj.__getitem__(key);
                }
                if (obj == null) {
                    break;
                }
            }
        }
        return obj;
    }

    private void renderField(PyObject fieldObj, String formatSpec, StringBuilder result) {
        PyString formatSpecStr = formatSpec == null ? Py.EmptyString : new PyString(formatSpec);
        result.append(fieldObj.__format__(formatSpecStr).asString());
    }

    @Override
    public PyObject __format__(PyObject formatSpec) {
        return str___format__(formatSpec);
    }

    @ExposedMethod(doc = BuiltinDocs.str___format___doc)
    final PyObject str___format__(PyObject formatSpec) {
        if (!(formatSpec instanceof PyString)) {
            throw Py.TypeError("__format__ requires str or unicode");
        }

        PyString formatSpecStr = (PyString)formatSpec;
        String result;
        try {
            String specString = formatSpecStr.getString();
            InternalFormatSpec spec = new InternalFormatSpecParser(specString).parse();
            result = formatString(getString(), spec);
        } catch (IllegalArgumentException e) {
            throw Py.ValueError(e.getMessage());
        }
        return formatSpecStr.createInstance(result);
    }

    /**
     * Internal implementation of str.__format__()
     *
     * @param text the text to format
     * @param spec the PEP 3101 formatting specification
     * @return the result of the formatting
     */
    public static String formatString(String text, InternalFormatSpec spec) {
        if (spec.sign != '\0') {
            throw new IllegalArgumentException("Sign not allowed in string format specifier");
        }
        if (spec.alternate) {
            throw new IllegalArgumentException(
                    "Alternate form (#) not allowed in string format specifier");
        }
        if (spec.align == '=') {
            throw new IllegalArgumentException(
                    "'=' alignment not allowed in string format specifier");
        }
        if (spec.precision >= 0 && text.length() > spec.precision) {
            text = text.substring(0, spec.precision);
        }
        return spec.pad(text, '<', 0);
    }

    /* arguments' conversion helper */

    @Override
    public String asString(int index) throws PyObject.ConversionException {
        return getString();
    }

    @Override
    public String asString() {
        return getString();
    }

    @Override
    public int asInt() {
        // We have to override asInt/Long/Double because we override __int/long/float__,
        // but generally don't want implicit atoi conversions for the base types. blah
        asNumberCheck("__int__", "an integer");
        return super.asInt();
    }

    @Override
    public long asLong() {
        asNumberCheck("__long__", "an integer");
        return super.asLong();
    }

    @Override
    public double asDouble() {
        asNumberCheck("__float__", "a float");
        return super.asDouble();
    }

    private void asNumberCheck(String methodName, String description) {
        PyType type = getType();
        if (type == PyString.TYPE || type == PyUnicode.TYPE || type.lookup(methodName) == null) {
            throw Py.TypeError(description + " is required");
        }
    }

    @Override
    public String asName(int index) throws PyObject.ConversionException {
        return internedString();
    }

    @Override
    protected String unsupportedopMessage(String op, PyObject o2) {
        if (op.equals("+")) {
            return "cannot concatenate ''{1}'' and ''{2}'' objects";
        }
        return super.unsupportedopMessage(op, o2);
    }
}


final class StringFormatter {

    int index;
    String format;
    StringBuilder buffer;
    boolean negative;
    int precision;
    int argIndex;
    PyObject args;
    boolean unicodeCoercion;

    final char pop() {
        try {
            return format.charAt(index++);
        } catch (StringIndexOutOfBoundsException e) {
            throw Py.ValueError("incomplete format");
        }
    }

    final char peek() {
        return format.charAt(index);
    }

    final void push() {
        index--;
    }

    public StringFormatter(String format) {
        this(format, false);
    }

    public StringFormatter(String format, boolean unicodeCoercion) {
        index = 0;
        this.format = format;
        this.unicodeCoercion = unicodeCoercion;
        buffer = new StringBuilder(format.length() + 100);
    }

    PyObject getarg() {
        PyObject ret = null;
        switch (argIndex) {
        // special index indicating a mapping
            case -3:
                return args;
                // special index indicating a single item that has already been
                // used
            case -2:
                break;
            // special index indicating a single item that has not yet been
            // used
            case -1:
                argIndex = -2;
                return args;
            default:
                ret = args.__finditem__(argIndex++);
                break;
        }
        if (ret == null) {
            throw Py.TypeError("not enough arguments for format string");
        }
        return ret;
    }

    int getNumber() {
        char c = pop();
        if (c == '*') {
            PyObject o = getarg();
            if (o instanceof PyInteger) {
                return ((PyInteger)o).getValue();
            }
            throw Py.TypeError("* wants int");
        } else {
            if (Character.isDigit(c)) {
                int numStart = index - 1;
                while (Character.isDigit(c = pop())) {}
                index -= 1;
                Integer i = Integer.valueOf(format.substring(numStart, index));
                return i.intValue();
            }
            index -= 1;
            return 0;
        }
    }

    private void checkPrecision(String type) {
        if (precision > 250) {
            // A magic number. Larger than in CPython.
            throw Py.OverflowError("formatted " + type + " is too long (precision too long?)");
        }

    }

    private String formatLong(PyObject arg, char type, boolean altFlag) {
        PyString argAsString;
        switch (type) {
            case 'o':
                argAsString = arg.__oct__();
                break;
            case 'x':
            case 'X':
                argAsString = arg.__hex__();
                break;
            default:
                argAsString = arg.__str__();
                break;
        }
        checkPrecision("long");
        String s = argAsString.toString();
        int end = s.length();
        int ptr = 0;

        int numnondigits = 0;
        if (type == 'x' || type == 'X') {
            numnondigits = 2;
        }

        if (s.endsWith("L")) {
            end--;
        }

        negative = s.charAt(0) == '-';
        if (negative) {
            ptr++;
        }

        int numdigits = end - numnondigits - ptr;
        if (!altFlag) {
            switch (type) {
                case 'o':
                    if (numdigits > 1) {
                        ++ptr;
                        --numdigits;
                    }
                    break;
                case 'x':
                case 'X':
                    ptr += 2;
                    numnondigits -= 2;
                    break;
            }
        }
        if (precision > numdigits) {
            StringBuilder buf = new StringBuilder();
            for (int i = 0; i < numnondigits; ++i) {
                buf.append(s.charAt(ptr++));
            }
            for (int i = 0; i < precision - numdigits; i++) {
                buf.append('0');
            }
            for (int i = 0; i < numdigits; i++) {
                buf.append(s.charAt(ptr++));
            }
            s = buf.toString();
        } else if (end < s.length() || ptr > 0) {
            s = s.substring(ptr, end);
        }

        switch (type) {
            case 'X':
                s = s.toUpperCase();
                break;
        }
        return s;
    }

    /**
     * Formats arg as an integer, with the specified radix
     *
     * type and altFlag are needed to be passed to {@link #formatLong(PyObject, char, boolean)} in
     * case the result of <code>arg.__int__()</code> is a PyLong.
     */
    private String formatInteger(PyObject arg, int radix, boolean unsigned, char type,
            boolean altFlag) {
        PyObject argAsInt;
        if (arg instanceof PyInteger || arg instanceof PyLong) {
            argAsInt = arg;
        } else {
            // use __int__ to get an int (or long)
            if (arg instanceof PyFloat) {
                // safe to call __int__:
                argAsInt = arg.__int__();
            } else {
                // Same case noted on formatFloatDecimal:
                // We can't simply call arg.__int__() because PyString implements
                // it without exposing it to python (i.e, str instances has no
                // __int__ attribute). So, we would support strings as arguments
                // for %d format, which is forbidden by CPython tests (on
                // test_format.py).
                try {
                    argAsInt = arg.__getattr__("__int__").__call__();
                } catch (PyException e) {
                    // XXX: Swallow customs AttributeError throws from __float__ methods
                    // No better alternative for the moment
                    if (e.match(Py.AttributeError)) {
                        throw Py.TypeError("int argument required");
                    }
                    throw e;
                }
            }
        }
        if (argAsInt instanceof PyInteger) {
            return formatInteger(((PyInteger)argAsInt).getValue(), radix, unsigned);
        } else { // must be a PyLong (as per __int__ contract)
            return formatLong(argAsInt, type, altFlag);
        }
    }

    private String formatInteger(long v, int radix, boolean unsigned) {
        checkPrecision("integer");
        if (unsigned) {
            if (v < 0) {
                v = 0x100000000l + v;
            }
        } else {
            if (v < 0) {
                negative = true;
                v = -v;
            }
        }
        String s = Long.toString(v, radix);
        while (s.length() < precision) {
            s = "0" + s;
        }
        return s;
    }

    private double asDouble(PyObject obj) {
        try {
            return obj.asDouble();
        } catch (PyException pye) {
            throw !pye.match(Py.TypeError) ? pye : Py.TypeError("float argument required");
        }
    }

    static class DecimalFormatTemplate {

        static DecimalFormat template;
        static {
            template =
                    new DecimalFormat("#,##0.#####", new DecimalFormatSymbols(java.util.Locale.US));
            DecimalFormatSymbols symbols = template.getDecimalFormatSymbols();
            symbols.setNaN("nan");
            symbols.setInfinity("inf");
            template.setDecimalFormatSymbols(symbols);
            template.setGroupingUsed(false);
        }
    }

    private static final DecimalFormat getDecimalFormat() {
        return (DecimalFormat)DecimalFormatTemplate.template.clone();
    }

    private String formatFloatDecimal(double v, boolean truncate) {
        checkPrecision("decimal");
        int prec = precision;
        if (prec == -1) {
            prec = 6;
        }
        if (v < 0) {
            v = -v;
            negative = true;
        }

        DecimalFormat decimalFormat = getDecimalFormat();
        decimalFormat.setMaximumFractionDigits(prec);
        decimalFormat.setMinimumFractionDigits(truncate ? 0 : prec);

        String ret = decimalFormat.format(v);
        return ret;
    }

    private String formatFloatExponential(PyObject arg, char e, boolean truncate) {
        StringBuilder buf = new StringBuilder();
        double v = asDouble(arg);
        boolean isNegative = false;
        if (v < 0) {
            v = -v;
            isNegative = true;
        }
        double power = 0.0;
        if (v > 0) {
            power = ExtraMath.closeFloor(Math.log10(v));
        }
        // System.err.println("formatExp: "+v+", "+power);
        int savePrecision = precision;
        precision = 2;

        String exp = formatInteger((long)power, 10, false);
        if (negative) {
            negative = false;
            exp = '-' + exp;
        } else {
            exp = '+' + exp;
        }

        precision = savePrecision;

        double base = v / Math.pow(10, power);
        buf.append(formatFloatDecimal(base, truncate));
        buf.append(e);

        buf.append(exp);
        negative = isNegative;

        return buf.toString();
    }

    @SuppressWarnings("fallthrough")
    public PyString format(PyObject args) {
        PyObject dict = null;
        this.args = args;
        boolean needUnicode = unicodeCoercion;
        if (args instanceof PyTuple) {
            argIndex = 0;
        } else {
            // special index indicating a single item rather than a tuple
            argIndex = -1;
            if (args instanceof PyDictionary || args instanceof PyStringMap
                    || (!(args instanceof PySequence) && args.__findattr__("__getitem__") != null)) {
                dict = args;
                argIndex = -3;
            }
        }

        while (index < format.length()) {
            boolean ljustFlag = false;
            boolean signFlag = false;
            boolean blankFlag = false;
            boolean altFlag = false;
            boolean zeroFlag = false;

            int width = -1;
            precision = -1;

            char c = pop();
            if (c != '%') {
                buffer.append(c);
                continue;
            }
            c = pop();
            if (c == '(') {
                if (dict == null) {
                    throw Py.TypeError("format requires a mapping");
                }
                int parens = 1;
                int keyStart = index;
                while (parens > 0) {
                    c = pop();
                    if (c == ')') {
                        parens--;
                    } else if (c == '(') {
                        parens++;
                    }
                }
                String tmp = format.substring(keyStart, index - 1);
                this.args = dict.__getitem__(needUnicode ? new PyUnicode(tmp) : new PyString(tmp));
            } else {
                push();
            }
            while (true) {
                switch (c = pop()) {
                    case '-':
                        ljustFlag = true;
                        continue;
                    case '+':
                        signFlag = true;
                        continue;
                    case ' ':
                        blankFlag = true;
                        continue;
                    case '#':
                        altFlag = true;
                        continue;
                    case '0':
                        zeroFlag = true;
                        continue;
                }
                break;
            }
            push();
            width = getNumber();
            if (width < 0) {
                width = -width;
                ljustFlag = true;
            }
            c = pop();
            if (c == '.') {
                precision = getNumber();
                if (precision < -1) {
                    precision = 0;
                }

                c = pop();
            }
            if (c == 'h' || c == 'l' || c == 'L') {
                c = pop();
            }
            if (c == '%') {
                buffer.append(c);
                continue;
            }
            PyObject arg = getarg();
            char fill = ' ';
            String string = null;
            negative = false;
            if (zeroFlag) {
                fill = '0';
            } else {
                fill = ' ';
            }
            switch (c) {
                case 's':
                    if (arg instanceof PyUnicode) {
                        needUnicode = true;
                    }
                case 'r':
                    fill = ' ';
                    if (c == 's') {
                        if (needUnicode) {
                            string = arg.__unicode__().toString();
                        } else {
                            string = arg.__str__().toString();
                        }
                    } else {
                        string = arg.__repr__().toString();
                    }
                    if (precision >= 0 && string.length() > precision) {
                        string = string.substring(0, precision);
                    }

                    break;
                case 'i':
                case 'd':
                    if (arg instanceof PyLong) {
                        string = formatLong(arg, c, altFlag);
                    } else {
                        string = formatInteger(arg, 10, false, c, altFlag);
                    }
                    break;
                case 'u':
                    if (arg instanceof PyLong) {
                        string = formatLong(arg, c, altFlag);
                    } else if (arg instanceof PyInteger || arg instanceof PyFloat) {
                        string = formatInteger(arg, 10, false, c, altFlag);
                    } else {
                        throw Py.TypeError("int argument required");
                    }
                    break;
                case 'o':
                    if (arg instanceof PyLong) {
                        string = formatLong(arg, c, altFlag);
                    } else if (arg instanceof PyInteger || arg instanceof PyFloat) {
                        string = formatInteger(arg, 8, false, c, altFlag);
                        if (altFlag && string.charAt(0) != '0') {
                            string = "0" + string;
                        }
                    } else {
                        throw Py.TypeError("int argument required");
                    }
                    break;
                case 'x':
                    if (arg instanceof PyLong) {
                        string = formatLong(arg, c, altFlag);
                    } else if (arg instanceof PyInteger || arg instanceof PyFloat) {
                        string = formatInteger(arg, 16, false, c, altFlag);
                        string = string.toLowerCase();
                        if (altFlag) {
                            string = "0x" + string;
                        }
                    } else {
                        throw Py.TypeError("int argument required");
                    }
                    break;
                case 'X':
                    if (arg instanceof PyLong) {
                        string = formatLong(arg, c, altFlag);
                    } else if (arg instanceof PyInteger || arg instanceof PyFloat) {
                        string = formatInteger(arg, 16, false, c, altFlag);
                        string = string.toUpperCase();
                        if (altFlag) {
                            string = "0X" + string;
                        }
                    } else {
                        throw Py.TypeError("int argument required");
                    }
                    break;
                case 'e':
                case 'E':
                    string = formatFloatExponential(arg, c, false);
                    if (c == 'E') {
                        string = string.toUpperCase();
                    }
                    break;
                case 'f':
                case 'F':
                    string = formatFloatDecimal(asDouble(arg), false);
                    if (c == 'F') {
                        string = string.toUpperCase();
                    }
                    break;
                case 'g':
                case 'G':
                    int origPrecision = precision;
                    if (precision == -1) {
                        precision = 6;
                    }

                    double v = asDouble(arg);
                    int exponent = (int)ExtraMath.closeFloor(Math.log10(Math.abs(v == 0 ? 1 : v)));
                    if (v == Double.POSITIVE_INFINITY) {
                        string = "inf";
                    } else if (v == Double.NEGATIVE_INFINITY) {
                        string = "-inf";
                    } else if (exponent >= -4 && exponent < precision) {
                        precision -= exponent + 1;
                        string = formatFloatDecimal(v, !altFlag);

                        // XXX: this block may be unnecessary now
                        if (altFlag && string.indexOf('.') == -1) {
                            int zpad = origPrecision - string.length();
                            string += '.';
                            if (zpad > 0) {
                                char zeros[] = new char[zpad];
                                for (int ci = 0; ci < zpad; zeros[ci++] = '0') {}
                                string += new String(zeros);
                            }
                        }
                    } else {
                        // Exponential precision is the number of digits after the decimal
                        // point, whereas 'g' precision is the number of significant digits --
                        // and exponential always provides one significant digit before the
                        // decimal point
                        precision--;
                        string = formatFloatExponential(arg, (char)(c - 2), !altFlag);
                    }
                    if (c == 'G') {
                        string = string.toUpperCase();
                    }
                    break;
                case 'c':
                    fill = ' ';
                    if (arg instanceof PyString) {
                        string = ((PyString)arg).toString();
                        if (string.length() != 1) {
                            throw Py.TypeError("%c requires int or char");
                        }
                        if (arg instanceof PyUnicode) {
                            needUnicode = true;
                        }
                        break;
                    }
                    int val;
                    try {
                        // Explicitly __int__ so we can look for an AttributeError (which is
                        // less invasive to mask than a TypeError)
                        val = arg.__int__().asInt();
                    } catch (PyException e) {
                        if (e.match(Py.AttributeError)) {
                            throw Py.TypeError("%c requires int or char");
                        }
                        throw e;
                    }
                    if (!needUnicode) {
                        if (val < 0) {
                            throw Py.OverflowError("unsigned byte integer is less than minimum");
                        } else if (val > 255) {
                            throw Py.OverflowError("unsigned byte integer is greater than maximum");
                        }
                    } else if (val < 0 || val > PySystemState.maxunicode) {
                        throw Py.OverflowError("%c arg not in range(0x110000) (wide Python build)");
                    }
                    string = new String(new int[] {val}, 0, 1);
                    break;

                default:
                    throw Py.ValueError("unsupported format character '"
                            + codecs.encode(Py.newString(c), null, "replace") + "' (0x"
                            + Integer.toHexString(c) + ") at index " + (index - 1));
            }
            int length = string.length();
            int skip = 0;
            String signString = null;
            if (negative) {
                signString = "-";
            } else {
                if (signFlag) {
                    signString = "+";
                } else if (blankFlag) {
                    signString = " ";
                }
            }

            if (width < length) {
                width = length;
            }
            if (signString != null) {
                if (fill != ' ') {
                    buffer.append(signString);
                }
                if (width > length) {
                    width--;
                }
            }
            if (altFlag && (c == 'x' || c == 'X')) {
                if (fill != ' ') {
                    buffer.append('0');
                    buffer.append(c);
                    skip += 2;
                }
                width -= 2;
                if (width < 0) {
                    width = 0;
                }
                length -= 2;
            }
            if (width > length && !ljustFlag) {
                do {
                    buffer.append(fill);
                } while (--width > length);
            }
            if (fill == ' ') {
                if (signString != null) {
                    buffer.append(signString);
                }
                if (altFlag && (c == 'x' || c == 'X')) {
                    buffer.append('0');
                    buffer.append(c);
                    skip += 2;
                }
            }
            if (skip > 0) {
                buffer.append(string.substring(skip));
            } else {
                buffer.append(string);
            }

            while (--width >= length) {
                buffer.append(' ');
            }
        }
        if (argIndex == -1 || (argIndex >= 0 && args.__finditem__(argIndex) != null)) {
            throw Py.TypeError("not all arguments converted during string formatting");
        }
        if (needUnicode) {
            return new PyUnicode(buffer);
        }
        return new PyString(buffer);
    }

}
