/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.java.caching;

import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import org.sonar.api.batch.fs.InputFile;
import org.sonar.api.batch.sensor.SensorContext;
import org.sonar.api.batch.sensor.cache.ReadCache;
import org.sonar.api.batch.sensor.cache.WriteCache;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.java.caching.CacheContextImpl;
import org.sonar.java.caching.FileHashingUtils;

public class ContentHashCache {
    private static final Logger LOG = Loggers.get(ContentHashCache.class);
    private static final String CONTENT_HASH_KEY = String.format("java:contentHash:%s:", "MD5");
    private static final String HASH_COMPUTE_FAIL_MSG = "Failed to compute content hash for file %s";
    private ReadCache readCache;
    private WriteCache writeCache;
    private final boolean enabled;

    public ContentHashCache(SensorContext context) {
        CacheContextImpl cacheContext = CacheContextImpl.of(context);
        this.enabled = cacheContext.isCacheEnabled();
        if (this.enabled) {
            this.readCache = context.previousCache();
            this.writeCache = context.nextCache();
        }
    }

    public boolean hasSameHashCached(InputFile inputFile) {
        if (!this.enabled) {
            if (inputFile.status() == InputFile.Status.SAME) {
                LOG.trace(() -> "Cache is disabled. File status is: " + String.valueOf(inputFile.status()) + ". File can be skipped.");
                return true;
            }
            LOG.trace(() -> "Cache is disabled. File status is: " + String.valueOf(inputFile.status()) + ". File can't be skipped.");
            return false;
        }
        String cacheKey = ContentHashCache.getCacheKey(inputFile);
        try {
            LOG.trace(() -> "Reading cache for the file " + inputFile.key());
            byte[] cachedHash = this.readCache.read(cacheKey).readAllBytes();
            byte[] fileHash = FileHashingUtils.inputFileContentHash(inputFile);
            boolean isHashEqual = MessageDigest.isEqual(fileHash, cachedHash);
            if (isHashEqual) {
                this.copyFromPrevious(inputFile);
            } else {
                this.writeToCache(inputFile);
            }
            return isHashEqual;
        }
        catch (IllegalArgumentException e) {
            LOG.warn(String.format("Could not find key %s in the cache", cacheKey));
            this.writeToCache(inputFile);
        }
        catch (IOException | NoSuchAlgorithmException e) {
            LOG.warn(String.format(HASH_COMPUTE_FAIL_MSG, inputFile.key()));
        }
        return false;
    }

    public boolean contains(InputFile inputFile) {
        if (!this.enabled) {
            LOG.trace(() -> "Cannot lookup cached hashes when the cache is disabled (" + inputFile.key() + ").");
            return false;
        }
        return this.readCache.contains(ContentHashCache.getCacheKey(inputFile));
    }

    public boolean writeToCache(InputFile inputFile) {
        if (!this.enabled) {
            LOG.trace(() -> "Cannot write hashes to the cache when the cache is disabled (" + inputFile.key() + ").");
            return false;
        }
        LOG.trace(() -> "Writing to the cache for file " + inputFile.key());
        String cacheKey = ContentHashCache.getCacheKey(inputFile);
        try {
            this.writeCache.write(cacheKey, FileHashingUtils.inputFileContentHash(inputFile));
            return true;
        }
        catch (IllegalArgumentException e) {
            LOG.warn(String.format("Tried to write multiple times to cache key %s. Ignoring writes after the first.", cacheKey));
        }
        catch (IOException | NoSuchAlgorithmException e) {
            LOG.warn(String.format(HASH_COMPUTE_FAIL_MSG, inputFile.key()));
        }
        return false;
    }

    private void copyFromPrevious(InputFile inputFile) {
        LOG.trace(() -> "Copying cache from previous for file " + inputFile.key());
        this.writeCache.copyFromPrevious(ContentHashCache.getCacheKey(inputFile));
    }

    private static String getCacheKey(InputFile inputFile) {
        return CONTENT_HASH_KEY + inputFile.key();
    }
}

