/*
 * SonarQube JavaScript Plugin
 * Copyright (C) 2011-2016 SonarSource SA
 * mailto:contact AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.javascript.checks;

import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;
import org.sonar.check.Rule;
import org.sonar.javascript.checks.utils.CheckUtils;
import org.sonar.javascript.tree.symbols.Scope;
import org.sonar.javascript.tree.symbols.type.FunctionType;
import org.sonar.plugins.javascript.api.symbols.Symbol;
import org.sonar.plugins.javascript.api.symbols.Type;
import org.sonar.plugins.javascript.api.tree.Tree;
import org.sonar.plugins.javascript.api.tree.Tree.Kind;
import org.sonar.plugins.javascript.api.tree.declaration.FunctionTree;
import org.sonar.plugins.javascript.api.tree.expression.CallExpressionTree;
import org.sonar.plugins.javascript.api.visitors.DoubleDispatchVisitorCheck;

@Rule(key = "S930")
public class TooManyArgumentsCheck extends DoubleDispatchVisitorCheck {

  private static final String MESSAGE = "%s expects %s argument%s, but %s %s provided.";

  @Override
  public void visitCallExpression(CallExpressionTree tree) {
    FunctionTree functionTree = getFunction(tree);

    if (functionTree != null) {

      int parametersNumber = functionTree.parameterList().size();
      int argumentsNumber = tree.arguments().parameters().size();

      if (!hasRestParameter(functionTree) && !builtInArgumentsUsed(functionTree) && argumentsNumber > parametersNumber) {
        String message = getMessage(tree, parametersNumber, argumentsNumber);
        addIssue(tree.arguments(), message)
          .secondary(functionTree.parameterClause(), "Formal parameters");
      }

    }

    super.visitCallExpression(tree);
  }

  private static String getMessage(CallExpressionTree tree, int parametersNumber, int argumentsNumber) {
    String callee;
    if (CheckUtils.removeParenthesis(tree.callee()).is(Kind.FUNCTION_EXPRESSION)) {
      callee = "This function";
    } else {
      callee = "\"" + CheckUtils.asString(tree.callee()) + "\"";
    }
    return String.format(MESSAGE, callee, parametersNumber, parametersNumber == 1 ? "" : "s", argumentsNumber, argumentsNumber > 1 ? "were" : "was");
  }

  /*
   * @return true if function's last parameter has "... p" format and stands for all rest parameters
   */
  private static boolean hasRestParameter(FunctionTree functionTree) {
    List<Tree> parameters = functionTree.parameterList();
    return !parameters.isEmpty() && (parameters.get(parameters.size() - 1).is(Tree.Kind.REST_ELEMENT));
  }


  @Nullable
  private static FunctionTree getFunction(CallExpressionTree tree) {
    Set<Type> types = tree.callee().types();

    if (types.size() == 1 && types.iterator().next().kind().equals(Type.Kind.FUNCTION)) {
      return ((FunctionType) types.iterator().next()).functionTree();
    }

    return null;
  }


  private boolean builtInArgumentsUsed(FunctionTree tree) {
    Scope scope = getContext().getSymbolModel().getScope(tree);
    if (scope == null) {
      throw new IllegalStateException("No scope found for FunctionTree");
    }

    Symbol argumentsBuiltInVariable = scope.lookupSymbol("arguments");
    if (argumentsBuiltInVariable == null) {
      if (!tree.is(Kind.ARROW_FUNCTION)) {
        throw new IllegalStateException("No 'arguments' symbol found for function scope");
      } else {
        return false;
      }
    }

    boolean isUsed = !argumentsBuiltInVariable.usages().isEmpty();
    return argumentsBuiltInVariable.builtIn() && isUsed;
  }

}
