/*
 * SonarQube JavaScript Plugin
 * Copyright (C) 2011-2017 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.javascript.tree.impl.expression;

import com.google.common.base.Functions;
import com.google.common.collect.Iterators;
import java.util.Iterator;
import org.sonar.javascript.tree.impl.JavaScriptTree;
import org.sonar.javascript.tree.impl.lexical.InternalSyntaxToken;
import org.sonar.javascript.tree.symbols.type.TypableTree;
import org.sonar.plugins.javascript.api.symbols.Type;
import org.sonar.plugins.javascript.api.symbols.TypeSet;
import org.sonar.plugins.javascript.api.tree.SeparatedList;
import org.sonar.plugins.javascript.api.tree.Tree;
import org.sonar.plugins.javascript.api.tree.expression.ObjectLiteralTree;
import org.sonar.plugins.javascript.api.tree.lexical.SyntaxToken;
import org.sonar.plugins.javascript.api.visitors.DoubleDispatchVisitor;

public class ObjectLiteralTreeImpl extends JavaScriptTree implements ObjectLiteralTree, TypableTree {

  private final SyntaxToken openCurlyBrace;
  private final SeparatedList<Tree> properties;
  private final SyntaxToken closeCurlyBrace;
  private TypeSet types = TypeSet.emptyTypeSet();

  public ObjectLiteralTreeImpl(InternalSyntaxToken openCurlyBrace, SeparatedList<Tree> properties, InternalSyntaxToken closeCurlyBrace) {
    this.openCurlyBrace = openCurlyBrace;
    this.properties = properties;
    this.closeCurlyBrace = closeCurlyBrace;
  }

  @Override
  public SyntaxToken openCurlyBraceToken() {
    return openCurlyBrace;
  }

  @Override
  public SeparatedList<Tree> properties() {
    return properties;
  }

  @Override
  public SyntaxToken closeCurlyBraceToken() {
    return closeCurlyBrace;
  }

  @Override
  public Kind getKind() {
    return Kind.OBJECT_LITERAL;
  }

  @Override
  public TypeSet types() {
    return types.immutableCopy();
  }

  @Override
  public void add(Type type) {
    this.types.add(type);
  }

  @Override
  public Iterator<Tree> childrenIterator() {
    return Iterators.concat(
      Iterators.singletonIterator(openCurlyBrace),
      properties.elementsAndSeparators(Functions.<Tree>identity()),
      Iterators.singletonIterator(closeCurlyBrace)
    );
  }

  @Override
  public void accept(DoubleDispatchVisitor visitor) {
    visitor.visitObjectLiteral(this);
  }
}
